function A = make_sparse_randn_matrix_A(N,type,density)
%
%  This function employs the Matlab function "sprandn" to generate a random 
%  sparse N x N matrix A.  The input density is passed onto "sprandn" and 
%  controls the sparsity:  A has approximately
%
%               density * N * N
%
%  normally distributed nonzero entries.  Note that the function "sprandn" is
%  designed to produce large matrices with small density and will generate
%  significantly fewer nonzeros than requested if N is small or density is
%  large.
%
%  The input flag "type" is used to control the type of matrix A that is
%  generated:
%
%  type =  0:  A is real (and nonsymmetric in general)
%
%  type =  1:  A is complex (and non-Hermitian in general)
%
%  type =  2:  A is real symmetric
%
%  type = -2:  A is real skew-symmetric
%
%  type =  3:  A is complex symmetric
%
%  type = -3:  A is complex skew-symmetric  
%
%  type =  4:  A is Hermitian (with complex entries)
%
%  type = -4:  A is skew-Hermitian (with complex entries)
%
%  type =  9:  A is a diagonal matrix with nonzero real diagonal entries
%
%  type = -9:  A is a diagonal matrix with nonzero complex diagonal entries
%
% -----------------------------------------------------------------------------
%
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

if abs(type) < 9,  
   A = sprandn(N,N,density);
%
   if abs(type) > 1,
      S = triu(A);
   end
end
%
switch type
   case 1
      A = A + sqrt(-1) * sprandn(A);
   case 2
      A = S + (triu(S,1))';
   case -2
      S = triu(S,1);
      A = S - S';
   case 3
      S = S + sqrt(-1) * sprandn(S);
      A = S + (triu(S,1)).';
   case -3 
      S = triu(S,1);
      S = S + sqrt(-1) * sprandn(S);
      A = S - S.';
   case 4
      S0 = triu(S,1);
      S = S + sqrt(-1) * sprandn(S0);
      A = S + (triu(S,1))';
   case -4
      S = triu(S,1) + sqrt(-1) * sprandn(S);
      A = S - (triu(S,1))';
   case 9
      d = 2 * rand(n,1) - 1;
      A = spdiags(d,0,N,N);
   case -9
      d = (2 * rand(n,1) - 1) + sqrt(-1) * (2 * rand(n,1) - 1);
      A = spdiags(d,0,N,N);
end

