function result = make_matrices_for_compact_recursions(result)
%
%  This function makes the matrices that are needed to state the recurrence
%  relations used in the various band algorithms in compact form and adds them
%  to the structure "result".
%
% -----------------------------------------------------------------------------
%
%  Input:  A structure "result" generated by a call to one of the functions
%          "band_Arnoldi", "band_Lanczos", "sym_band_Lanczos", 
%          "Herm_band_Lanczos", "Jsym_band_Lanczos", "JHerm_band_Lanczos",
%          which implement the band Arnoldi process and the band Lanczos method
%          and its variants
%
% -----------------------------------------------------------------------------
%
%  On return, the structure "result" contains all the fields (with unchanged
%  values) of the input structure "result" and the following new fields:
%
%     When the input structure "result" was produced by the function
%     "band_Arnoldi", the 4 fields
%
%           result.H_s,  result.rho_s,  result.V_dh,  result.R_dh
%
%     are added to "result".
%  
%     When the input structure "result" was produced by the function
%     "band_Lanczos", the 8 fields
%
%           result.T_s,  result.rho_s,  result.V_dh,  result.R_dh
%
%     and
%
%           result.Tt_s,  result.eta_s,  result.W_dh,  result.L_dh
% 
%     are added to "result".
%
%     When the input structure "result" was produced by any of the functions
%     "sym_band_Lanczos", "Herm_band_Lanczos", "Jsym_band_Lanczos", or
%     "JHerm_band_Lanczos", the 4 fields
%
%           result.T_s,  result.rho_s,  result.V_dh,  result.R_dh
%
%     are added to "result".
%
% -----------------------------------------------------------------------------
%
%  Notes:  1) The various band algorithms construct basis vectors for the n-th
%             right block Krylov subspace (induced by A and R) by means of
%             recurrence relations that can be stated in compact form as
%             follows:
%
%                  A V = V H_s + V_dh     (for the band Arnoldi process)
%
%                  A V = V T_s + V_dh     (for the band Lanczos method and
%                                                             its variants)
%
%                    R = V(:,1:mt) rho_s + R_dh
%
%             Here, V is the matrix the columns of which are the n constructed
%             basis vectors and mt = min(m,n).  The matrices H_s, T_s, and 
%             rho_s contain the recurrence coefficients, and V_dh and R_dh are
%             sparse matrices the nonzero columns of which are the candidate
%             vectors and, if deflations have occurred, the deflated vectors.
%
%          2) The band Lanzcos method also constructs basis vectors for the
%             n-th left block Krylov subspace (induced by A^T and L) by means
%             of recurrence relations that can be stated in compact form as
%             follows:
%
%                  A^T W = W Tt_s + W_dh
%
%                      L = W(:,1:pt) eta_s + L_dh	    
%
%             Here, W is the matrix the columns of which are the n constructed
%             basis vectors and pt = min(p,n).  The matrices Tt_s and eta_s
%             contain the recurrence coefficients, and W_dh and L_dh are sparse
%             matrices the nonzero columns of which are the candidate vectors
%	      and, if deflations have occurred, the deflated vectors.
%
%          3) The matrices H_s, T_s, Tt_s, rho_s, and eta_s are readily 
%             obtained by zeroing out the vertical spikes (due to possible
%	      deflations) of the matrices H, T, Tt, rho, and eta that are
%             generated by the functions "band_Arnoldi", "band_Lanczos",
%	      "sym_band_Lanczos", "Herm_band_Lanczos", "Jsym_band_Lanczos", or
%             "JHerm_band_Lanczos".  The information that is needed to set up
%             the sparse matrices V_dh, R_dh, W_dh, and L_dh is also provided
%	      in the structures "result" returned by these functions.
%
% -----------------------------------------------------------------------------
%  
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

I = result.Iv.I;
mc = result.mc;
nd = result.Iv.nd;
m = mc + nd;
n = result.n;
%
%  Remove the vertical spikes from T and rho, respectively H and rho
%
if isfield(result,'T'),
   T_s = result.T;
else
   T_s = result.H;
end
%
rho_s = result.rho;
%
for j = 1 : nd;
   Ij = I(j);
   r  = Ij + m - j + 1;
%   
   if Ij > 0,
      c = Ij;
      T_s(r:end,c) = zeros(size(T_s(r:end,c)));
   else
      c = Ij + m;
      rho_s(r:end,c) = zeros(size(rho_s(r:end,c)));
   end
end
%
result.rho_s = rho_s;
%
if isfield(result,'T'),
   result.T_s = T_s;
else
   result.H_s = T_s;
end
%
%  Make the sparse matrices whose nonzero columns are the right candidate
%  vectors and the deflated v vectors
%
Vh_defl = result.Vh_defl;
pd = result.Iv.pd;
ph = result.Iv.ph;
% 
N = size(result.V,1);
V_dh = sparse(N,n);
R_dh = sparse(N,m);
%
Ktmp = find(I > 0);
V_dh(:,I(Ktmp)) = Vh_defl(:,pd(Ktmp));
%
Ktmp = find(I <= 0);
R_dh(:,m+I(Ktmp)) = Vh_defl(:,pd(Ktmp));
%
Ktmp = find([1:mc] > mc-n);
V_dh(:,n-mc+Ktmp) = Vh_defl(:,ph(Ktmp));
%
Ktmp = find([1:mc] <= mc-n);
R_dh(:,n+nd+Ktmp) = Vh_defl(:,ph(Ktmp));
%
result.V_dh = V_dh;
result.R_dh = R_dh;
%
%  Check if there are w vectors, and if not, return
% 
if isfield(result,'pc') == 0,
   return
end
%
%  Remove the vertical spikes from Tt and eta
% 
I = result.Iw.I;
pc = result.pc;
nd = result.Iw.nd;
p = pc + nd;
% 
Tt_s = result.Tt;
eta_s = result.eta;
%   
for j = 1 : nd;
   Ij = I(j);
   r  = Ij + p - j + 1;
%   
   if Ij > 0,
      c = Ij;
      Tt_s(r:end,c) = zeros(size(Tt_s(r:end,c)));
   else
      c = Ij + p;
      eta_s(r:end,c) = zeros(size(eta_s(r:end,c)));
   end
end
%   
result.Tt_s = Tt_s;
result.eta_s = eta_s;
%
%  Make the sparse matrices whose nonzero columns are the left candidate
%  vectors and the deflated w vectors
%
Wh_defl = result.Wh_defl;
pd = result.Iw.pd;
ph = result.Iw.ph;
% 
W_dh = sparse(N,n);
L_dh = sparse(N,p);
%
Ktmp = find(I > 0);
W_dh(:,I(Ktmp)) = Wh_defl(:,pd(Ktmp));
%
Ktmp = find(I <= 0);
L_dh(:,p+I(Ktmp)) = Wh_defl(:,pd(Ktmp));
%
Ktmp = find([1:pc] > pc-n);
W_dh(:,n-pc+Ktmp) = Wh_defl(:,ph(Ktmp));
%
Ktmp = find([1:pc] <= pc-n);
L_dh(:,n+nd+Ktmp) = Wh_defl(:,ph(Ktmp));
%
result.W_dh = W_dh;
result.L_dh = L_dh;

