/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file contains routines for multiplying vector/matrix-valued           */
/* polynomials.                                                               */
/*                                                                            */
/* (written by Dean Hickerson, 8/29/94)                                       */
/******************************************************************************/

#include <stdio.h>
#include "mexdefines.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
void matmult(double a[], double b[], int m1, int m2, int m3, double p[],
															boolean subtract);
void addmatprodh(double a[], int adeg, double b[], int bdeg, int insize,
                            int m1, int m2, int m3, double prod[],
							boolean subtract);
void addmatprod(double a[], int adeg0, int adeg1,
                double b[], int bdeg0, int bdeg1,
                int insize, int m1, int m2, int m3,
                double prod[], int pdeg0, int pdeg1,
				boolean subtract);
void addscalarprodh(double a[], int adeg, int astep,
                    double b[], int bdeg, int bstep,
                    int insize,
                    double prod[], int pstep, boolean subtract);
void addscalarprod(double a[], int adeg0, int adeg1, int astep,
                   double b[], int bdeg0, int bdeg1, int bstep,
                   int insize, double prod[],
				   int pdeg0, int pdeg1, int pstep, boolean subtract);
#endif

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern void printmonom(struct monom *m);
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
extern boolean initmonom(struct monom *m, int deg, int insz);
extern boolean nextmonom(struct monom *m, int *changeptr);
#else
extern void printmonom();
extern int crd();
extern int crdsum();
extern boolean initmonom();
extern boolean nextmonom();
#endif

/******************************************************************************/

#ifdef __STDC__
void matmult(double a[], double b[], int m1, int m2, int m3, double p[],
															boolean subtract)
#else
matmult(a, b, m1, m2, m3, p, subtract)
double a[], b[];
int m1, m2, m3;
double p[];
boolean subtract;
#endif
/*  a[0:m1*m2-1],  b[0:m2*m3-1],  and  p[0:m1*m3-1]  are,
    respectively, m1 by m2, m2 by m3, and m1 by m3 matrices.  This
    routine adds (or subtracts) a*b to p.
*/
{   int i, j, k;
    double sum;
    double aval, *aptr, *bptr, *pptr0, *pptr;

/*  The code below implements the following loop, but without
    extraneous multiplications:

        for (i=0; i<m1; i++)
          for (j=0; j<m2; j++)
            for (k=0; k<m3; k++)
              p[i*m3+k] += a[i*m2+j] * b[j*m3+k];

    aptr  points to  a[i*m2+j]
    bptr  points to  b[j*m3+k]
    pptr0 points to  p[i*m3]
    pptr  points to  p[i*m3+k]
*/

	if (subtract)
      { for (i=0, aptr=a, pptr0=p; i<m1; i++, pptr0+=m3)
          for (j=0, bptr=b; j<m2; j++, aptr++)
            for (k=0, pptr=pptr0, aval=*aptr; k<m3; k++)
              *pptr++ -= aval * *bptr++;
	  }

	else	/* Add */
      { for (i=0, aptr=a, pptr0=p; i<m1; i++, pptr0+=m3)
          for (j=0, bptr=b; j<m2; j++, aptr++)
            for (k=0, pptr=pptr0, aval=*aptr; k<m3; k++)
              *pptr++ += aval * *bptr++;
	  }
}

/******************************************************************************/

#ifdef __STDC__
void addmatprodh(double a[], int adeg, double b[], int bdeg, int insize,
                    int m1, int m2, int m3, double prod[], boolean subtract)
#else
addmatprodh(a, adeg, b, bdeg, insize, m1, m2, m3, prod, subtract)
double a[], b[], prod[];
int adeg, bdeg, insize, m1, m2, m3;
boolean subtract;
#endif
/*  a[0:m1*m2*crd(insize,adeg)-1]  and
    b[0:m2*m3*crd(insize,bdeg)-1]  represent matrix-valued (m1 by m2
    and m2 by m3, respectively) polynomials of degrees adeg and bdeg
    in insize variables.  This routine adds (or subtracts) their product to
    prod[0:m1*m3*crd(insize,adeg+bdeg)-1].
*/
{   struct monom ma, mb, mprod;
    int sum, change, i, j, dist, newfac;
    int aoutsize=m1*m2, boutsize=m2*m3, prodoutsize=m1*m3;

    /*********************/
    /* Set ma to x0^adeg */
    /*********************/
    if (!initmonom(&ma, adeg, insize))  return;

    /*******************************/
    /* Init constant part of mprod */
    /*******************************/
    mprod.insize = insize;
    mprod.degree = adeg + bdeg;

    do
      { /*********************/
        /* Set mb to x0^bdeg */
        /*********************/
        if (!initmonom(&mb, bdeg, insize))  return;

        /*******************/
        /* mprod = ma * mb */
        /*******************/
        mprod.index = ma.index;
        for (i=0; i<bdeg; i++)  mprod.factor[i] = 0;
        for (i=0; i<adeg; i++)  mprod.factor[bdeg+i] = ma.factor[i];

        /**************************************************/
        /* prod[mprod.index] += a[ma.index] * b[mb.index] */
        /**************************************************/
        matmult(&a[aoutsize*ma.index], &b[boutsize*mb.index],
            m1, m2, m3, &prod[prodoutsize*mprod.index], subtract);

        while (nextmonom(&mb, &change))
          { /******************************************************************/
            /* Update mprod.  mb.factor has been changed as follows:          */
            /*                                                                */
            /* Index:  0   ...  change-1   change   change+1   ...    bdeg-1  */
            /* Was:    ?   ...     ?         fac    insize-1   ...   insize-1 */
            /* Is:     ?   ...     ?        fac+1     fac+1    ...     fac+1  */
            /******************************************************************/
            dist = mb.degree - change - 1;
            newfac = mb.factor[change];

            if (dist)
              for (i=mprod.degree-1; mprod.factor[i-dist] >= newfac; i--)
                { mprod.index += crd(insize-1-mprod.factor[i], mprod.degree-i);
                  mprod.factor[i] = mprod.factor[i-dist];
                  mprod.index -= crd(insize-1-mprod.factor[i], mprod.degree-i);
                }
            else
              for (i=mprod.degree-1; mprod.factor[i] >= newfac; i--);

            for (j=dist; j>=0; j--)
              { mprod.index += crd(insize-1-mprod.factor[i], mprod.degree-i);
                mprod.factor[i] = newfac;
                mprod.index -= crd(insize-1-mprod.factor[i], mprod.degree-i);
                i--;
              }

            /**************************************************/
            /* prod[mprod.index] += a[ma.index] * b[mb.index] */
            /**************************************************/
            matmult(&a[aoutsize*ma.index], &b[boutsize*mb.index],
                m1, m2, m3, &prod[prodoutsize*mprod.index], subtract);
          }
      }
    while (nextmonom(&ma, NULL));
}

/******************************************************************************/

#ifdef __STDC__
void addmatprod(double a[], int adeg0, int adeg1,
            double b[], int bdeg0, int bdeg1,
            int insize, int m1, int m2, int m3,
            double prod[], int pdeg0, int pdeg1, boolean subtract)
#else
addmatprod(a, adeg0, adeg1, b, bdeg0, bdeg1, insize, m1, m2, m3,
           prod, pdeg0, pdeg1, subtract)
double a[], b[], prod[];
int adeg0, adeg1, bdeg0, bdeg1, insize, m1, m2, m3, pdeg0, pdeg1;
boolean subtract;
#endif
/*  a[0:m1*m2*crdsum(insize,adeg,adeg1)-1]  and
    b[0:m2*m3*crdsum(insize,bdeg0,bdeg1)-1]  represent matrix-valued
    (m1 by m2 and m2 by m3, respectively) polynomials of degrees adeg0
    thru adeg1 and bdeg0 thru bdeg1, respectively, in insize variables.
    This routine adds (or subtracts) the part of their product with degrees
	pdeg0 thru pdeg1 to prod[0:m1*m3*crdsum(insize,pdeg0,pdeg1)-1].
*/
{   double *adegptr, *proddegptr;
    int adeg, bdeg, pdeg;

    for (pdeg=pdeg0, proddegptr = prod;  pdeg<=pdeg1;
                            proddegptr += m1*m3*crd(insize,pdeg), pdeg++)
      for (adeg=adeg0, adegptr = a;  adeg<=adeg1;
                                adegptr += m1*m2*crd(insize,adeg), adeg++)
        if ((bdeg = pdeg-adeg) >= bdeg0 && bdeg <= bdeg1)
          addmatprodh(adegptr, adeg,
                            b+m2*m3*crdsum(insize,bdeg0,bdeg-1), bdeg,
                            insize, m1, m2, m3, proddegptr, subtract);
}

/******************************************************************************/

#ifdef __STDC__
void addscalarprodh(double a[], int adeg, int astep,
                    double b[], int bdeg, int bstep,
                    int insize,
                    double prod[], int pstep, boolean subtract)
#else
addscalarprodh(a, adeg, astep, b, bdeg, bstep, insize, prod, pstep, subtract)
double a[], b[], prod[];
int adeg, astep, bdeg, bstep, insize, pstep;
boolean subtract;
#endif
/*  This is similar to addmatprodh, but the outputs of the functions are
    scalars rather than matrices, and the coefficients of a, b, and prod
    are not stored contiguously.  a[0 (astep) astep*(crd(insize,adeg)-1)]
    and  b[0 (bstep) bstep*(crd(insize,bdeg)-1)]  represent real-valued
    polynomials of degrees adeg and bdeg in insize variables.  This routine
    adds (or subtracts) their product to
	prod[0 (pstep) pstep*(crd(insize,adeg+bdeg)-1)].
*/
{   struct monom ma, mb, mprod;
    int sum, change, i, j, dist, newfac;

    /*********************/
    /* Set ma to x0^adeg */
    /*********************/
    if (!initmonom(&ma, adeg, insize))  return;

    /*******************************/
    /* Init constant part of mprod */
    /*******************************/
    mprod.insize = insize;
    mprod.degree = adeg + bdeg;

    do
      { /*********************/
        /* Set mb to x0^bdeg */
        /*********************/
        if (!initmonom(&mb, bdeg, insize))  return;

        /*******************/
        /* mprod = ma * mb */
        /*******************/
        mprod.index = ma.index;
        for (i=0; i<bdeg; i++)  mprod.factor[i] = 0;
        for (i=0; i<adeg; i++)  mprod.factor[bdeg+i] = ma.factor[i];

        /**************************************************/
        /* prod[mprod.index] += a[ma.index] * b[mb.index] */
        /**************************************************/
		if (subtract)
          prod[pstep * mprod.index] -= a[astep * ma.index]*b[bstep * mb.index];
		else
          prod[pstep * mprod.index] += a[astep * ma.index]*b[bstep * mb.index];

        while (nextmonom(&mb, &change))
          { /******************************************************************/
            /* Update mprod.  mb.factor has been changed as follows:          */
            /*                                                                */
            /* Index:  0   ...  change-1   change   change+1   ...    bdeg-1  */
            /* Was:    ?   ...     ?         fac    insize-1   ...   insize-1 */
            /* Is:     ?   ...     ?        fac+1     fac+1    ...     fac+1  */
            /******************************************************************/
            dist = mb.degree - change - 1;
            newfac = mb.factor[change];

            if (dist)
              for (i=mprod.degree-1; mprod.factor[i-dist] >= newfac; i--)
                { mprod.index += crd(insize-1-mprod.factor[i], mprod.degree-i);
                  mprod.factor[i] = mprod.factor[i-dist];
                  mprod.index -= crd(insize-1-mprod.factor[i], mprod.degree-i);
                }
            else
              for (i=mprod.degree-1; mprod.factor[i] >= newfac; i--);

            for (j=dist; j>=0; j--)
              { mprod.index += crd(insize-1-mprod.factor[i], mprod.degree-i);
                mprod.factor[i] = newfac;
                mprod.index -= crd(insize-1-mprod.factor[i], mprod.degree-i);
                i--;
              }

            /**************************************************/
            /* prod[mprod.index] += a[ma.index] * b[mb.index] */
            /**************************************************/
			if (subtract)
              prod[pstep * mprod.index] -=
                                  a[astep * ma.index] * b[bstep * mb.index];
			else
              prod[pstep * mprod.index] +=
                                  a[astep * ma.index] * b[bstep * mb.index];
          }
      }
    while (nextmonom(&ma, NULL));
}

/******************************************************************************/

#ifdef __STDC__
void addscalarprod(double a[], int adeg0, int adeg1, int astep,
                double b[], int bdeg0, int bdeg1, int bstep,
                int insize, double prod[],
				int pdeg0, int pdeg1, int pstep, boolean subtract)
#else
addscalarprod(a, adeg0, adeg1, astep, b, bdeg0, bdeg1, bstep, insize,
              prod, pdeg0, pdeg1, pstep, subtract)
double a[], b[], prod[];
int adeg0, adeg1, astep, bdeg0, bdeg1, bstep, insize, pdeg0, pdeg1, pstep;
boolean subtract;
#endif
/*  This is similar to addmatprod, but the outputs of the functions are
    scalars rather than matrices, and the coefficients of a, b, and prod
    are not stored contiguously.
    a[0 (astep) astep*(crdsum(insize,adeg0,adeg1)-1)]  and
    b[0 (bstep) bstep*(crdsum(insize,bdeg0,bdeg1)-1)] represent real-valued
    polynomials of degrees adeg and bdeg in insize variables.  This routine
    adds (or subtracts) that part of their product with degrees pdeg0 thru
	pdeg1 to  prod[0 (pstep) pstep*(crdsum(insize,pdeg0,pdeg1)-1)].
*/
{   double *adegptr, *proddegptr;
    int adeg, bdeg, pdeg;

    for (pdeg=pdeg0, proddegptr = prod;  pdeg<=pdeg1;
            proddegptr += pstep*crd(insize,pdeg), pdeg++)
      for (adeg=adeg0, adegptr = a;  adeg<=adeg1;
            adegptr += astep*crd(insize,adeg), adeg++)
        if ((bdeg = pdeg-adeg) >= bdeg0 && bdeg <= bdeg1)
          addscalarprodh(adegptr, adeg, astep,
                b+bstep*crdsum(insize,bdeg0,bdeg-1), bdeg, bstep,
                insize, proddegptr, pstep, subtract);
}
