/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/*  This file contains various low-level routines, specifically choose, crd   */
/*  crd, crdsum, power, and samesize.                                         */
/*                                                                            */
/*  (written by Dean Hickerson, 8/6/94)                                       */
/******************************************************************************/

#include <stdio.h>
#include "mexdefines.h"
#include "mex.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
int computechoose(int m, int n);
int crd(int m, int n);
int choose(int m, int n);
int crdsum(int m, int n0, int n1);
int power(int m, int n);
boolean samesize(int fr, int fc, int wantedfr, int wantedfc);
#endif

/******************************************************************************/

#ifdef __STDC__
int computechoose(int m, int n)
#else
int computechoose(m,n)
int m, n;
#endif
/*  Computes the binomial coefficient m choose n, defined whenever n is an
    integer by if n<0 then 0 else  m (m-1) ... (m-n+1) / n! .
*/
{   int i;
    double val;     /* double to avoid overflow from multiplication */

    if (n<0)  return 0;
    for (i=val=1; i<=n; i++)  val = val*(m+1-i)/i;
    return val;
}

/******************************************************************************/

#define MAXM    50
#define MAXN    50
static int crdtable[MAXM][MAXN];
/******************************************************************************/
/* This table is used to make crd calculations more efficient.  When a value  */
/* of crd(m,n) is needed with 1<=m<MAXM and 0<=n<MAXN, it is first looked up  */
/* in the table.  If the result is 0, then the value is computed and saved.   */
/* Initially the table is filled with 0's.                                    */
/* Each mexFunction has its own copy of this table, which is maintained from  */
/* one call to the next.  So, e.g., multiple calls to mply will only compute  */
/* crd(5,3) once, but a subsequent call to cmp may compute it again.          */
/******************************************************************************/

#ifdef __STDC__
int crd(int m, int n)
#else
int crd(m,n)
int m, n;
#endif
/*  Returns choose(m+n-1,n).  This is the number of ways of choosing n not
    necessarily distinct items from a set of m items.  This routine should
    only be called with m>=0.
*/
{   int val;
    char outbuff[200];

    if (m<0)           /* This shouldn't happen */
      { sprintf(outbuff,"ERROR:  Called crd(%ld,%ld)",m,n);
        err(outbuff)
      }

    if (n<0)  return 0;

    if (m>=MAXM || n>=MAXN)   /* Too big for table */
      return computechoose(m+n-1,n);

    if (m==0)  return (n==0);               /* crd(0,n)=0 except crd(0,0)=1 */

    if (val=crdtable[m][n])  return val;
    else  return crdtable[m][n] = computechoose(m+n-1,n);
}

/******************************************************************************/

#ifdef __STDC__
int choose(int m, int n)
#else
int choose(m,n)
int m, n;
#endif
/*  Returns m choose n, looking it up in crdtable if possible.*/
{   if (n>=0 && n<=m+1)  return crd(m-n+1,n);
    return computechoose(m,n);
}

/******************************************************************************/

#ifdef __STDC__
int crdsum(int m, int n0, int n1)
#else
int crdsum(m,n0,n1)
int m, n0, n1;
#endif
/*  Returns crd(m,n0)+crd(m,n0+1)+...+crd(m,n1) = crd(m+1,n1) - crd(m+1,n0-1).
    This is the number of ways of choosing between n0 and n1 not necessarily
    distinct items from a set of m items.  Assumes n1>=n0-1.
*/
{   if (n1>n0)  return crd(n1+1,m) - crd(n0,m);
    if (n1==n0)  return crd(m,n0);
    return 0;
}

/******************************************************************************/

/*? This belongs elsewhere. */

#ifdef __STDC__
boolean samesize(int fr, int fc, int wantedfr, int wantedfc)
#else
boolean samesize(fr, fc, wantedfr, wantedfc)
int fr, fc, wantedfr, wantedfc;
#endif
/*  Checks to see if a matrix has the right dimensions.  The actual dimensions
    are fr by fc, the desired dimensions are wantedfr by wantedfc.  This takes
    into account the fact that MATLAB treats all empty matrices as 0 by 0,
    although it's possible that exactly one of wantedfr and wantedfc is 0.
*/
{   if (fr)  return (fr==wantedfr && fc==wantedfc);
    else  return (wantedfr==0 || wantedfc==0);
}
