/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file implements the function [h,nh]=cmp(f,nf,df,g,ng,dg,d).           */
/*                                                                            */
/* (written by Dean Hickerson, 8/24/94 - 9/29/94)                             */
/* (modified 8/28/2001 to work with complex numbers)                          */
/******************************************************************************/

#define shortname   "cmp"
#define fullname    "cmp(f,nf,df,g,ng,dg,d)"

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crdsum(int m, int n0, int n1);
extern boolean samesize(int fr, int fc, int wantedfr, int wantedfc);
extern void initvecdesc(int numsubvecs, int subveclth[]);
extern void red2lex(double f[], int deg0,int deg1,int outsize, double *fl[]);
extern void lex2red(double fl[], int deg0,int deg1,int outsize, double *f[]);
extern void red2lexcomplex(double f[], double fi[], int deg0, int deg1,
		int outsize, double *fl[], double *fli[]);
extern void lex2redcomplex(double fl[], double fli[], int deg0, int deg1,
		int outsize, double *f[], double *fi[]);
extern int squeezeinput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[]);
extern int squeezeinputcomplex(double f[], double fi[], int insize,
                        int deg0, int deg1, int outsize, boolean varneeded[]);
extern int squeezeoutput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[]);
extern int squeezeoutputcomplex(double f[], double fi[], int insize,
                         int deg0, int deg1, int outsize, boolean varneeded[]);
extern void compose(double f[], int fdeg0, int fdeg1, double g[],
    int gdeg0, int gdeg1, int ginsize, int midsize, int gr, int fr,
                                            double h[], int hdeg0, int hdeg1);
extern void composecomplex(double f[], double fi[], int fdeg0, int fdeg1,
    double g[], double gi[], int gdeg0, int gdeg1, int ginsize,
	int midsize, int gr, int fr,
	double h[], double hi[], int hdeg0, int hdeg1);
#else
extern int crdsum();
extern boolean samesize();
extern void initvecdesc();
extern void red2lex();
extern void lex2red();
extern void red2lexcomplex();
extern void lex2redcomplex();
extern int squeezeinput();
extern int squeezeinputcomplex();
extern int squeezeoutput();
extern int squeezeoutputcomplex();
extern void compose();
extern void composecomplex();
#endif

/******************************************************************************/

#ifdef __STDC__
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
#else
mexFunction(nlhs, plhs, nrhs, prhs)
int nlhs, nrhs;
mxArray *plhs[], *prhs[];
#endif
{   double *f, *nf, *df, *g, *ng, *dg, *d, *h, *nh, *fl, *gl, *hl,
           *fi, *gi, *hi, *fli, *gli, *hli;
    int fr, fc, nfr, nfc, dfr, dfc, gr, gc, ngr, ngc, dgr, dgc,
        dr, dc, fdeg0, fdeg1, gdeg0, gdeg1, hdeg0, hdeg1,
        fnumsubvecs, gnumsubvecs, hnumsubvecs,
        r, c, rc, i0, i1, i2, il, i, colsum,
        finsize, ginsize, midsize, foutsize, goutsize;
    int fsubveclth[MAXSUBVECS], gsubveclth[MAXSUBVECS];
    boolean fvarneeded[MAXINSIZE], gvarneeded[MAXINSIZE], complex;
    char outbuff[200];
    int inputargcounter;

    /***************************************************/
    /* Check that there are 7 inputs and <= 2 outputs. */
    /***************************************************/
    if (nrhs != 7)  err("ERROR: cmp requires 7 inputs.")
    if (nlhs > 2)   err("ERROR: cmp has only 2 outputs.")
    checktypes
	complex = mxIsComplex(prhs[0]);

    /****************************************/
    /* Get sizes of and pointers to inputs. */
    /****************************************/
    f  = mxGetPr(prhs[0]);  fr  = mxGetM(prhs[0]);  fc  = mxGetN(prhs[0]);
    nf = mxGetPr(prhs[1]);  nfr = mxGetM(prhs[1]);  nfc = mxGetN(prhs[1]);
    df = mxGetPr(prhs[2]);  dfr = mxGetM(prhs[2]);  dfc = mxGetN(prhs[2]);
    g  = mxGetPr(prhs[3]);  gr  = mxGetM(prhs[3]);  gc  = mxGetN(prhs[3]);
    ng = mxGetPr(prhs[4]);  ngr = mxGetM(prhs[4]);  ngc = mxGetN(prhs[4]);
    dg = mxGetPr(prhs[5]);  dgr = mxGetM(prhs[5]);  dgc = mxGetN(prhs[5]);
    d  = mxGetPr(prhs[6]);  dr  = mxGetM(prhs[6]);  dc  = mxGetN(prhs[6]);
	if (complex)
	  { fi = mxGetPi(prhs[0]);
	    gi = mxGetPi(prhs[3]);
	  }

    if (nfc < 2)  err(
      "ERROR: In cmp(f,nf,df,g,ng,dg,d), nf must have at least 2 columns.")

    if (ngc != 2)  err(
      "ERROR: In cmp(f,nf,df,g,ng,dg,d), ng must have exactly 2 columns.")

    if (nfr != ngr)  err(
"ERROR: In cmp(f,nf,df,g,ng,dg,d), nf and ng must have same number of rows.")

    checkrange(2,"df")
    checkrange(5,"dg")
    checkrange(6,"d")

    /************************************************/
    /* Get the min and max degrees for f, g, and h. */
    /************************************************/
    fdeg0 = df[0];  fdeg1 = df[dfc-1];
    gdeg0 = dg[0];  gdeg1 = dg[dgc-1];
    hdeg0 = d[0];   hdeg1 = d[dc-1];

    /*********************************************/
    /* Scan nf and ng to determine input size of */
    /* f and input and output sizes of g.        */
    /*********************************************/
    finsize=ginsize = fnumsubvecs=gnumsubvecs = goutsize = 0;

    /***********************************************/
    /* In the following loop:                      */
    /* i0 is an index into the first column of g,  */
    /* i1 is an index into the second column of g, */
    /* il is an index into the last column of f.   */
    /***********************************************/
    for (i0=0, i1=ngr, il=nfr*(nfc-1);  i0<ngr; i0++, i1++, il++)
      { /**************************************/
        /* Record info about input size of g. */
        /**************************************/
        if (ng[i1])  ginsize += (gsubveclth[gnumsubvecs++] = ng[i1]);

        /***********************************************************/
        /* Record info about input size of f and output size of g. */
        /***********************************************************/
        if (nf[il])
          { fsubveclth[fnumsubvecs++] = nf[il];
            if (ng[i0])         /* nf[il] != 0,  ng[i0] != 0 */
              if (nf[il] == ng[i0])
                for (i=nf[il]; i; i--)
                  fvarneeded[finsize++] = gvarneeded[goutsize++] = TRUE;
              else
#ifdef THINK_C
                err3("ERROR: In cmp(f,nf,df,g,ng,dg,d), corresponding ",
                  "elements of first column\rof ng and last column of nf ",
                  "must be equal or zero.")
#else
                err3("ERROR: In cmp(f,nf,df,g,ng,dg,d), corresponding ",
                  "elements of first column\nof ng and last column of nf ",
                  "must be equal or zero.")
#endif
            else                /* nf[il] != 0,  ng[i0] = 0 */
              for (i=nf[il]; i; i--)  fvarneeded[finsize++]  = FALSE;
          }
        else if (ng[i0])            /* nf[il] = 0,  ng[i0] != 0 */
          for (i=ng[i0]; i; i--)  gvarneeded[goutsize++] = FALSE;
      }

    if (fdeg0 < 0 || fdeg1 < fdeg0)  err(
      "ERROR: In cmp(f,nf,[df0 df1],g,ng,dg,d), must have 0 <= df0 <= df1.")

    if (gdeg0 < 0 || gdeg1 < gdeg0)  err(
      "ERROR: In cmp(f,nf,df,g,ng,[dg0 dg1],d), must have 0 <= dg0 <= dg1.")

    if (hdeg0 < 0 || hdeg1 < hdeg0)  err(
      "ERROR: In cmp(f,nf,df,g,ng,dg,[d0 d1]), must have 0 <= d0 <= d1.")

    /*******************************************************************/
    /* Compute the output size from nf; i.e. the product of all column */
    /* sums except the last.  This must equal the number of rows of f. */
    /*******************************************************************/
    for (rc=c=0, foutsize=1; c<nfc-1; c++)
      { for (r=colsum=0; r<nfr; r++, rc++)  colsum += nf[rc];
        foutsize *= colsum;
      }

    /*************************************************************/
    /* Check that f and g have the sizes specified by nf and ng. */
    /*************************************************************/
    if (!samesize(fr, fc, foutsize, crdsum(finsize,fdeg0,fdeg1)))
      { sprintf(outbuff,
"ERROR: In cmp(f,nf,df,g,ng,dg,d), f should be %ld by %ld, not %ld by %ld.",
          foutsize, crdsum(finsize,fdeg0,fdeg1), fr, fc);
        err(outbuff)
      }

    if (!samesize(gr, gc, goutsize, crdsum(ginsize,gdeg0,gdeg1)))
      { sprintf(outbuff,
"ERROR: In cmp(f,nf,df,g,ng,dg,d), g should be %ld by %ld, not %ld by %ld.",
          goutsize, crdsum(ginsize,gdeg0,gdeg1), gr, gc);
        err(outbuff)
      }

    /****************************************************************/
    /* If 2 output args, compute nh.  nh has same size as nf.  Last */
    /* column is copied from last column of ng, others from nf.     */
    /****************************************************************/
    if (nlhs == 2)
      { plhs[1] = mxCreateDoubleMatrix(nfr,nfc,mxREAL);
        nh = mxGetPr(plhs[1]);
        for (i=nfr*(nfc-1)-1; i>=0; i--)
          nh[i] = nf[i];
        for (i0=0, i1=ngr, il=nfr*(nfc-1);  i0<ngr; i0++, i1++, il++)
          nh[il] = ng[i1];
      }

    /**************************************************/
    /* Convert f and g to lexicographic reduced form. */
    /**************************************************/
	if (complex)
	  { initvecdesc(fnumsubvecs, fsubveclth);
		fl = NULL;
		red2lexcomplex(f, fi, fdeg0, fdeg1, foutsize, &fl, &fli);
        initvecdesc(gnumsubvecs, gsubveclth);
		gl = NULL;
		red2lexcomplex(g, gi, gdeg0, gdeg1, goutsize, &gl, &gli);
	  }
    else
	  { initvecdesc(fnumsubvecs, fsubveclth);
    	fl = NULL;
		red2lex(f, fdeg0, fdeg1, foutsize, &fl);
    	initvecdesc(gnumsubvecs, gsubveclth);
    	gl = NULL;
		red2lex(g, gdeg0, gdeg1, goutsize, &gl);
	  }

    /*****************************************************/
    /* If necessary, squeeze input of f and output of g. */
    /*****************************************************/
	if (complex)
	  { squeezeinputcomplex(fl,fli,finsize,fdeg0,fdeg1,foutsize,fvarneeded);
        midsize = squeezeoutputcomplex(gl,gli,ginsize,gdeg0,gdeg1,goutsize,
		                                                            gvarneeded);
	  }
    else
	  { squeezeinput(fl,finsize,fdeg0,fdeg1,foutsize,fvarneeded);
        midsize = squeezeoutput(gl,ginsize,gdeg0,gdeg1,goutsize,gvarneeded);
	  }

    /**********************/
    /* Compose fl and gl. */
    /**********************/
    hl = mxCalloc(foutsize*crdsum(ginsize,hdeg0,hdeg1), sizeof(double));
    if (complex)
	  hli = mxCalloc(foutsize*crdsum(ginsize,hdeg0,hdeg1), sizeof(double));

	if (complex)
      composecomplex(fl, fli, fdeg0, fdeg1,
                     gl, gli, gdeg0, gdeg1,
                     ginsize, midsize, goutsize, foutsize,
                     hl, hli, hdeg0, hdeg1);
	else
      compose(fl, fdeg0, fdeg1,
              gl, gdeg0, gdeg1,
              ginsize, midsize, goutsize, foutsize,
              hl, hdeg0, hdeg1);

    mxFree(fl);
    mxFree(gl);

	if (complex)
	  { mxFree(fli);
        mxFree(gli);
	  }

    /***************************************/
    /* Allocate (and clear) output matrix. */
    /***************************************/
	if (complex)
      plhs[0] = mxCreateDoubleMatrix(foutsize, crdsum(ginsize,hdeg0,hdeg1),
																	mxCOMPLEX);
    else
      plhs[0] = mxCreateDoubleMatrix(foutsize, crdsum(ginsize,hdeg0,hdeg1),
																	mxREAL);
    h = mxGetPr(plhs[0]);
	if (complex)
      hi = mxGetPi(plhs[0]);

    /************************************/
    /* Convert product to reduced form. */
    /************************************/
	if (complex)
	  { initvecdesc(gnumsubvecs, gsubveclth);
        lex2redcomplex(hl, hli, hdeg0, hdeg1, foutsize, &h, &hi);
	  }
	else
	  { initvecdesc(gnumsubvecs, gsubveclth);
        lex2red(hl, hdeg0, hdeg1, foutsize, &h);
	  }
}
