/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file contains various routines used by cmp(f,nf,df,g,ng,dg,d).        */
/*                                                                            */
/* (written by Dean Hickerson, 8/27/94)                                       */
/* (modified 8/28/2001 to work with complex numbers)                          */
/******************************************************************************/

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
void addcomph(double f[], int fdeg, double g[], int gdeg0, int gdeg1,
            int ginsize, int midsize, int goutsize, int foutsize,
            double h[], int hdeg0, int hdeg1, double *monoming[]);
void compose(double f[], int fdeg0, int fdeg1,
             double g[], int gdeg0, int gdeg1,
             int ginsize, int midsize, int goutsize, int foutsize,
             double h[], int hdeg0, int hdeg1);
void addcomphcomplex(double f[], double fi[], int fdeg,
            double g[], double gi[], int gdeg0, int gdeg1,
            int ginsize, int midsize, int goutsize, int foutsize,
            double h[], double hi[], int hdeg0, int hdeg1,
			double *monoming[], double *monomingi[]);
void composecomplex(double f[], double fi[], int fdeg0, int fdeg1,
             double g[], double gi[], int gdeg0, int gdeg1,
             int ginsize, int midsize, int goutsize, int foutsize,
             double h[], double hi[], int hdeg0, int hdeg1);
#endif

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
extern boolean initmonom(struct monom *m, int deg, int insz);
extern boolean nextmonom(struct monom *m, int *changeptr);
extern void printmonom(struct monom *m);
extern void addscalarprod(double a[], int adeg0, int adeg1, int astep,
                double b[], int bdeg0, int bdeg1, int bstep,
                int insize, double prod[],
				int pdeg0, int pdeg1, int pstep, boolean subtract);
extern void addscalarprodcomplex(double a[], double ai[],
                                               int adeg0, int adeg1, int astep,
                          double b[], double bi[],
						                       int bdeg0, int bdeg1, int bstep,
                          int insize, double prod[], double prodi[],
				          int pdeg0, int pdeg1, int pstep, boolean subtract);
#else
extern int crd();
extern int crdsum();
extern boolean initmonom();
extern boolean nextmonom();
extern void printmonom();
extern void addscalarprod();
extern void addscalarprodcomplex();
#endif

/******************************************************************************/

#ifdef __STDC__
void addcomph(double f[], int fdeg, double g[], int gdeg0, int gdeg1,
            int ginsize, int midsize, int goutsize, int foutsize,
            double h[], int hdeg0, int hdeg1, double *monoming[])
#else
void addcomph(f, fdeg, g, gdeg0, gdeg1, ginsize, midsize, goutsize, foutsize,
            h, hdeg0, hdeg1, monoming)
double f[], g[], h[];
int fdeg, gdeg0, gdeg1, ginsize, midsize, goutsize, foutsize, hdeg0, hdeg1;
double *monoming[];
#endif
/*  g[0:goutsize*crdsum(ginsize,gdeg0,gdeg1)-1] represents a midsize dimensional
    vector-valued function of degrees gdeg0 to gdeg1 in ginsize variables.  Here
    midsize <= goutsize;  only the top midsize rows of g are actually used.
    f[0:foutsize*crd(midsize,fdeg)-1] represents an foutsize dimensional vector
    valued function of degree fdeg in midsize variables.
    h[0:foutsize*crdsum(ginsize,hdeg0,hdeg1)-1] represents an foutsize
    dimensional vector-valued function of degree fdeg*gdeg in ginsize variables.
    This routine adds the composition f(g) to h.
    monoming[0:fdeg] is an array of pointers to storage for the f-monomials
    with the outputs of g substituted for the inputs of f.  monoming[i] has
    room for a real-valued polynomial of degrees i*gdeg0 to i*gdeg1 in ginsize
    variables.
    It is assumed that the degree range [hdeg0, hdeg1] is contained in the
    range [fdeg*gdeg0, fdeg*gdeg1].  This routine should only be called by
    compose(), which ensures that this is true.
*/
{   int change, i, r, c, hc, k;
    double *fptr, *fptr0, *hptr, *mptr;
    double mval;
    struct monom m;
    int mindeg[MAXDEGREE+1], maxdeg[MAXDEGREE+1];

    hc = crdsum(ginsize,hdeg0,hdeg1);   /* # of columns of h */
    fptr0 = f;

    if (!initmonom(&m, fdeg, midsize))  return;
    change = 0;
    *monoming[0] = 1;

    /**************************************************************************/
    /* For each monomial in f, we compute its value when the input variables  */
    /* of f are replaced by the output variables of g.  If f is the product   */
    /* x[m[0]] * ... * x[m[fdeg-1]],  then, for 0<=i<=fdeg, monoming[i] is    */
    /* the product  g[m[0]] * ... * g[m[i-1]].  All we really want is the     */
    /* portion of monoming[fdeg] with degrees hdeg0 to hdeg1, so in           */
    /* monoming[i] we only need degrees  max(i*gdeg0, hdeg0-(fdeg-i)*gdeg1)   */
    /* to  min(i*gdeg1, hdeg1-(fdeg-i)*gdeg0).                                */
    /* Note that for i=0 the range is [0,0]; for i=fdeg it's  [hdeg0,hdeg1].  */
    /**************************************************************************/
    for (i=0; i<=fdeg; i++)
      { mindeg[i] = max(i*gdeg0, hdeg0-(fdeg-i)*gdeg1);
        maxdeg[i] = min(i*gdeg1, hdeg1-(fdeg-i)*gdeg0);
      }

    do                                  /* Once for each f-monomial */
      { for (i=change+1; i<=fdeg; i++)
          { /*********************/
            /* Clear monoming[i] */
            /*********************/
            for (c=crdsum(ginsize, mindeg[i], maxdeg[i]), mptr=monoming[i];
                                                                        c; c--)
              *mptr++ = 0;

            addscalarprod(monoming[i-1], mindeg[i-1], maxdeg[i-1], 1,
                g+m.factor[i-1], gdeg0, gdeg1, goutsize, ginsize,
                monoming[i], mindeg[i], maxdeg[i], 1, FALSE);
          }

        /**********************************************************************/
        /* Add monoming[fdeg] times column of f to h.  Conceptually,          */
        /* r runs from  0  to  foutsize-1,                                    */
        /* c runs from  0  to  crdsum(ginsize,hdeg0,hdeg1)-1,                 */
        /* and we do                                                          */
        /*     h[r+c*foutsize] += f[r+m.index*foutsize] * monoming[fdeg][c].  */
        /**********************************************************************/
        for (c=hc, hptr = h, mptr = monoming[fdeg]; c; c--)
          { fptr = fptr0;
            mval = *mptr;

            for (r=foutsize; r; r--)
#ifdef THINK_C
              /***********************************************************/
              /* There's a bug in the Think C compiler which would cause */
              /* hptr to be pre- rather than post-incremented.           */
              /***********************************************************/
              { *hptr += *fptr++ * mval;
                hptr++;
              }
#else
              *hptr++ += *fptr++ *mval;
#endif
            mptr++;
          }

        fptr0 += foutsize;
      }
    while (nextmonom(&m, &change));
}

/******************************************************************************/

#ifdef __STDC__
void compose(double f[], int fdeg0, int fdeg1,
             double g[], int gdeg0, int gdeg1,
             int ginsize, int midsize, int goutsize, int foutsize,
             double h[], int hdeg0, int hdeg1)
#else
void compose(f, fdeg0, fdeg1, g, gdeg0, gdeg1, ginsize, midsize, goutsize,
             foutsize, h, hdeg0, hdeg1)
double f[], g[], h[];
int fdeg0, fdeg1, gdeg0, gdeg1, ginsize, midsize, goutsize, foutsize,
    hdeg0, hdeg1;
#endif
/*  g[0:goutsize*crdsum(ginsize,gdeg0,gdeg1)-1] represents a midsize
    dimensional vector-valued function of degrees gdeg0 through gdeg1
    in ginsize variables.  Here  midsize <= goutsize;  only the top
    midsize rows of g are actually used.
    f[0:foutsize*crdsum(midsize,fdeg0,fdeg1)-1] represents an foutsize
    dimensional vector-valued function of degrees fdeg0 through fdeg1
    in midsize variables.
    This routine forms that part of the composition f(g) with degrees
    hdeg0 through hdeg1, storing it in
    h[0:foutsize*crdsum(ginsize,hdeg0,hdeg1)-1].
*/
{   double *monoming[MAXDEGREE+1];
    int i, fdeg, mindeg, maxdeg;

    /***********************************/
    /* Initialize array of pointers to */
    /* f-monomials in the outputs of g */
    /***********************************/
    for (i=0; i<=fdeg1; i++)
      monoming[i] = mxCalloc(crdsum(ginsize,i*gdeg0,i*gdeg1), sizeof(double));

    /**************************************************************************/
    /* For each degree fdeg in [fdeg0,fdeg1] for which there's some overlap   */
    /* between the intervals [fdeg*gdeg0, fdeg*gdeg1] and [hdeg0,hdeg1],      */
    /* add to h the composition of the degree fdeg part of f with g.          */
    /**************************************************************************/
    for (fdeg=fdeg0; fdeg<=fdeg1; fdeg++)
      if (hdeg0 <= fdeg*gdeg1 && fdeg*gdeg0 <= hdeg1)
        { mindeg = max(hdeg0, fdeg*gdeg0);
          maxdeg = min(hdeg1, fdeg*gdeg1);
          addcomph(f+foutsize*crdsum(midsize,fdeg0,fdeg-1), fdeg,
                 g, gdeg0, gdeg1,
                 ginsize, midsize, goutsize, foutsize,
                 h+foutsize*crdsum(ginsize,hdeg0,mindeg-1), mindeg, maxdeg,
                 monoming);
        }

    for (i=0; i<=fdeg1; i++)
      mxFree(monoming[i]);
}

/******************************************************************************/

#ifdef __STDC__
void addcomphcomplex(double f[], double fi[], int fdeg,
            double g[], double gi[], int gdeg0, int gdeg1,
            int ginsize, int midsize, int goutsize, int foutsize,
            double h[], double hi[], int hdeg0, int hdeg1,
			double *monoming[], double *monomingi[])
#else
void addcomphcomplex(f, fi, fdeg, g, gi, gdeg0, gdeg1, ginsize, midsize,
                  goutsize, foutsize, h, hi, hdeg0, hdeg1, monoming, monomingi)
double f[], fi[], g[], gi[], h[], hi[];
int fdeg, gdeg0, gdeg1, ginsize, midsize, goutsize, foutsize, hdeg0, hdeg1;
double *monoming[], *monomingi[];
#endif
/*  g[0:goutsize*crdsum(ginsize,gdeg0,gdeg1)-1] represents a midsize dimensional
    vector-valued function of degrees gdeg0 to gdeg1 in ginsize variables.  Here
    midsize <= goutsize;  only the top midsize rows of g are actually used.
    f[0:foutsize*crd(midsize,fdeg)-1] represents an foutsize dimensional vector
    valued function of degree fdeg in midsize variables.
    h[0:foutsize*crdsum(ginsize,hdeg0,hdeg1)-1] represents an foutsize
    dimensional vector-valued function of degree fdeg*gdeg in ginsize variables.
    This routine adds the composition f(g) to h.
    monoming[0:fdeg] is an array of pointers to storage for the f-monomials
    with the outputs of g substituted for the inputs of f.  monoming[i] has
    room for a real-valued polynomial of degrees i*gdeg0 to i*gdeg1 in ginsize
    variables.
    It is assumed that the degree range [hdeg0, hdeg1] is contained in the
    range [fdeg*gdeg0, fdeg*gdeg1].  This routine should only be called by
    compose(), which ensures that this is true.
*/
{   int change, i, r, c, hc, k;
    double *fptr, *fptr0, *hptr, *mptr, *fiptr, *fiptr0, *hiptr, *miptr;
    double mval, mvali;
    struct monom m;
    int mindeg[MAXDEGREE+1], maxdeg[MAXDEGREE+1];

    hc = crdsum(ginsize,hdeg0,hdeg1);   /* # of columns of h */
    fptr0 = f;
	fiptr0 = fi;

    if (!initmonom(&m, fdeg, midsize))  return;
    change = 0;
    *monoming[0] = 1;
    *monomingi[0] = 0;

    /**************************************************************************/
    /* For each monomial in f, we compute its value when the input variables  */
    /* of f are replaced by the output variables of g.  If f is the product   */
    /* x[m[0]] * ... * x[m[fdeg-1]],  then, for 0<=i<=fdeg, monoming[i] is    */
    /* the product  g[m[0]] * ... * g[m[i-1]].  All we really want is the     */
    /* portion of monoming[fdeg] with degrees hdeg0 to hdeg1, so in           */
    /* monoming[i] we only need degrees  max(i*gdeg0, hdeg0-(fdeg-i)*gdeg1)   */
    /* to  min(i*gdeg1, hdeg1-(fdeg-i)*gdeg0).                                */
    /* Note that for i=0 the range is [0,0]; for i=fdeg it's  [hdeg0,hdeg1].  */
    /**************************************************************************/
    for (i=0; i<=fdeg; i++)
      { mindeg[i] = max(i*gdeg0, hdeg0-(fdeg-i)*gdeg1);
        maxdeg[i] = min(i*gdeg1, hdeg1-(fdeg-i)*gdeg0);
      }

    do                                  /* Once for each f-monomial */
      { for (i=change+1; i<=fdeg; i++)
          { /*********************/
            /* Clear monoming[i] */
            /*********************/
            for (c=crdsum(ginsize, mindeg[i], maxdeg[i]),
			     mptr=monoming[i], miptr=monomingi[i];   c;   c--)
              { *mptr++ = 0;
			    *miptr++ = 0;
			  }

            addscalarprodcomplex(monoming[i-1], monomingi[i-1],
			                                 mindeg[i-1], maxdeg[i-1], 1,
                g+m.factor[i-1], gi+m.factor[i-1], gdeg0, gdeg1,
				                             goutsize, ginsize,
                monoming[i], monomingi[i], mindeg[i], maxdeg[i], 1, FALSE);
          }

        /**********************************************************************/
        /* Add monoming[fdeg] times column of f to h.  Conceptually,          */
        /* r runs from  0  to  foutsize-1,                                    */
        /* c runs from  0  to  crdsum(ginsize,hdeg0,hdeg1)-1,                 */
        /* and we do                                                          */
        /*     h[r+c*foutsize] += f[r+m.index*foutsize] * monoming[fdeg][c].  */
        /**********************************************************************/
        for (c=hc, hptr = h, hiptr = hi,
		     mptr = monoming[fdeg], miptr = monomingi[fdeg];   c;   c--)
          { fptr = fptr0;
		    fiptr = fiptr0;
            mval = *mptr;
			mvali = *miptr;

            for (r=foutsize; r; r--)
              { *hptr  += *fptr  * mval;
			    *hptr  -= *fiptr * mvali;
				*hiptr += *fptr  * mvali;
				*hiptr += *fiptr * mval;
				fptr++;  fiptr++;  hptr++;  hiptr++;
			  }
            mptr++;
            miptr++;
          }

        fptr0 += foutsize;
        fiptr0 += foutsize;
      }
    while (nextmonom(&m, &change));
}

/******************************************************************************/

#ifdef __STDC__
void composecomplex(double f[], double fi[], int fdeg0, int fdeg1,
             double g[], double gi[], int gdeg0, int gdeg1,
             int ginsize, int midsize, int goutsize, int foutsize,
             double h[], double hi[], int hdeg0, int hdeg1)
#else
void composecomplex(f, fi, fdeg0, fdeg1, g, gi, gdeg0, gdeg1, ginsize, midsize,
                    goutsize, foutsize, h, hi, hdeg0, hdeg1)
double f[], fi[], g[], gi[], h[], hi[];
int fdeg0, fdeg1, gdeg0, gdeg1, ginsize, midsize, goutsize, foutsize,
    hdeg0, hdeg1;
#endif
/*  g[0:goutsize*crdsum(ginsize,gdeg0,gdeg1)-1] represents a midsize
    dimensional vector-valued function of degrees gdeg0 through gdeg1
    in ginsize variables.  Here  midsize <= goutsize;  only the top
    midsize rows of g are actually used.
    f[0:foutsize*crdsum(midsize,fdeg0,fdeg1)-1] represents an foutsize
    dimensional vector-valued function of degrees fdeg0 through fdeg1
    in midsize variables.
    This routine forms that part of the composition f(g) with degrees
    hdeg0 through hdeg1, storing it in
    h[0:foutsize*crdsum(ginsize,hdeg0,hdeg1)-1].
*/
{   double *monoming[MAXDEGREE+1], *monomingi[MAXDEGREE+1];
    int i, fdeg, mindeg, maxdeg;

    /***********************************/
    /* Initialize array of pointers to */
    /* f-monomials in the outputs of g */
    /***********************************/
    for (i=0; i<=fdeg1; i++)
      { monoming[i] =mxCalloc(crdsum(ginsize,i*gdeg0,i*gdeg1), sizeof(double));
        monomingi[i]=mxCalloc(crdsum(ginsize,i*gdeg0,i*gdeg1), sizeof(double));
	  }

    /**************************************************************************/
    /* For each degree fdeg in [fdeg0,fdeg1] for which there's some overlap   */
    /* between the intervals [fdeg*gdeg0, fdeg*gdeg1] and [hdeg0,hdeg1],      */
    /* add to h the composition of the degree fdeg part of f with g.          */              
    /**************************************************************************/
    for (fdeg=fdeg0; fdeg<=fdeg1; fdeg++)
      if (hdeg0 <= fdeg*gdeg1 && fdeg*gdeg0 <= hdeg1)
        { mindeg = max(hdeg0, fdeg*gdeg0);
          maxdeg = min(hdeg1, fdeg*gdeg1);
          addcomphcomplex(f+foutsize*crdsum(midsize,fdeg0,fdeg-1),
		                  fi+foutsize*crdsum(midsize,fdeg0,fdeg-1), fdeg,
                 g, gi, gdeg0, gdeg1,
                 ginsize, midsize, goutsize, foutsize,
                 h+foutsize*crdsum(ginsize,hdeg0,mindeg-1),
				 hi+foutsize*crdsum(ginsize,hdeg0,mindeg-1), mindeg, maxdeg,
                 monoming, monomingi);
        }

    for (i=0; i<=fdeg1; i++)
      { mxFree(monoming[i]);
        mxFree(monomingi[i]);
	  }
}
