/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file implements the function [h,nh]=dd(f,nf,df,g,ng,dg,d).            */
/*                                                                            */
/* (written by Dean Hickerson, 8/24/94)                                       */
/* (modified 8/31/2001 to work with complex numbers)                          */
/******************************************************************************/

#define shortname   "dd"
#define fullname    "dd(f,nf,df,g,ng,dg,d)"

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crdsum(int m, int n0, int n1);
extern boolean samesize(int fr, int fc, int wantedfr, int wantedfc);
extern void initvecdesc(int numsubvecs, int subveclth[]);
extern void red2lex(double f[], int deg0,int deg1,int outsize, double *fl[]);
extern void lex2red(double fl[], int deg0,int deg1,int outsize, double *f[]);
extern void red2lexcomplex(double f[], double fi[], int deg0, int deg1,
		int outsize, double *fl[], double *fli[]);
extern void lex2redcomplex(double fl[], double fli[], int deg0, int deg1,
		int outsize, double *f[], double *fi[]);
extern void jacobian(double f[], int fdeg0, int fdeg1, int finsize,
        int foutsize, boolean varneeded[], int jdeg0, int jdeg1, double *j[]);
extern void jacobiancomplex(double f[], double fi[], int fdeg0, int fdeg1,
		int finsize, int foutsize, boolean varneeded[], int jdeg0, int jdeg1,
		double *j[], double *ji[]);
extern int squeezeoutput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[]);
extern int squeezeoutputcomplex(double f[], double fi[], int insize,
                         int deg0, int deg1, int outsize, boolean varneeded[]);
extern extendinputs(double f[], int insize, int deg0, int deg1,
         int outsize, boolean varpresent[], double *fext[], boolean *extending);
extern void extendinputscomplex(double f[], double fi[], int insize,
                         int deg0, int deg1, int outsize, boolean varpresent[],
                         double *fext[], double *fexti[], boolean *extending);
extern void addscalarprod(double a[], int adeg0, int adeg1, int astep,
                double b[], int bdeg0, int bdeg1, int bstep,
                int insize, double prod[],
                int pdeg0, int pdeg1, int pstep, boolean subtract);
extern void addscalarprodcomplex(double a[], double ai[],
                                               int adeg0, int adeg1, int astep,
                          double b[], double bi[],
						                       int bdeg0, int bdeg1, int bstep,
                          int insize, double prod[], double prodi[],
				          int pdeg0, int pdeg1, int pstep, boolean subtract);
#else
extern int crdsum();
extern boolean samesize();
extern void initvecdesc();
extern void red2lex();
extern void lex2red();
extern void red2lexcomplex();
extern void lex2redcomplex();
extern void jacobian();
extern void jacobiancomplex();
extern int squeezeoutput();
extern int squeezeoutputcomplex();
extern void extendinputs();
extern void extendinputscomplex();
extern void addscalarprod();
#endif

/******************************************************************************/

#ifdef __STDC__
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
#else
mexFunction(nlhs, plhs, nrhs, prhs)
int nlhs, nrhs;
Matrix *plhs[], *prhs[];
#endif
{   double *f, *fi, *nf, *df, *g, *gi, *ng, *dg, *d, *h, *hi, *nh, *fl, *fli,
    *gl, *gli, *hl, *hli, *hle, *hlei, *j, *ji, *jext, *jexti, *gle, *glei;
    int fr, fc, nfr, nfc, dfr, dfc, gr, gc, ngr, ngc, dgr, dgc,
        dr, dc, fdeg0, fdeg1, gdeg0, gdeg1, hdeg0, hdeg1, jdeg0, jdeg1,
        fnumsubvecs, gnumsubvecs, hnumsubvecs,
        r, c, rc, i0, i1, i2, il, i, colsum, v,
        fin_size, fin_int_gout_size, all_size, fin_union_gin_size, gout_size,
        gin_size, foutsize, val;
    int fsubveclth[MAXSUBVECS], gsubveclth[MAXSUBVECS], hsubveclth[MAXSUBVECS];
    boolean fin_int_gout_among_fin[MAXINSIZE],
        fin_among_fin_union_gin[MAXINSIZE], gin_among_fin_union_gin[MAXINSIZE],
        fin_int_gout_among_gout[MAXINSIZE], fin_union_gin_among_all[MAXINSIZE];
    boolean extending, complex;
    char outbuff[200];
    int inputargcounter;

    /***************************************************/
    /* Check that there are 7 inputs and <= 2 outputs. */
    /***************************************************/
    if (nrhs != 7)  err("ERROR: dd requires 7 inputs.")
    if (nlhs > 2)   err("ERROR: dd has only 2 outputs.")
    checktypes
	complex = mxIsComplex(prhs[0]);

    /****************************************/
    /* Get sizes of and pointers to inputs. */
    /****************************************/
    f  = mxGetPr(prhs[0]);  fr  = mxGetM(prhs[0]);  fc  = mxGetN(prhs[0]);
    nf = mxGetPr(prhs[1]);  nfr = mxGetM(prhs[1]);  nfc = mxGetN(prhs[1]);
    df = mxGetPr(prhs[2]);  dfr = mxGetM(prhs[2]);  dfc = mxGetN(prhs[2]);
    g  = mxGetPr(prhs[3]);  gr  = mxGetM(prhs[3]);  gc  = mxGetN(prhs[3]);
    ng = mxGetPr(prhs[4]);  ngr = mxGetM(prhs[4]);  ngc = mxGetN(prhs[4]);
    dg = mxGetPr(prhs[5]);  dgr = mxGetM(prhs[5]);  dgc = mxGetN(prhs[5]);
    d  = mxGetPr(prhs[6]);  dr  = mxGetM(prhs[6]);  dc  = mxGetN(prhs[6]);

	if (complex)
	  { fi = mxGetPi(prhs[0]);
	    gi = mxGetPi(prhs[3]);
	  }

    if (nfc < 2)  err(
      "ERROR: In dd(f,nf,df,g,ng,dg,d), nf must have at least 2 columns.")

    if (ngc != 2)  err(
      "ERROR: In dd(f,nf,df,g,ng,dg,d), ng must have exactly 2 columns.")

    if (nfr != ngr)  err(
"ERROR: In dd(f,nf,df,g,ng,dg,d), nf and ng must have same number of rows.")

    checkrange(2, "df")
    checkrange(5, "dg")
    checkrange(6, "d")

    /************************************************/
    /* Get the min and max degrees for f, g, and h. */
    /************************************************/
    fdeg0 = df[0];  fdeg1 = df[dfc-1];
    gdeg0 = dg[0];  gdeg1 = dg[dgc-1];
    hdeg0 = d[0];   hdeg1 = d[dc-1];

    /*********************************************/
    /* Scan nf and ng to determine input size of */
    /* f and input and output sizes of g.        */
    /*********************************************/
    fin_size = fin_int_gout_size = all_size = fin_union_gin_size = gout_size =
    gin_size = fnumsubvecs = gnumsubvecs = hnumsubvecs = 0;

    /*************************************************/
    /* Record info about sizes of f and g.           */
    /* In the following loop:                        */
    /* i0 is an index into the first column of ng,   */
    /* i1 is an index into the second column of ng,  */
    /* il is an index into the last column of nf.    */
    /*************************************************/
    for (i0=0, i1=nfr, il=nfr*(nfc-1);  i0<nfr;  i0++, i1++, il++)
      { /*****************************************************/
        /* Check that nf[il], ng[i0], ng[i1] are equal or 0, */
        /* and set val to the common nonzero value, if any.  */
        /*****************************************************/

#define fin     nf[il]
#define gin     ng[i1]
#define gout    ng[i0]

        val = fin;

        if (gin)
#ifdef THINK_C
          if (val && (gin != val))  err2(
            "ERROR: In dd(f,nf,df,g,ng,dg,d), corresponding elements in\r",
            "last column of nf and both columns of ng must be equal or zero.")
#else
          if (val && (gin != val))  err2(
            "ERROR: In dd(f,nf,df,g,ng,dg,d), corresponding elements in\n",
            "last column of nf and both columns of ng must be equal or zero.")
#endif
          else  val = gin;

        if (gout)
#ifdef THINK_C
          if (val && (gout != val))  err2(
            "ERROR: In dd(f,nf,df,g,ng,dg,d), corresponding elements in\r",
            "last column of nf and both columns of ng must be equal or zero.")
#else
          if (val && (gout != val))  err2(
            "ERROR: In dd(f,nf,df,g,ng,dg,d), corresponding elements in\n",
            "last column of nf and both columns of ng must be equal or zero.")
#endif
          else  val = gout;

        /****************************************************************/
        /* Compute sizes of various boolean combinations of input of f  */
        /* and input and output of g.  Also fill several boolean arrays */
        /* with info on which variables are in which sets.  And         */
		/* determine the subvector lengths of f, g, and h.              */
        /****************************************************************/
        if (fin)                /* If vector is in input(f) */
          { for (i=val; i; i--)
              fin_int_gout_among_fin[fin_size++] = (gout != 0);
            fsubveclth[fnumsubvecs++] = val;
          }

        if (gin)                /* If vector is in input(g) */
          gin_size += (gsubveclth[gnumsubvecs++] = val);

        if (fin || gin || gout)
          { for (i=val; i; i--)
              fin_union_gin_among_all[all_size++] = (fin || gin);
            hsubveclth[hnumsubvecs++] = val;
          }

        if (gout)               /* If vector is in output(g) */
          for (i=val; i; i--)
            fin_int_gout_among_gout[gout_size++] = (fin != 0);

        if (fin && gout)    /* If in input(f) intersect output(g) */
          fin_int_gout_size += val;

        if (fin || gin) /* If in input(f) union input(g) */
          for (i=val; i; i--)
            { fin_among_fin_union_gin[fin_union_gin_size] = (fin != 0);
              gin_among_fin_union_gin[fin_union_gin_size++] = (gin != 0);
            }
      }

    if (fdeg0 < 0 || fdeg1 < fdeg0)  err(
      "ERROR: In dd(f,nf,[df0 df1],g,ng,dg,d), must have 0 <= df0 <= df1.")

    if (gdeg0 < 0 || gdeg1 < gdeg0)  err(
      "ERROR: In dd(f,nf,df,g,ng,[dg0 dg1],d), must have 0 <= dg0 <= dg1.")

    if (hdeg0 < 0 || hdeg1 < hdeg0)  err(
      "ERROR: In dd(f,nf,df,g,ng,dg,[d0 d1]), must have 0 <= d0 <= d1.")

    /*******************************************************************/
    /* Compute the output size from nf; i.e. the product of all column */
    /* sums except the last.  This must equal the number of rows of f. */
    /*******************************************************************/
    for (rc=c=0, foutsize=1; c<nfc-1; c++)
      { for (r=colsum=0; r<nfr; r++, rc++)  colsum += nf[rc];
        foutsize *= colsum;
      }

    /*************************************************************/
    /* Check that f and g have the sizes specified by nf and ng. */
    /*************************************************************/
    if (!samesize(fr, fc, foutsize, crdsum(fin_size,fdeg0,fdeg1)))
      { sprintf(outbuff,
          "ERROR: In %s, f should be %ld by %ld, not %ld by %ld.",
              fullname, foutsize, (int)crdsum(fin_size,fdeg0,fdeg1), fr, fc);
        err(outbuff)
      }

    if (!samesize(gr, gc, gout_size, crdsum(gin_size,gdeg0,gdeg1)))
      { sprintf(outbuff,
          "ERROR: In %s, g should be %ld by %ld, not %ld by %ld.",
          fullname, gout_size, (int)crdsum(gin_size,gdeg0,gdeg1), gr, gc);
        err(outbuff)
      }

    /****************************************************************/
    /* If 2 output args, compute nh.  nh has same size as nf.  Last */
    /* column is max of last column of nf and both columns of ng;   */
    /* others are copied from nf.                                   */
    /****************************************************************/
    if (nlhs == 2)
      { plhs[1] = mxCreateDoubleMatrix(nfr,nfc,mxREAL);
        nh = mxGetPr(plhs[1]);
        for (i=nfr*(nfc-1)-1; i>=0; i--)
          nh[i] = nf[i];
        for (i0=0, i1=ngr, il=ngr*(ngc-1);  i0<ngr; i0++, i1++, il++)
          nh[il] = max3(nf[il],ng[i0],ng[i1]);
      }

    /**************************************************/
    /* Convert f and g to lexicographic reduced form. */
    /**************************************************/
    if (complex)
	  { initvecdesc(fnumsubvecs, fsubveclth);
        fl = NULL;
		red2lexcomplex(f, fi, fdeg0, fdeg1, foutsize, &fl, &fli);
        initvecdesc(gnumsubvecs, gsubveclth);
        gl = NULL;
		red2lexcomplex(g, gi, gdeg0, gdeg1, gout_size, &gl, &gli);
	  }
    else
	  { initvecdesc(fnumsubvecs, fsubveclth);
    	fl = NULL;
		red2lex(f, fdeg0, fdeg1, foutsize, &fl);
    	initvecdesc(gnumsubvecs, gsubveclth);
    	gl = NULL;
		red2lex(g, gdeg0, gdeg1, gout_size, &gl);
	  }

    /**************************************************************************/
    /* Compute j = Jacobian of f w.r.t. vars in input(f) intersect output(g). */
    /**************************************************************************/
    jdeg0 = fdeg0 ? fdeg0-1 : 0;
    jdeg1 = fdeg1 ? fdeg1-1 : 0;

    if (complex)
      { jacobiancomplex(fl, fli, fdeg0, fdeg1, fin_size, foutsize,
	                          fin_int_gout_among_fin, jdeg0, jdeg1, &j, &ji);
        mxFree(fl);
        mxFree(fli);
	  }
    else
	  { jacobian(fl, fdeg0, fdeg1, fin_size, foutsize,
	                            fin_int_gout_among_fin, jdeg0, jdeg1, &j);
        mxFree(fl);
	  }

    /**************************************************************************/
    /* Extend inputs of j and gl to include  vars in input(f) union input(g). */
    /**************************************************************************/
	if (complex)
	  { jext = gle = NULL;
        extendinputscomplex(j, ji, fin_union_gin_size, jdeg0, jdeg1,
		             foutsize*fin_int_gout_size, fin_among_fin_union_gin,
					 &jext, &jexti, &extending);
        if (extending)  { mxFree(j); mxFree(ji); }

	    extendinputscomplex(gl, gli, fin_union_gin_size, gdeg0, gdeg1,
                          gout_size, gin_among_fin_union_gin,
						  &gle, &glei, &extending);
        if (extending)  { mxFree(gl); mxFree(gli); }

        squeezeoutputcomplex(gle, glei, fin_union_gin_size, gdeg0, gdeg1,
                                            gout_size, fin_int_gout_among_gout);
	  }
    else
	  { jext = gle = NULL;
        extendinputs(j, fin_union_gin_size, jdeg0, jdeg1,
		             foutsize*fin_int_gout_size, fin_among_fin_union_gin,
					 &jext, &extending);
        if (extending)  mxFree(j);

	    extendinputs(gl, fin_union_gin_size, gdeg0, gdeg1,
                          gout_size, gin_among_fin_union_gin, &gle, &extending);
        if (extending)  mxFree(gl);

        squeezeoutput(gle, fin_union_gin_size, gdeg0, gdeg1,
                                            gout_size, fin_int_gout_among_gout);
	  }

    /********************************************************************/
    /* Allocate hl (=h but with input vars in input(f) union input(g)). */
    /********************************************************************/
    hl = mxCalloc(foutsize*crdsum(fin_union_gin_size, hdeg0, hdeg1),
                                                                sizeof(double));

	if (complex)
	  { hli = mxCalloc(foutsize*crdsum(fin_union_gin_size, hdeg0, hdeg1),
                                                                sizeof(double));
	  }

	if (complex)
      { for (r=0; r<foutsize; r++)    /* Once for each output var */
          for (v=0; v<fin_int_gout_size; v++)
                            /* Once per var in input(f) intersect output(g) */
            addscalarprodcomplex(gle+v, glei+v, gdeg0, gdeg1, gout_size,
              jext+v+r*fin_int_gout_size,
              jexti+v+r*fin_int_gout_size, jdeg0, jdeg1,
              foutsize*fin_int_gout_size, fin_union_gin_size,
              hl+r, hli+r, hdeg0, hdeg1, foutsize, FALSE);
	    mxFree(jext);
	    mxFree(jexti);
		mxFree(gle);
		mxFree(glei);
	  }

    else
      { for (r=0; r<foutsize; r++)    /* Once for each output var */
          for (v=0; v<fin_int_gout_size; v++)
                            /* Once per var in input(f) intersect output(g) */
            addscalarprod(gle+v, gdeg0, gdeg1, gout_size,
              jext+v+r*fin_int_gout_size, jdeg0, jdeg1,
              foutsize*fin_int_gout_size, fin_union_gin_size,
              hl+r, hdeg0, hdeg1, foutsize, FALSE);
	    mxFree(jext);
		mxFree(gle);
	  }

    /**************************************************/
    /* Extend inputs of h to include all variables in */
    /* input(f) union input(g) union output(g).       */
    /**************************************************/
	if (complex)
	  { hle = NULL;
	    extendinputscomplex(hl, hli, all_size, hdeg0, hdeg1, foutsize,
                              fin_union_gin_among_all, &hle, &hlei, &extending);
        if (extending)  { mxFree(hl); mxFree(hli); }
	  }
    else
	  { hle = NULL;
	    extendinputs(hl, all_size, hdeg0, hdeg1, foutsize,
                                     fin_union_gin_among_all, &hle, &extending);
        if (extending)  mxFree(hl);
	  }

    /***************************************/
    /* Allocate (and clear) output matrix. */
    /***************************************/
    plhs[0] = mxCreateDoubleMatrix(foutsize, crdsum(all_size,hdeg0,hdeg1),
                                                  complex ? mxCOMPLEX : mxREAL);
    h = mxGetPr(plhs[0]);

	if (complex)
      hi = mxGetPi(plhs[0]);

    /******************************/
    /* Convert h to reduced form. */
    /******************************/
	if (complex)
	  { initvecdesc(hnumsubvecs, hsubveclth);
        lex2redcomplex(hle, hlei, hdeg0, hdeg1, foutsize, &h, &hi);
	  }
	else
	  { initvecdesc(hnumsubvecs, hsubveclth);
    	lex2red(hle, hdeg0, hdeg1, foutsize, &h);
	  }
}
