/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file contains routines for computing Jacobians of polynomials.        */
/*                                                                            */
/* (written by Dean Hickerson, 9/1/94 - 9/30/94)                              */
/******************************************************************************/

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
void addjacobianh(double f[], int fdeg, int finsize, int foutsize,
                boolean varneeded[], int numneeded, int position[], double h[]);
void jacobian(double f[], int fdeg0, int fdeg1, int finsize, int foutsize,
                    boolean varneeded[], int jdeg0, int jdeg1, double *j[]);
void addjacobianhcomplex(double f[], double fi[], int fdeg, int finsize,
        int foutsize, boolean varneeded[], int numneeded, int position[],
		double h[], double hi[]);
void jacobiancomplex(double f[], double fi[], int fdeg0, int fdeg1,
		int finsize, int foutsize, boolean varneeded[], int jdeg0, int jdeg1,
		double *j[], double *ji[]);
#endif

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
extern boolean initmonom(struct monom *m, int deg, int insz);
extern boolean nextmonom(struct monom *m, int *changeptr);
extern int lexindex(struct monom *m);
#else
extern int crd();
extern int crdsum();
extern boolean initmonom();
extern boolean nextmonom();
extern int lexindex();
#endif

/******************************************************************************/

#ifdef __STDC__
void addjacobianh(double f[], int fdeg, int finsize, int foutsize,
                boolean varneeded[], int numneeded, int position[], double h[])
#else
void addjacobianh(f, fdeg, finsize, foutsize, varneeded, numneeded, position, h)
double f[], h[];
int fdeg, finsize, foutsize, numneeded;
int position[];
boolean varneeded[];
#endif
/*  f[0:foutsize*crd(finsize,fdeg)-1]  represents a foutsize-dimensional
    vector-valued polynomial of degree fdeg in finsize variables.
    varneeded[0:finsize-1] is a boolean array telling which variables the
    Jacobian should be computed with respect to.  numneeded is the number of
    such variables.  position[0:finsize-1] tells, for each needed variable, its
    position in the list of such variables.  This routine adds the Jacobian of f
    (with degree fdeg-1) to  h[0:outsize*numneeded*crd(finsize,fdeg-1)-1].
*/
{   int hidx, houtsize, exc, i, j;
    double *fptr;
    struct monom m, hm;

    if (fdeg==0)  return;

    houtsize = foutsize*numneeded;
    if (!initmonom(&m, fdeg, finsize))  return;
    if (!initmonom(&hm, fdeg-1, finsize))  return;
    fptr = f;

    do                  /* Once for each f monomial */
      { for (i=fdeg-2; i>=0; i--)               /* Init hm=h */
          hm.factor[i] = m.factor[i];
        hidx = lexindex(&hm);

        for (exc=fdeg-1; exc>=0; exc--)
          { /*******/
            /* Add */
            /*******/
            if (varneeded[m.factor[exc]])
              for (j=0; j<foutsize; j++)
                h[hidx*houtsize+j*numneeded+position[m.factor[exc]]] += fptr[j];

            if (exc)
              { /***********/
                /* Next hm */
                /***********/
                hm.factor[exc-1] = m.factor[exc];

                /**************************************************************/
                /* hidx = crd(finsize,fdeg-1) - 1 -                           */
                /*     SUM(crd(finsize-1-hm.factor[i], fdeg-1-i),             */
                /*                                             {i,0,fdeg-2}). */
                /* We're changing the i=exc-1 term.                           */
                /**************************************************************/
                hidx += crd(finsize-1-m.factor[exc-1], fdeg-exc)
                                    - crd(finsize-1-m.factor[exc], fdeg-exc);
              }
          }

        fptr += foutsize;
      }
    while (nextmonom(&m, NULL));
}

/******************************************************************************/

#ifdef __STDC__
void jacobian(double f[], int fdeg0, int fdeg1, int finsize, int foutsize,
               boolean varneeded[], int jdeg0, int jdeg1, double *j[])
#else
void jacobian(f, fdeg0, fdeg1, finsize, foutsize, varneeded, jdeg0, jdeg1, j)
double f[], *j[];
int fdeg0, fdeg1, finsize, foutsize, jdeg0, jdeg1;
boolean varneeded[];
#endif
/*  f[0:outsize*crdsum(insize,fdeg0,fdeg1)-1]  represents a foutsize-dimensional
    vector-valued polynomial of degrees fdeg0 thru fdeg1 in insize variables.
    varneeded[0:insize-1] is a boolean array telling which variables the
    Jacobian should be computed with respect to.  If there are numneeded such
    variables, then this routine computes that part of the Jacobian with degrees
    jdeg0 thru jdeg1 and stores it in
    j[0:outsize*numneeded*crdsum(insize,jdeg0,jdeg1)-1], which it allocates.
*/
{   double *fdegptr, *jdegptr;
    int i, fdeg, jdeg, numneeded, joutsize;
    int position[MAXINSIZE];

    for (i=numneeded=0; i<finsize; i++)
      if (varneeded[i])  position[i] = numneeded++;

    joutsize = foutsize*numneeded;

    *j = mxCalloc(joutsize * crdsum(finsize, jdeg0, jdeg1), sizeof(double));

    for (fdeg=max(fdeg0,1+jdeg0), jdeg=fdeg-1,
        fdegptr=f+foutsize*crdsum(finsize,fdeg0,fdeg-1),
        jdegptr=*j+joutsize*crdsum(finsize,jdeg0,jdeg-1);
                fdeg<=fdeg1 && jdeg<=jdeg1;
                        fdegptr += foutsize*crd(finsize,fdeg), fdeg++,
                        jdegptr += joutsize*crd(finsize,jdeg), jdeg++)
      addjacobianh(fdegptr, fdeg, finsize, foutsize,
                                    varneeded, numneeded, position, jdegptr);
}

/******************************************************************************/

#ifdef __STDC__
void addjacobianhcomplex(double f[], double fi[], int fdeg, int finsize,
        int foutsize, boolean varneeded[], int numneeded, int position[],
		double h[], double hi[])
#else
void addjacobianhcomplex(f, fi, fdeg, finsize, foutsize,
        varneeded, numneeded, position, h, hi)
double f[], fi[], h[], hi[];
int fdeg, finsize, foutsize, numneeded;
int position[];
boolean varneeded[];
#endif
/*  f[0:foutsize*crd(finsize,fdeg)-1]  represents a foutsize-dimensional
    vector-valued polynomial of degree fdeg in finsize variables.
    varneeded[0:finsize-1] is a boolean array telling which variables the
    Jacobian should be computed with respect to.  numneeded is the number of
    such variables.  position[0:finsize-1] tells, for each needed variable, its
    position in the list of such variables.  This routine adds the Jacobian of f
    (with degree fdeg-1) to  h[0:outsize*numneeded*crd(finsize,fdeg-1)-1].
*/
{   int hidx, houtsize, exc, i, j;
    double *fptr, *fiptr;
    struct monom m, hm;

    if (fdeg==0)  return;

    houtsize = foutsize*numneeded;
    if (!initmonom(&m, fdeg, finsize))  return;
    if (!initmonom(&hm, fdeg-1, finsize))  return;
    fptr = f;
    fiptr = fi;

    do                  /* Once for each f monomial */
      { for (i=fdeg-2; i>=0; i--)               /* Init hm=h */
          hm.factor[i] = m.factor[i];
        hidx = lexindex(&hm);

        for (exc=fdeg-1; exc>=0; exc--)
          { /*******/
            /* Add */
            /*******/
            if (varneeded[m.factor[exc]])
              for (j=0; j<foutsize; j++)
			    { h[hidx*houtsize+j*numneeded+position[m.factor[exc]]] +=
																	fptr[j];
			      hi[hidx*houtsize+j*numneeded+position[m.factor[exc]]] +=
																	fiptr[j];
				}

            if (exc)
              { /***********/
                /* Next hm */
                /***********/
                hm.factor[exc-1] = m.factor[exc];

                /**************************************************************/
                /* hidx = crd(finsize,fdeg-1) - 1 -                           */
                /*     SUM(crd(finsize-1-hm.factor[i], fdeg-1-i),             */
                /*                                             {i,0,fdeg-2}). */
                /* We're changing the i=exc-1 term.                           */
                /**************************************************************/
                hidx += crd(finsize-1-m.factor[exc-1], fdeg-exc)
                                    - crd(finsize-1-m.factor[exc], fdeg-exc);
              }
          }

        fptr += foutsize;
        fiptr += foutsize;
      }
    while (nextmonom(&m, NULL));
}

/******************************************************************************/

#ifdef __STDC__
void jacobiancomplex(double f[], double fi[], int fdeg0, int fdeg1,
	int finsize, int foutsize, boolean varneeded[], int jdeg0, int jdeg1,
	double *j[], double *ji[])
#else
void jacobiancomplex(f, fi, fdeg0, fdeg1, finsize, foutsize, varneeded,
	jdeg0, jdeg1, j, ji)
double f[], fi[], *j[], *ji[];
int fdeg0, fdeg1, finsize, foutsize, jdeg0, jdeg1;
boolean varneeded[];
#endif
/*  f[0:outsize*crdsum(insize,fdeg0,fdeg1)-1]  represents a foutsize-dimensional
    vector-valued polynomial of degrees fdeg0 thru fdeg1 in insize variables.
    varneeded[0:insize-1] is a boolean array telling which variables the
    Jacobian should be computed with respect to.  If there are numneeded such
    variables, then this routine computes that part of the Jacobian with degrees
    jdeg0 thru jdeg1 and stores it in
    j[0:outsize*numneeded*crdsum(insize,jdeg0,jdeg1)-1], which it allocates.
*/
{   double *fdegptr, *fidegptr, *jdegptr, *jidegptr;
    int i, fdeg, jdeg, numneeded, joutsize;
    int position[MAXINSIZE];

    for (i=numneeded=0; i<finsize; i++)
      if (varneeded[i])  position[i] = numneeded++;

    joutsize = foutsize*numneeded;

    *j = mxCalloc(joutsize * crdsum(finsize, jdeg0, jdeg1), sizeof(double));
    *ji = mxCalloc(joutsize * crdsum(finsize, jdeg0, jdeg1), sizeof(double));

    for (fdeg=max(fdeg0,1+jdeg0), jdeg=fdeg-1,
        fdegptr=f+foutsize*crdsum(finsize,fdeg0,fdeg-1),
        fidegptr=fi+foutsize*crdsum(finsize,fdeg0,fdeg-1),
        jdegptr=*j+joutsize*crdsum(finsize,jdeg0,jdeg-1),
        jidegptr=*ji+joutsize*crdsum(finsize,jdeg0,jdeg-1);
                fdeg<=fdeg1 && jdeg<=jdeg1;
                        fdegptr += foutsize*crd(finsize,fdeg),
                        fidegptr += foutsize*crd(finsize,fdeg), fdeg++,
                        jdegptr += joutsize*crd(finsize,jdeg),
                        jidegptr += joutsize*crd(finsize,jdeg), jdeg++)
      addjacobianhcomplex(fdegptr, fidegptr, fdeg, finsize, foutsize,
                       varneeded, numneeded, position, jdegptr, jidegptr);
}
