/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file implements the function [h,nh]=jcbn(f,nf,df,nj,d).               */
/*                                                                            */
/* (written by Dean Hickerson, 8/30/94)                                       */
/* (modified 8/29/2001 to work with complex numbers)                          */
/******************************************************************************/

#define shortname   "jcbn"
#define fullname    "jcbn(f,nf,df,nj,d)"

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crdsum(int m, int n0, int n1);
extern boolean samesize(int fr, int fc, int wantedfr, int wantedfc);
extern void initvecdesc(int numsubvecs, int subveclth[]);
extern void red2lex(double f[], int deg0,int deg1,int outsize, double *fl[]);
extern void lex2red(double fl[], int deg0,int deg1,int outsize, double *f[]);
extern void red2lexcomplex(double f[], double fi[], int deg0, int deg1,
		int outsize, double *fl[], double *fli[]);
extern void lex2redcomplex(double fl[], double fli[], int deg0, int deg1,
		int outsize, double *f[], double *fi[]);
extern void extendinputs(double f[], int insize, int deg0, int deg1,
        int outsize, boolean varpresent[], double *fext[], boolean *extending);
extern void extendinputscomplex(double f[], double fi[], int insize,
                         int deg0, int deg1, int outsize, boolean varpresent[],
                         double *fext[], double *fexti[], boolean *extending);
extern void jacobian(double f[], int fdeg0, int fdeg1, int finsize,
        int foutsize, boolean varneeded[], int jdeg0, int jdeg1, double *j[]);
extern void jacobiancomplex(double f[], double fi[], int fdeg0, int fdeg1,
		int finsize, int foutsize, boolean varneeded[], int jdeg0, int jdeg1,
		double *j[], double *ji[]);
#else
extern int crdsum();
extern boolean samesize();
extern void initvecdesc();
extern void red2lex();
extern void lex2red();
extern void red2lexcomplex();
extern void lex2redcomplex();
extern void extendinputs();
extern void extendinputscomplex();
extern void jacobian();
extern void jacobiancomplex();
#endif

/******************************************************************************/

#ifdef __STDC__
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
#else
mexFunction(nlhs, plhs, nrhs, prhs)
int nlhs, nrhs;
mxArray *plhs[], *prhs[];
#endif
{   double *f, *fi, *nf, *df, *nj, *d, *h, *hi, *nh, *fl, *fli, *hl, *hli,
           *fle, *flei;
    int fr, fc, nfr, nfc, dfr, dfc, njr, njc, dr, dc, nfrows, nhr,
        fdeg0, fdeg1, hdeg0, hdeg1, i, i0, il, ie, rc, r, c, colsum, rj,
        fnumsubvecs, jnumsubvecs, hnumsubvecs,
        finsize, foutsize, jinsize, hinsize;
    int fsubveclth[MAXSUBVECS], jsubveclth[MAXSUBVECS], hsubveclth[MAXSUBVECS];
    boolean fvarpresent[MAXINSIZE], jvarneeded[MAXINSIZE];
    boolean extending, complex;
    char outbuff[200];
    int inputargcounter;

    /***************************************************/
    /* Check that there are 5 inputs and <= 2 outputs. */
    /***************************************************/
    if (nrhs != 5)  err("ERROR: jcbn requires 5 inputs.")
    if (nlhs > 2)   err("ERROR: jcbn has only 2 outputs.")
    checktypes
    complex = mxIsComplex(prhs[0]);

    /****************************************/
    /* Get sizes of and pointers to inputs. */
    /****************************************/
    f  = mxGetPr(prhs[0]);  fr  = mxGetM(prhs[0]);  fc  = mxGetN(prhs[0]);
    nf = mxGetPr(prhs[1]);  nfr = mxGetM(prhs[1]);  nfc = mxGetN(prhs[1]);
    df = mxGetPr(prhs[2]);  dfr = mxGetM(prhs[2]);  dfc = mxGetN(prhs[2]);
    nj = mxGetPr(prhs[3]);  njr = mxGetM(prhs[3]);  njc = mxGetN(prhs[3]);
    d  = mxGetPr(prhs[4]);  dr  = mxGetM(prhs[4]);  dc  = mxGetN(prhs[4]);

    if (complex)
      fi = mxGetPi(prhs[0]);

    if (nfc < 2)  err(
      "ERROR: In jcbn(f,nf,df,nj,d), nf must have at least 2 columns.")

    if (njc != 1)  err(
      "ERROR: In jcbn(f,nf,df,nj,d), nj must be a column vector.")

    if (nfr != njr)  err(
      "ERROR: In jcbn(f,nf,df,nj,d), nf and nj must have same number of rows.")

    checkrange(2, "df")
    checkrange(4, "d")

    /********************************************/
    /* Get the min and max degrees for f and h. */
    /********************************************/
    fdeg0 = df[0];  fdeg1 = df[dfc-1];
    hdeg0 = d[0];   hdeg1 = d[dc-1];

    /************************************************/
    /* Scan nf and nj to determine the input sizes. */
    /************************************************/
    finsize=jinsize=hinsize = fnumsubvecs=jnumsubvecs=hnumsubvecs = 0;

    /*******************************************************/
    /* In the following loop:                              */
    /* i0 is an index into the first columns of nf and nj, */
    /* il is an index into the last column of nf.          */
    /*******************************************************/
    for (i0=0, il=nfr*(nfc-1);  i0<nfr;  i0++, il++)
      { /*********************************************/
        /* Record info about input sizes of f and j. */
        /*********************************************/
        if (nf[il])
          { finsize += (fsubveclth[fnumsubvecs++] = nf[il]);
            hsubveclth[hnumsubvecs++] = nf[il];
            if (nj[i0])         /* nf[il] != 0,  nj[i0] != 0 */
              { jinsize += (jsubveclth[jnumsubvecs++] = nj[i0]);
                if (nf[il] == nj[i0])
                  for (i=nf[il]; i; i--)
                    { fvarpresent[hinsize]  = TRUE;
                      jvarneeded[hinsize++] = TRUE;
                    }
#ifdef THINK_C
                else  err2(
                  "ERROR: In jcbn(f,nf,df,nj,d), corresponding elements of nj",
                  "\rand last column of nf must be equal or zero.")
#else
                else  err2(
                  "ERROR: In jcbn(f,nf,df,nj,d), corresponding elements of nj",
                  "\nand last column of nf must be equal or zero.")
#endif
              }
            else                /* nf[il] != 0,  nj[i0] = 0 */
              { for (i=nf[il]; i; i--)
                  { fvarpresent[hinsize]  = TRUE;
                    jvarneeded[hinsize++] = FALSE;
                  }
              }
          }
        else
          { if (nj[i0])         /* nf[il] = 0,  nj[i0] != 0 */
              { jinsize += (jsubveclth[jnumsubvecs++] = nj[i0]);
                hsubveclth[hnumsubvecs++] = nj[i0];
                for (i=nj[i0]; i; i--)
                  { fvarpresent[hinsize]  = FALSE;
                    jvarneeded[hinsize++] = TRUE;
                  }
              }
          }
      }

    if (fdeg0 < 0 || fdeg1 < fdeg0)  err(
      "ERROR: In jcbn(f,nf,[df0 df1],nj,d), must have 0 <= df0 <= df1.")

    if (hdeg0 < 0 || hdeg1 < hdeg0)  err(
      "ERROR: In jcbn(f,nf,df,nj,[d0 d1]), must have 0 <= d0 <= d1.")

    /*******************************************************************/
    /* Compute the output size from nf; i.e. the product of all column */
    /* sums except the last.  This must equal the number of rows of f. */
    /*******************************************************************/
    for (rc=c=0, foutsize=1; c<nfc-1; c++)
      { for (r=colsum=0; r<nfr; r++, rc++)  colsum += nf[rc];
        foutsize *= colsum;
      }

    /**********************************************/
    /* Check that f has the size specified by nf. */
    /**********************************************/
    if (!samesize(fr, fc, foutsize, crdsum(finsize,fdeg0,fdeg1)))
      { sprintf(outbuff,
          "ERROR: In %s, f should be %ld by %ld, not %ld by %ld.",
          fullname, foutsize, crdsum(finsize,fdeg0,fdeg1), fr, fc);
        err(outbuff)
      }

    /******************************************************************/
    /* If 2 output args, compute nh.  nh has one more column than nf. */
    /******************************************************************/
    if (nlhs == 2)
      { /**************************************************/
        /* Find last row of nf that has a nonzero element */
        /* before the last column.                        */
        /**************************************************/
        for (r=nfr-1; r>=0; r--)
          { for (c=nfc-2; c>=0; c--)
              if (nf[r+c*nfr])  break;
            if (c>=0)  break;
          }

        nfrows = r+1;
        nhr = max(nfrows+jnumsubvecs, nfr);
        plhs[1] = mxCreateDoubleMatrix(nhr,nfc+1,mxREAL);
        nh = mxGetPr(plhs[1]);
        for (r=0; r<nfrows; r++)
          for (c=0; c<nfc-1; c++)
            nh[r+c*nhr] = nf[r+c*nfr];
        for (rj=0, r=nfrows; rj<nfr; rj++)
          if (nj[rj])  { nh[r+(nfc-1)*nhr] = nj[rj]; r++; }
        for (r=0; r<nfr; r++)
          nh[r+nfc*nhr] = max(nf[r+(nfc-1)*nfr], nj[r]);
      }

    /********************************************/
    /* Convert f to lexicographic reduced form. */
    /********************************************/
	if (complex)
	  { initvecdesc(fnumsubvecs, fsubveclth);
        fl = NULL;
		red2lexcomplex(f, fi, fdeg0, fdeg1, foutsize, &fl, &fli);
	  }
    else
	  { initvecdesc(fnumsubvecs, fsubveclth);
    	fl = NULL;
		red2lex(f, fdeg0, fdeg1, foutsize, &fl);
	  }

    /*************************************/
    /* If necessary, extend inputs of f. */
    /*************************************/
	if (complex)
	  { fle = NULL;
	    extendinputscomplex(fl,fli,hinsize,fdeg0,fdeg1,foutsize,fvarpresent,
		                                                 &fle,&flei,&extending);
        if (extending)  { mxFree(fl); mxFree(fli); }
	  }
    else
	  { fle = NULL;
	    extendinputs(fl,hinsize,fdeg0,fdeg1,foutsize,fvarpresent,&fle,
                                                                    &extending);
        if (extending)  mxFree(fl);
	  }

    /*********************/
    /* Compute Jacobian. */
    /*********************/
	if (complex)
	  { jacobiancomplex(fle, flei, fdeg0, fdeg1, hinsize, foutsize, jvarneeded,
                                                      hdeg0, hdeg1, &hl, &hli);
        mxFree(fle);
        mxFree(flei);
	  }
    else
	  { jacobian(fle, fdeg0, fdeg1, hinsize, foutsize, jvarneeded,
                                                           hdeg0, hdeg1, &hl);
        mxFree(fle);
	  }

    /***************************************/
    /* Allocate (and clear) output matrix. */
    /***************************************/
	if (complex)
	  { plhs[0] = mxCreateDoubleMatrix(foutsize*jinsize,
                                        crdsum(hinsize,hdeg0,hdeg1), mxCOMPLEX);
        h = mxGetPr(plhs[0]);
		hi = mxGetPi(plhs[0]);
	  }
    else
	  { plhs[0] = mxCreateDoubleMatrix(foutsize*jinsize,
                                        crdsum(hinsize,hdeg0,hdeg1), mxREAL);
        h = mxGetPr(plhs[0]);
	  }

    /*************************************/
    /* Convert Jacobian to reduced form. */
    /*************************************/
	if (complex)
	  { initvecdesc(hnumsubvecs, hsubveclth);
        lex2redcomplex(hl, hli, hdeg0, hdeg1, foutsize*jinsize, &h, &hi);
	  }
    else
	  { initvecdesc(hnumsubvecs, hsubveclth);
    	lex2red(hl, hdeg0, hdeg1, foutsize*jinsize, &h);
	  }
}
