/* Copyright 2001 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file implements the function z=mon(x,n,d)                             */
/*                                                                            */
/* (written by Dean Hickerson, 10/19/2001)                                    */
/******************************************************************************/

#define shortname   "mon"
#define fullname    "mon(x,n,d)"

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
extern void lex2red(double fl[], int deg0,int deg1,int outsize, double *f[]);
extern void lex2redcomplex(double fl[], double fli[], int deg0, int deg1,
        int outsize, double *f[], double *fi[]);
extern void initvecdesc(int numsubvecs, int subveclth[]);

#else
extern int crd();
extern int crdsum();
extern void lex2red();
extern void lex2redcomplex();
extern void initvecdesc();
#endif

/******************************************************************************/

#ifdef __STDC__
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
#else
mexFunction(nlhs, plhs, nrhs, prhs)
int nlhs, nrhs;
mxArray *plhs[], *prhs[];
#endif
{   double *x, *xi, *n, *d, *z, *zi, *zl, *zli;
    double realpart;
    int xr, xc, nr, nc, dr, dc, xsize, i, deg0, deg1;
    int oldstart, newstart, oldlth, newlth, oldend, newend, t, src, dest,
        srcstop, deg;
    int inputargcounter;
    int nint[MAXSUBVECS];
    char outbuff[200];
    boolean complex;

    /***********************************************/
    /* Check that there are 3 inputs and 1 output. */
    /***********************************************/
    if (nrhs != 3)  err("ERROR: mon requires 3 inputs.")
    if (nlhs > 1)  err("ERROR: mon has only 1 output.")
    checktypes
    complex = mxIsComplex(prhs[0]);

    /****************************************/
    /* Get sizes of and pointers to inputs. */
    /****************************************/
    x = mxGetPr(prhs[0]);  xr = mxGetM(prhs[0]);  xc = mxGetN(prhs[0]);
    n = mxGetPr(prhs[1]);  nr = mxGetM(prhs[1]);  nc = mxGetN(prhs[1]);
    d = mxGetPr(prhs[2]);  dr = mxGetM(prhs[2]);  dc = mxGetN(prhs[2]);
    if (complex)
      xi = mxGetPi(prhs[0]);

    if (xc != 1)  err(
      "ERROR: In mon(x,n,d), x must be a column vector.")

    if (nc != 1)  err(
      "ERROR: In mon(x,n,d), n must be a column vector.")

    checkrange(2, "d")

    /****************************/
    /* Get min and max degrees. */
    /****************************/
    deg0 = d[0];  deg1 = d[dc-1];

    if (deg0 < 0)
      err("ERROR: In mon(x,n,d), elements of d must be >= 0.")

    if (deg1 < deg0)
      err("ERROR: In mon(x,n,d), must have d(1) <= d(2).")

    /**********************************************/
    /* Check that size of x is consistent with n. */
    /**********************************************/
    for (i=xsize=0; i<nr; i++)
      { xsize += (nint[i] = n[i]);
        if (n[i] < 0)
          err("ERROR: In mon(x,n,d), elements of n must be >= 0.")
      }

    if (xsize != xr)
      err("ERROR: In mon(x,n,d), size of x doesn't match n.")

    /***********************************************/
    /* Allocate space for lexicographic form of z. */
    /***********************************************/
    zl = mxCalloc(crdsum(xr,deg0,deg1), sizeof(double));
    if (complex)
      zli = mxCalloc(crdsum(xr,deg0,deg1), sizeof(double));

    /************************/
    /* Set up degree 0 part */
    /************************/
    zl[0] = 1;
    if (complex)  zli[0] = 0;
    oldstart = oldend = 0;
    oldlth = 1;

    /**********************************************************/
    /* Loop to compute degree deg part from degree deg-1 part */
    /**********************************************************/
    if (complex)
      for (deg=1; deg<=deg1; deg++)
        { newstart = deg>deg0 ? oldstart+oldlth : 0;
          newlth = crd(xr,deg);
          newend = newstart + newlth - 1;
          for (t=xr-1, dest=newend; t>=0; t--)
            for (src=oldend, srcstop=oldend-crd(xr-t,deg-1);  src>srcstop;
                                                            dest--, src--)
              { realpart = zl[src] * x[t] - zli[src] * xi[t];
                zli[dest] = zl[src] * xi[t] + zli[src] * x[t];
                zl[dest] = realpart;
              }
          oldstart = newstart;
          oldend = newend;
          oldlth = newlth;
        }
    else
      for (deg=1; deg<=deg1; deg++)
        { newstart = deg>deg0 ? oldstart+oldlth : 0;
          newlth = crd(xr,deg);
          newend = newstart + newlth - 1;
          for (t=xr-1, dest=newend; t>=0; t--)
            for (src=oldend, srcstop=oldend-crd(xr-t,deg-1);  src>srcstop;
                                                            dest--, src--)
              zl[dest] = zl[src] * x[t];
          oldstart = newstart;
          oldend = newend;
          oldlth = newlth;
        }

    /***************************************/
    /* Allocate (and clear) output matrix. */
    /***************************************/
    if (complex)
      plhs[0] = mxCreateDoubleMatrix(crdsum(xr,deg0,deg1), 1, mxCOMPLEX);
    else
      plhs[0] = mxCreateDoubleMatrix(crdsum(xr,deg0,deg1), 1, mxREAL);

    z = mxGetPr(plhs[0]);

    if (complex)
      zi = mxGetPi(plhs[0]);

    /******************************************/
    /* Convert monomial list to reduced form. */
    /******************************************/
    if (complex)
      { initvecdesc(nr, nint);
        lex2redcomplex(zl, zli, deg0, deg1, 1, &z, &zi);
      }
    else
      { initvecdesc(nr, nint);
        lex2red(zl, deg0, deg1, 1, &z);
      }
}
