/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/* This file contains routines for squeezing and extending inputs and outputs */
/* of fields.                                                                 */
/*                                                                            */
/* (written by Dean Hickerson, 9/7/94)                                        */
/******************************************************************************/

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
int squeezeinput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[]);
int squeezeoutput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[]);
void extendinputs(double f[], int insize, int deg0, int deg1,
         int outsize, boolean varpresent[], double *fext[], boolean *extending);
#endif

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
extern boolean initmonom(struct monom *m, int deg, int insz);
extern boolean nextmonom(struct monom *m, int *changeptr);
extern boolean initsubmonom(struct monom *m, int deg, int insz,
                                                          boolean varpresent[]);
extern boolean nextsubmonom(struct monom *m, boolean varpresent[]);
extern void printmonom(struct monom *m);
#else
extern int crd();
extern int crdsum();
extern boolean initmonom();
extern boolean nextmonom();
extern boolean initsubmonom();
extern boolean nextsubmonom();
extern void printmonom();
#endif

/******************************************************************************/

#ifdef __STDC__
int squeezeinput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[])
#else
int squeezeinput(f, insize, deg0, deg1, outsize, varneeded)
double f[];
int insize, deg0, deg1, outsize;
boolean varneeded[];
#endif
/*  f[0:outsize*crdsum(insize,deg0,deg1)-1] represents, in lexicographic
    reduced form, an outsize dimensional vector-valued function of degrees
    deg0 through deg1 in insize variables.  varneeded[0:insize-1] is a boolean
    array indicating which input variables are needed.  This routine
    squeezes f into f[0:outsize*crdsum(newinsize,deg0,deg1)-1], but
    treating it now as a function of only those variables x[i] for
    which varneeded[i] is TRUE.  The return value is the new insize,
    i.e. the number of i for which varneeded[i] is TRUE.  E.g. if
    insize=4, deg0=deg1=2, outsize=1, varneeded=[1 0 0 1], and
    f=[1:10] (representing x0^2 + 2 x0 x1 + 3 x0 x2 + ... + 10 x3^2),
    then f will be changed to [1 4 10].
*/
{   double *srcptr, *destptr;
    int deg, change, numgood, val, i, j;
    struct monom m;

    srcptr = destptr = f;

    for (deg=deg0; deg<=deg1; deg++)
      { /**********************************************************************/
        /* Walk through the monomials of degree deg.  numgood tells the       */
        /* length of the initial segment of the monomial which consists of    */
        /* needed variables; i.e. varneeded[m.factor[i]] is TRUE for          */
        /* 0 <= i < numgood,  but FALSE for  i=numgood.  If numgood=deg then  */
        /* this monomial is composed only of needed variables, so copy        */
        /* coefficients of f from srcptr to destptr; otherwise just update    */
        /* srcptr.                                                            */
        /**********************************************************************/
        if (!initmonom(&m, deg, insize))  return 0;
        change = numgood = 0;

        do
          { if (deg && (change <= numgood))
              if (varneeded[m.factor[change]])  numgood = deg;
              else  numgood = change;

            if (numgood == deg)
              for (i=outsize; i; i--)  *destptr++ = *srcptr++;
            else  srcptr += outsize;
          }
        while (nextmonom(&m, &change));
      }

    /************************************************/
    /* Count needed variables, the new insize of f. */
    /************************************************/
    for (i=val=0; i<insize; i++)
      if (varneeded[i])  val++;
    return val;
}

/******************************************************************************/

#ifdef __STDC__
int squeezeoutput(double f[], int insize, int deg0, int deg1,
                                            int outsize, boolean varneeded[])
#else
int squeezeoutput(f, insize, deg0, deg1, outsize, varneeded)
double f[];
int insize, deg0, deg1, outsize;
boolean varneeded[];
#endif
/*  f[0:outsize*crdsum(insize,deg0,deg1)-1] represents an outsize
    dimensional vector-valued function of degrees deg0 through deg1
    in insize variables.  varneeded[0:outsize-1] is a boolean array
    indicating which output variables are needed.  This routine
    squeezes the columns of f upward into the top newoutsize rows,
    effectively throwing away the unneeded outputs.  The return value
    is the new outsize, i.e. the number of i for which varneeded[i]
    is TRUE.  Note that the k'th column of f still starts at index
    k*outsize, not k*newoutsize.  E.g. if insize=2, deg0=deg1=1,
    outsize=4, varneeded=[1 0 0 1], and f=[1 2;3 4;5 6;7 8]
    (representing z0 = x0 + 2 x1, ..., z3 = 7 x0 + 8 x1), then f will
    be changed to [1 2;7 8;? ?;? ?].  (Actually the values represented
    by "?" are unchanged, but they shouldn't be used.)
*/
{   double *srcptr, *destptr;
    int r, c, numcols, val, i;

    numcols = crdsum(insize,deg0,deg1);

    for (c=0, srcptr=f; c<numcols; c++)
      for (r=0, destptr=srcptr; r<outsize; r++)
        if (varneeded[r])  *destptr++ = *srcptr++;
        else  srcptr++;
        
    /*************************************************/
    /* Count needed variables, the new outsize of f. */
    /*************************************************/
    for (i=val=0; i<outsize; i++)
      if (varneeded[i])  val++;
    return val; 
}

/******************************************************************************/

#ifdef __STDC__
void extendinputs(double f[], int insize, int deg0, int deg1,
         int outsize, boolean varpresent[], double *fext[], boolean *extending)
#else
void extendinputs(f,insize,deg0,deg1,outsize,varpresent,fext,extending)
double f[], *fext[];
int insize, deg0, deg1, outsize;
boolean varpresent[], extending;
#endif
/*  f[0:outsize*crdsum(something,deg0,deg1)-1] represents an outsize dimensional
    vector valued function of degrees deg0 through deg1 in at most insize
    variables, namely those x[i] for which varpresent[i] is TRUE.  This routine
    copies f into fext[0:outsize*crdsum(insize,deg0,deg1)-1] (which it
    allocates if fext is initially NULL), but treating it now as a function of
    all insize variables, and returns a pointer to fext.  E.g. if insize=4,
    deg0=deg1=2, outsize=1, varpresent=[1 0 0 1], and f=[1 2 3] (representing
    x0^2 + 2 x0 x3 + 3 x3^2), then fext=[1 0 0 2 0 0 0 0 0 3].
    (Added 9/8/2001:  If all varpresent[0:insize-1] are TRUE, then there's
    no need to do anything; the routine just sets *extending to FALSE and
    returns f.  Otherwise *extending is set to TRUE.)
*/
{   struct monom m;
    double *fptr, *fextptr, *fextdegstart;
    int i, deg;

    for (i=0, *extending=FALSE;  i<insize;  i++)
      if (!varpresent[i])  *extending=TRUE;

    if (!*extending)
      { *fext = f;
        return;
      }

    if (*fext == NULL)
      *fext = mxCalloc(outsize*crdsum(insize,deg0,deg1), sizeof(double));

    fptr = f;
    for (deg=deg0, fextdegstart = *fext;  deg<=deg1;
                fextdegstart += outsize*crd(insize,deg), deg++)
      { if (initsubmonom(&m, deg, insize, varpresent))
          do
            { fextptr = fextdegstart + outsize * m.index;
              for (i=outsize; i; i--)  *fextptr++ = *fptr++;
            }
          while (nextsubmonom(&m, varpresent));
      }

    *extending = TRUE;
}
