/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/*  This file contains the routines for walking through the lexicographically */
/*  ordered list of monomials of a given degree in a given number of          */
/*  variables.  For example, if there are 3 variables (x0, x1, and x2), then  */
/*  the monomials of degree 3 are ordered and indexed as follows:             */
/*                                                                            */
/*   3     2       2         2                 2     3     2         2     3  */
/*  x     x x     x x     x x     x x x     x x     x     x x     x x     x   */
/*   0     0 1     0 2     0 1     0 1 2     0 2     1     1 2     1 2     2  */
/*                                                                            */
/*   0     1       2       3         4       5       6     7       8       9  */
/*                                                                            */
/*  A monomial is represented by a 'struct monom', which contains fields for  */
/*  the degree, number of variables, and index, and an array (factor) of      */
/*  length degree containing a sorted list of the indices of the variables    */
/*  involved.                                                                 */
/*                                                                            */
/*  To walk through the monomials of degree deg in insz variables, first call */
/*  initmonom(&m, deg, insz), which sets m to represent x0^deg.  Then         */
/*  repeatedly call nextmonom(&m, &change), which tries to change monom to    */
/*  represent the next monomial, and returns in change the least i for which  */
/*  m.factor[i] has been changed.  (Or use nextmonom(&m, NULL) if change      */
/*  isn't needed.)  nextmonom normally returns TRUE, but returns FALSE if m   */
/*  already represents the last monomial (x[insz-1]^deg).                     */
/*                                                                            */
/*  Sometimes we need to walk through only those monomials over a given       */
/*  subset of the variables.  To do so, first set up a boolean array          */
/*  varpresent[0:insz-1]  which tells which variables are to be involved.     */
/*  Then call initsubmonom(&m, deg, insz, varpresent), which sets m to        */
/*  represent xi^deg, where i is the first index for which varpresent[i] is   */
/*  TRUE.  (If there's no such i, then initsubmonom returns FALSE.)  Then     */
/*  repeatedly call nextsubmonom(&m, varpresent), which tries to change       */
/*  m to represent the next monomial in the given variables, returning FALSE  */
/*  when there are no more.                                                   */
/*                                                                            */
/*  Sometimes it's necessary to compute the index of a monomial from scratch. */
/*  To do so, call lexindex(&m).                                              */
/*                                                                            */
/*  The routine printmonom(&m) is for debugging only.                         */
/*                                                                            */
/*  (written by Dean Hickerson, 8/4/94 - 9/26/94)                             */
/******************************************************************************/

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
boolean initmonom(struct monom *m, int deg, int insz);
boolean nextmonom(struct monom *m, int *changeptr);
boolean initsubmonom(struct monom *m, int deg, int insz, boolean varpresent[]);
boolean nextsubmonom(struct monom *m, boolean varpresent[]);
int lexindex(struct monom *m);
void printmonom(struct monom *m);
#endif

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
#else

#endif

/******************************************************************************/

#ifdef __STDC__
boolean initmonom(struct monom *m, int deg, int insz)
#else
boolean initmonom(m, deg, insz)
struct monom *m;
int deg, insz;
#endif
/*  Initialize m to represent the monomial x0^deg.  Return FALSE if
	insz is 0, unless deg is also 0.
*/
{   int i;

	if (insz)
      { m->degree = deg;
    	for (i=0; i<deg; i++)  m->factor[i] = 0;
    	m->insize = insz;
    	m->index = 0;
		return TRUE;
	  }
	else  return deg==0;
}

/******************************************************************************/

#ifdef __STDC__
boolean nextmonom(struct monom *m, int *changeptr)
#else
boolean nextmonom(m, changeptr)
struct monom *m;
int *changeptr;
#endif
/*  This routine is used to walk through the lexicographically reduced list
    of monomials.  If possible, this routine changes m to represent the next
    monomial, and returns TRUE; it returns FALSE iff the current monomial
    is the last one.
    If changeptr is not NULL, then *changeptr is set to the index of the first
    factor being changed.
*/
{   int i, j, fac;

    for (i=m->degree-1; i>=0; i--)
      if (m->factor[i] != m->insize - 1)
        { fac = m->factor[i] + 1;
          for (j=i; j<m->degree; j++)  m->factor[j] = fac;
          if (changeptr)  *changeptr = i;
          m->index++;
          return TRUE;
        }

    return FALSE;
}

/******************************************************************************/

#ifdef __STDC__
boolean initsubmonom(struct monom *m, int deg, int insz, boolean varpresent[])
#else
boolean initsubmonom(m, deg, insz, varpresent)
struct monom *m;
int deg, insz;
boolean varpresent[];
#endif
/*  Initializes m to represent  x[r]^deg,  where r is minimal such that
    varpresent[r] is TRUE, among monomials of degree deg over variables
    x[0], ..., x[insz-1].  If there are no variables for which varpresent[r]
    is TRUE, returns FALSE.
*/
{   int i, r;

    for (r=0; r<insz; r++)
      if (varpresent[r])  break;

    if (r==insz)  return FALSE;

    for (i=0; i<deg; i++)  m->factor[i] = r;
    m->degree = deg;
    m->insize = insz;
    m->index = crd(insz, deg) - 1 - crdsum(insz-1-r, 1, deg);
    return TRUE;
}

/******************************************************************************/

#ifdef __STDC__
boolean nextsubmonom(struct monom *m, boolean varpresent[])
#else
boolean nextsubmonom(m, varpresent)
struct monom *m;
boolean varpresent[];
#endif
/*  Changes m to represent the lexicographically next monomial among those in
    the variables x[i] with varpresent[i] equal to TRUE, if possible, else
    returns FALSE.
*/
{   int lastpres, nextpres, i, j;

    for (lastpres=m->insize-1; !varpresent[lastpres]; lastpres--);

    for (i = m->degree - 1; i >= 0 && m->factor[i] == lastpres; i--);

    if (i<0)  return FALSE;     /* All factors equal lastpres */

    for (nextpres = m->factor[i] + 1; !varpresent[nextpres]; nextpres++);

    m->index +=
      crd(m->insize - 1 - m->factor[i], m->degree - i)
                   /* Old factor[i] */
      + crdsum(m->insize - 1 - lastpres, 1, m->degree - i - 1)
                   /* Old factor[i+1] to factor[deg-1] all equal lastpres */
      - crdsum(m->insize - 1 - nextpres, 1, m->degree - i);
                   /* New factor[i] to factor[deg-1] all equal nextpres */

    for (; i<m->degree; i++)  m->factor[i] = nextpres;
    return TRUE;
}

/******************************************************************************/

#ifdef __STDC__
int lexindex(struct monom *m)
#else
int lexindex(m)
struct monom *m;
#endif
/*  Computes index of given monomial in lexicographic reduced form.  This
    ranges from 0 to  crd(insize,degree)-1  and equals
        crd(insize,degree) - 1 - crd(insize-1-factor[0],degree) - ...
                                        - crd(insize-1-factor[degree-1],1)
*/
{   int i, val;

    for (i=0, val=crd(m->insize,m->degree)-1; i<m->degree; i++)
      val -= crd(m->insize - 1 - (m->factor[i]), m->degree-i);
    return val;
}

/******************************************************************************/

#ifdef __STDC__
void printmonom(struct monom *m)
#else
printmonom(m)
struct monom *m;
#endif
/* Print the current monomial, showing each factor's name and index.  For
   debugging purposes only.
*/
{   int i;

    for (i=0; i<m->degree && i<MAXDEGREE; i++)
      printf("x%ld ", m->factor[i]);
#ifdef THINK_C
    printf("\r");
#else
    printf("\n");
#endif
}
