/* Copyright 1996 by A. J. Krener and Dean Hickerson */

/******************************************************************************/
/*  This file contains the routines for converting between two                */
/*  representations of a vector (or matrix or tensor) field, the so-called    */
/*  reduced form used within MATLAB, and the lexicographic form used within   */
/*  these c programs.  In each form, a vector-valued polynomial is            */
/*  represented by a list of (vector-valued) coefficients, one for each       */
/*  monomial in the input variables.  (Two monomials are considered equal if  */
/*  they are permutations of each other; e.g. x0 x1 = x1 x0.)  The only       */
/*  difference between the two representations is the ordering of the         */
/*  monomials.  In both cases, if two monomials have different degrees, then  */
/*  the one with smaller degree comes first.  If they have the same degree,   */
/*  then in the lexicographic ordering, the one that's lexicographically      */
/*  smallest comes first; e.g.  x0 x2^2 x3 x5 x7  precedes  x0 x2 x3^2 x8^2.  */
/*                                                                            */
/*  The reduced ordering is more complicated:  The variables are considered   */
/*  to be components of a number of subvectors of various lengths.  The first */
/*  3 vectors are usually called "x", "u", and "y", and the variables are     */
/*  x0, x1, ..., u0, ..., y0, ...  For example, suppose there are 3           */
/*  subvectors, x, u, and y, with 2, 3, and 2 dimensions respectively.  Then  */
/*  the variables are:                                                        */
/*                                                                            */
/*      name     x0  x1  u0  u1  u2  y0  y1                                   */
/*      index    0   1   2   3   4   5   6                                    */
/*                                                                            */
/*  The 'reduced' ordering of the monomials is defined as follows:  Break     */
/*  up each monomial into groups of factors from the same subvector, call     */
/*  them the x-group, the u-group, etc.  If two monomials have x-groups of    */
/*  different lengths, then the one with the longer x-group comes first.  If  */
/*  the x-groups have the same length but are different, then the one with    */
/*  the lexicographically smaller x-group comes first.  If the x-groups are   */
/*  identical, then examine the u-groups.  If one is longer, it comes first,  */
/*  etc.                                                                      */
/*                                                                            */
/*  A monomial is represented by the struct monom redmonom, which is local to */
/*  to this file.  The form of the representation is the same as in           */
/*  walkmonom.c; e.g. in the above example, the monomial  x0^2 u1 y0  would   */
/*  be represented by:                                                        */
/*                                                                            */
/*      i                      0 1 2 3                                        */
/*      redmonom.factor[i]     0 0 3 5                                        */
/*                                                                            */
/*  To convert from reduced form to lexicographic form, first call            */
/*  initvecdesc(numsv, subveclth), where numsv is the number of input         */
/*  subvectors and subveclth is the address of an array containing, for       */
/*  0 <= i < numsv,  the length of the i'th subvector.  Then call             */
/*  red2lex(f, deg0, deg1, outsz, &fl), where f is an outsz-dimensional       */
/*  vector field of degrees deg0 to deg1 in reduced form and fl is either a   */
/*  pointer to a preallocated space in which to store the same field in       */
/*  lexicographic form, or is NULL, in which case space will be allocated.    */
/*  In either case, red2lex returns a pointer to the new field.               */
/*                                                                            */
/*  Similarly, to convert from lexicographic form to reduced form, first call */
/*  initvecdesc and then call lex2red(fl, deg0, deg1, outsz, f), where fl is  */
/*  an outsz-dimensional vector field of degrees deg0 to deg1 in              */
/*  lexicographic form and f is either a pointer to a preallocated space in   */
/*  which to store the same field in reduced form, or is NULL, in which case  */
/*  space will be allocated.  In either case, lex2red returns a pointer to    */
/*  the new field.                                                            */
/*                                                                            */
/*  The conversion from reduced form to lexicographic form or vice versa is   */
/*  done by walking through the list of monomials in reduced form, computing  */
/*  the index of each monomial in lexicographic form, and copying the         */
/*  coefficient from the given representation to the one we are constructing. */
/*  The routines initredmonom and nextredmonom, which are local to this file, */
/*  are used to do this.  These routines use the following local variables:   */
/*                                                                            */
/*      redmonom                    Struct representing the current monomial. */
/*                                                                            */
/*      numsv                       Number of input subvectors                */
/*                                                                            */
/*      svlth[0:numsv-1]            svlth[i] is the length of the i'th        */
/*                                  subvector.                                */
/*                                                                            */
/*      insz                        = svlth[0]+...+svlth[numsv-1]             */
/*                                                                            */
/*      subvec[0:insz]             For  i < insz,  subvec[i] is the number    */
/*                                 of the subvector that factor i belongs to. */
/*                                 subvec[insz] is effectively infinite.      */
/*                                                                            */
/*      nextsvstart[0:insz-1],     The indices of the components of the       */
/*      currsvstart[0:insz-1];     subvector that includes i run from         */
/*                                 currsvstart[i] to nextsvstart[i]-1.        */
/*                                                                            */
/*  To begin walking through the monomials, call initvecdesc, as described    */
/*  above, which initializes the tables subvec, currsvstart, and nextsvstart. */
/*  Then call initredmonom(degree), which sets redmonom to represent          */
/*  x0^degree.  Then repeatedly call nextredmonom(&change), which tries to    */
/*  change monom to represent the next monomial, and returns in change the    */
/*  least i for which monom[i] has been changed.  (Or use nextredmonom(NULL)  */
/*  if change isn't needed.)  nextredmonom normally returns TRUE, but returns */
/*  FALSE if monom represents the last monomial.                              */
/*                                                                            */
/*  nextredmonom works as follows:  We start at the end of the list of        */
/*  factors and identify the last group; suppose its factors are components   */
/*  of subvector lastsv, with  0 <= lastsv < numsv.  If possible, we          */
/*  change only this group, to the lexicographically next product of          */
/*  components of lastsv.  If the last group is a power of the last           */
/*  component of lastsv, and  lastsv < numsv-1,  then we shorten the last     */
/*  group by 1, changing it to a power of the first component of lastsv, and  */
/*  append the first component of lastsv+1.  If the last group is a power of  */
/*  the last component of numsv-1, then we identify the penultimate group;    */
/*  suppose its factors are components of subvector pensv.  (If there isn't   */
/*  a penultimate group, then we've reached the end of the list of            */
/*  monomials.)  If possible, we change the penultimate group to the          */
/*  lexicographically next product of components of pensv, and replace the    */
/*  last group by a power of the first component of pensv+1.  If the          */
/*  penultimate group is a power of the last component of pensv, then we      */
/*  shorten it by 1, changing it to a power of the first component of pensv,  */
/*  and replace the last element of the penultimate group and the entire last */
/*  group by a power of the first component of pensv+1.                       */
/*                                                                            */
/*  (written by Dean Hickerson, 8/4/94-9/27/94)                               */
/******************************************************************************/

#include <stdio.h>
#include "mex.h"
#include "mexdefines.h"

/*****************************************/
/* Prototypes of functions defined below */
/*****************************************/
#ifdef __STDC__
void initvecdesc(int numsubvecs, int subveclth[]);
static boolean initredmonom(int degree);
static boolean nextredmonom(int *changeptr);
void *red2lex(double f[], int deg0, int deg1, int outsize, double *fl[]);
void *lex2red(double fl[], int deg0, int deg1, int outsize, double *f[]);
void *red2lexcomplex(double f[], double fi[], int deg0, int deg1, int outsize, double *fl[], double *fli[]);
void *lex2redcomplex(double fl[], double fli[], int deg0, int deg1, int outsize, double *f[], double *fi[]);
#endif

/*************/
/* Externals */
/*************/
#ifdef __STDC__
extern int crd(int m, int n);
extern int crdsum(int m, int n0, int n1);
extern boolean initmonom(struct monom *m, int deg, int insz);
extern int lexindex(struct monom *m);
#else
extern int crd();
extern int crdsum();
extern boolean initmonom();
extern int lexindex();
#endif

/******************************************************************************/

static struct monom redmonom;      /* The monomial being converted */

static int numsv;                  /* Number of input subvectors */

static int insz;                   /* Total length of subvectors */

static int *svlth;                 /* For  0 <= i < numsv,  svlth[i] is the */
                                   /* length of the i'th subvector.         */

static int subvec[MAXINSIZE+1];    /* For  0 <= i < insz,  subvec[i] is the */
                                   /* number of the subvector that factor   */
                                   /* number i belongs to.  subvec[insz] is */
                                   /* effectively infinite. */

static int nextsvstart[MAXINSIZE], /* The indices of the components of the */
           currsvstart[MAXINSIZE]; /* subvector that includes i run from   */
                                   /* currsvstart[i] to nextsvstart[i]-1.  */

/******************************************************************************/

#ifdef __STDC__
void initvecdesc(int numsubvecs, int subveclth[])
#else
void initvecdesc(numsubvecs, subveclth)
int numsubvecs;
int subveclth[];
#endif
/*  Initializes numsv, svlth, and insz to describe the inputs of a vector
    field.  Also initializes subvec[0:insz], nextsvstart[0:insz-1], and
    currsvstart[0:insz-1] so that, given an overall index, we can find what
    subvector it belongs to, and where that subvector begins and ends.  Note:
    svlth is set to point to the given array subveclth, so subveclth should not
    be modified until we're done with it.
*/
{   int i, j, currstart, nextstart;

    numsv = numsubvecs;
    svlth = subveclth;

    for (i=insz=0; i<numsv; i++)
      { currstart = insz;
        nextstart = insz + svlth[i];
        for (j=0; j<svlth[i]; j++)
          { subvec[insz] = i;
            currsvstart[insz] = currstart;
            nextsvstart[insz++] = nextstart;
          }
      }
    subvec[insz] = numsv; /* Effectively infinite */
}

/******************************************************************************/

#ifdef __STDC__
static boolean initredmonom(int deg)
#else
static boolean initredmonom(deg)
int deg;
#endif
/*  Initialize redmonom to represent the monomial x0^deg.  Returns FALSE iff
    insz=0 (in which case x0 doesn't exist), unless deg==0.
*/
{   int i;

    return  initmonom(&redmonom,deg,insz);
}

/******************************************************************************/

#ifdef __STDC__
static boolean nextredmonom(int *changeptr)
#else
static boolean nextredmonom(changeptr)
int *changeptr;
#endif
/*  This routine is used to walk through the reduced list of monomials.  If
    possible, it changes redmonom to represent the next monomial, and returns
    TRUE; it returns FALSE iff the current monomial is the last one.
    If changeptr is not NULL, then *changeptr is set to the index of the first
    factor being changed.
*/
{   int lastsv, pensv, laststart, penstart, i, j, fac, newfac;

    if (redmonom.degree == 0)  return FALSE;

    /*************************************************************/
    /* Increment last group if possible, else find its beginning */
    /*************************************************************/
    fac = redmonom.factor[redmonom.degree-1];
    lastsv = subvec[fac];

    if (subvec[fac+1] == lastsv)
      { redmonom.factor[redmonom.degree-1]++;
        if (changeptr)  *changeptr = redmonom.degree-1;
        return TRUE;    /* Inc'd last element of factor */
      }

    for (j=redmonom.degree-2; j>=0 && subvec[redmonom.factor[j]] == lastsv; j--)
      if (redmonom.factor[j] < fac)
        { newfac = redmonom.factor[j] + 1;
          for (i=j; i<redmonom.degree; i++)  redmonom.factor[i] = newfac;
          if (changeptr)  *changeptr = j;
          return TRUE;  /* Inc'd non-last element of last group */
        }
    laststart = j+1;

    /*********************************************/
    /* Last group runs from factor[laststart] to */
    /* factor[degree-1], can't be inc'd.         */
    /*********************************************/

    if (subvec[redmonom.factor[laststart]] < numsv-1)
      { penstart=laststart;
        laststart=redmonom.degree;
        /* Relabel last group as penultimate, with empty last group. */
      }

    else if (laststart==0)  return FALSE;

    else
      /***********************************************************************/
      /* Last group consists of copies of last component of last subvector.  */
      /* Increment penultimate group if possible, else find its beginning.   */
      /***********************************************************************/

      { fac = redmonom.factor[laststart-1];
        pensv = subvec[fac];

        if (subvec[fac+1] == pensv)
          { redmonom.factor[laststart-1]++;
            newfac = nextsvstart[fac];
            for (i=laststart; i<redmonom.degree; i++)
              redmonom.factor[i] = newfac;
            if (changeptr)  *changeptr = laststart-1;
            return TRUE;    /* Inc'd last element of penultimate group. */
          }

        for (j=laststart-2; j>=0 && subvec[redmonom.factor[j]] == pensv; j--)
          if (redmonom.factor[j] < fac)
            { newfac = redmonom.factor[j] + 1;
              for (i=j; i<laststart; i++)  redmonom.factor[i] = newfac;
              newfac = nextsvstart[fac];
              for (i=laststart; i<redmonom.degree; i++)
                redmonom.factor[i] = newfac;
              if (changeptr)  *changeptr = j;
              return TRUE;  /* Inc'd non-last element of penultimate group. */
            }
        penstart = j+1;
      }

    /***************************************************/
    /* Penultimate group runs from factor[penstart] to */
    /* factor[laststart-1] and can't be inc'd.         */
    /***************************************************/

    newfac = currsvstart[fac];
    for (i=penstart; i<laststart-1; i++)  redmonom.factor[i] = newfac;
    newfac = nextsvstart[fac];
    for (i=laststart-1; i<redmonom.degree; i++)  redmonom.factor[i] = newfac;
    if (changeptr)  *changeptr = penstart;
    return TRUE;    /* Shortened penultimate group. */
}

/******************************************************************************/

#ifdef __STDC__
void *red2lex(double f[], int deg0, int deg1, int outsize, double *fl[])
#else
void *red2lex(f, deg0, deg1, outsize, fl)
double f[], *fl[];
int deg0, deg1, outsize;
#endif
/*  Converts the vector field f from reduced form to lexicographic reduced form.
    Assumes that numsv, svlth[0:numsv-1], insz, ... have been initialized
    by initvecdesc(numsubvecs,subveclth).  If fl is NULL, space is allocated.
*/
{   int i, deg;
    double *fptr, *flptr, *fldegstart;

    if (*fl == NULL)
      *fl = mxCalloc(outsize * crdsum(insz,deg0,deg1), sizeof(double));

    fptr = f;

    for (deg=deg0, fldegstart = *fl;  deg<=deg1;
                fldegstart += outsize*crd(insz,deg), deg++)
      if (initredmonom(deg))
        do
          { flptr = fldegstart + outsize * lexindex(&redmonom);
            for (i=0; i<outsize; i++)  *flptr++ = *fptr++;
          }
        while (nextredmonom(NULL));  /*? Use change to avoid lexindex */
}

/******************************************************************************/

#ifdef __STDC__
void *lex2red(double fl[], int deg0, int deg1, int outsize, double *f[])
#else
void *lex2red(fl, deg0, deg1, outsize, f)
double fl[], *f[];
int deg0, deg1, outsize;
#endif
/*  Converts the vector field fl from lexicographic reduced form to reduced
    form.  Assumes that numsv, svlth[0:numsv-1], and insz have been initialized
    by initvecdesc(numsubvecs,subveclth).  If f is NULL, space is allocated.
*/
{   int i, deg;
    double *fptr, *flptr, *fldegstart;

    if (*f == NULL)
      *f = mxCalloc(outsize * crdsum(insz,deg0,deg1), sizeof(double));

    fptr = *f;

    for (deg=deg0, fldegstart = fl;  deg<=deg1;
                fldegstart += outsize*crd(insz,deg), deg++)
      if (initredmonom(deg))
        do
          { flptr = fldegstart + outsize * lexindex(&redmonom);
            for (i=0; i<outsize; i++)  *fptr++ = *flptr++;
          }
        while (nextredmonom(NULL));
}

/******************************************************************************/

#ifdef __STDC__
void *red2lexcomplex(double f[], double fi[], int deg0, int deg1,
    int outsize, double *fl[], double *fli[])
#else
void *red2lexcomplex(f, fi, deg0, deg1, outsize, fl, fli)
double f[], fi[], *fl[], *fli[];
int deg0, deg1, outsize;
#endif
/*  Converts the vector field f from reduced form to lexicographic reduced form.
    Assumes that numsv, svlth[0:numsv-1], insz, ... have been initialized
    by initvecdesc(numsubvecs,subveclth).  If fl is NULL, space is allocated.
*/
{   int i, deg;
    double *fptr, *fiptr, *flptr, *fliptr, *fldegstart, *flidegstart;

    if (*fl == NULL)
      { *fl = mxCalloc(outsize * crdsum(insz,deg0,deg1), sizeof(double));
        *fli = mxCalloc(outsize * crdsum(insz,deg0,deg1), sizeof(double));
	  }

    fptr = f;
	fiptr = fi;

    for (deg=deg0, fldegstart = *fl, flidegstart = *fli;  deg<=deg1;
                fldegstart += outsize*crd(insz,deg),
				flidegstart += outsize*crd(insz,deg),  deg++)
      if (initredmonom(deg))
        do
          { flptr = fldegstart + outsize * lexindex(&redmonom);
            fliptr = flidegstart + outsize * lexindex(&redmonom);
            for (i=0; i<outsize; i++)
			  { *flptr++ = *fptr++;
			    *fliptr++ = *fiptr++;
			  }
          }
        while (nextredmonom(NULL));  /*? Use change to avoid lexindex */
}

/******************************************************************************/

#ifdef __STDC__
void *lex2redcomplex(double fl[], double fli[], int deg0, int deg1,
    int outsize, double *f[], double *fi[])
#else
void *lex2redcomplex(fl, fli, deg0, deg1, outsize, f, fi)
double fl[], fli[], *f[], *fi[];
int deg0, deg1, outsize;
#endif
/*  Converts the vector field fl from lexicographic reduced form to reduced
    form.  Assumes that numsv, svlth[0:numsv-1], and insz have been initialized
    by initvecdesc(numsubvecs,subveclth).  If f is NULL, space is allocated.
*/
{   int i, deg;
    double *fptr, *fiptr, *flptr, *fliptr, *fldegstart, *flidegstart;

    if (f == NULL)
      { f = mxCalloc(outsize * crdsum(insz,deg0,deg1), sizeof(double));
        fi = mxCalloc(outsize * crdsum(insz,deg0,deg1), sizeof(double));
	  }

    fptr = *f;
    fiptr = *fi;

    for (deg=deg0, fldegstart = fl, flidegstart = fli;  deg<=deg1;
                fldegstart += outsize*crd(insz,deg),
                flidegstart += outsize*crd(insz,deg), deg++)
      if (initredmonom(deg))
        do
          { flptr = fldegstart + outsize * lexindex(&redmonom);
            fliptr = flidegstart + outsize * lexindex(&redmonom);
            for (i=0; i<outsize; i++)
			  { *fptr++ = *flptr++;
			    *fiptr++ = *fliptr++;
			  }
          }
        while (nextredmonom(NULL));
}
