unit GvEditX;

{ Graphics Vision Editor Extensions,
  Copr. 1996 Matthias K"oppe
}

interface

{$ifdef Windows}
uses Dos, Objects, Drivers, Views, GvEdit, GvViews;
{$else !Windows}
uses Dos, Objects, Drivers, Views, Outline, GvEdit, GvViews;
{$endif Windows}

type
  { TEditXWindow, an extended editor window, supporting cmOpenFile
    commands and relative pathnames.
  }
  PEditXWindow = ^TEditXWindow;
  TEditXWindow = object(TEditWindow)
    function GetTitle(MaxSize: Integer): TTitleStr; virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

type
  { Structure passed with cmOpenFile
  }
  POpenFile = ^TOpenFile;
  TOpenFile = record
    Name: string;
    Row: Word;
    Col: Word
  end;

const
  cmOpenFile = 810;

{$ifndef Windows}
const
  msgFile = 0;
  msgMessage = 1;
  msgCommand = 2;
  msgStart = 126;
  msgEnd = 127;
  msgExpanded = 128;

type
  { A command line, associated with its message output
  }
  PCommand = ^TCommand;
  TCommand = record
    Magic: Byte;
    Line: PString;
    Size: Word;
    Buffer: pointer;
    Next: PCommand
  end;

  { TMessageOutline object.
    Note: This is a TV object, which can be inserted into a GV window
    using GvTv. See the TMessageWindow implementation for details.
  }
  PMessageOutline = ^TMessageOutline;
  TMessageOutline = object(TOutlineViewer)
    First: PCommand;
    constructor Init(var Bounds: TRect;
      AHScrollBar, AVScrollBar: views.PScrollBar);
    destructor Done; virtual;
    function AddMessages(CommandLine: string; CommandOut: PathStr): Boolean;
    procedure Adjust(Node: Pointer; Expand: Boolean); virtual;
    function GetChild(Node: Pointer; I: Integer): Pointer; virtual;
    function GetNumChildren(Node: Pointer): Integer; virtual;
    function GetRoot: pointer; virtual;
    function GetText(Node: Pointer): String; virtual;
    function HasChildren(Node: Pointer): Boolean; virtual;
    function IsExpanded(Node: Pointer): Boolean; virtual;
    procedure Selected(I: Integer); virtual;
  private
    function FileOfMessage(Node: pointer): string;
    function NextNode(Node: pointer): pointer;
  end;

  PMessageWindow = ^TMessageWindow;
  TMessageWindow = object(TWindow)
    MsgOutline: PMessageOutline;
    constructor Init(var Bounds: TRect; ATitle: string);
    procedure ChangeBounds(var Bounds: TRect); virtual;
  end;

{$else Windows}

type
  PMessageWindow = ^TMessageWindow;
  TMessageWindow = TWindow;

{$endif !Windows}

implementation

{$ifdef Windows}
uses Misc, GvTexts, GvApp;
{$else !Windows}
uses Memory, Misc, Strings, GvTexts, GvTv, GvApp;
{$endif Windows}

function TEditXWindow.GetTitle(MaxSize: Integer): TTitleStr;
var
  s: PathStr;
begin
  if PEditor(Editor) = Clipboard then GetTitle := GetStr(87) else
    if Editor^.FileName = '' then GetTitle := GetStr(88)
    else begin
      s := Editor^.FileName;
      GetTitle := Relativate(s)
    end;
end;

procedure TEditXWindow.HandleEvent(var Event: TEvent);
begin
  inherited HandleEvent(Event);
  If (Event.What = evBroadcast) and (Event.Command = cmOpenFile) and
     ((POpenFile(Event.InfoPtr)^.Name = '') or
      (GetTitle(255) = POpenFile(Event.InfoPtr)^.Name)) then
  with POpenFile(Event.InfoPtr)^ do
  begin
    Editor^.GotoPos(Row, Col);
    ClearEvent(Event)
  end
end;

{$ifndef Windows}

{ 
}

constructor TMessageOutline.Init(var Bounds: TRect;
  AHScrollBar, AVScrollBar: views.PScrollBar);
begin
  inherited Init(Bounds, AHScrollbar, AVScrollbar);
  Update
end;

destructor TMessageOutline.Done;
var
  P, N: PCommand;
begin
  P := First;
  while P <> nil do
  begin
    N := P^.Next;
    FreeMem(P^.Buffer, P^.Size);
    DisposeStr(P^.Line);
    Dispose(P);
    P := N
  end;
  inherited Done
end;

function TMessageOutline.AddMessages(CommandLine: string; CommandOut: PathStr): Boolean;
var
  S: TDosStream;
  Magic: array[0..9] of Char;
  Command: PCommand;
begin
  S.Init(CommandOut, stOpenRead);
  S.Read(Magic, 10);
  If StrComp(Magic, 'BI#PIP#OK') = 0
  then begin
    New(Command);
    with Command^ do
    begin
      Magic := msgCommand;
      Line := NewStr(CommandLine);
      Size := S.GetSize - 8;
      Buffer := MemAllocSeg(Size);
      S.Read((StrECopy(Buffer, Char(msgStart) + #0) + 1)^, Size - 2);
      Next := First
    end;
    First := Command;
    AddMessages := true;
    Adjust(Command, true);
    Update;
    DrawView
  end
  else
    AddMessages := false;
  S.Done
end;

procedure TMessageOutline.Adjust(Node: Pointer; Expand: Boolean);
begin
  If (Node <> @Self) and (Byte(Node^) <> msgMessage) then
  If Expand
  then Byte(Node^) := Byte(Node^) or msgExpanded
  else Byte(Node^) := Byte(Node^) and not msgExpanded
end;

function TMessageOutline.FileOfMessage(Node: pointer): string;
var
  P: PChar;
begin
  FileOfMessage := '';
  P := Ptr(Seg(Node^), 0);
  while P <> nil do
  begin
    If Byte(P^) and not msgExpanded = msgFile
    then FileOfMessage := StrPas(P + 1);
    If Node = P then Exit;
    P := NextNode(P)
  end;
end;

function TMessageOutline.GetChild(Node: Pointer; I: Integer): Pointer;
var
  num: Integer;
  P: PChar;
  C: PCommand;
begin
  If Node = @Self
  then begin
    C := First;
    For num := 1 to I do
      If C <> nil then C := C^.Next;
    GetChild := C
  end
  else begin
    num := 0;
    P := PCommand(Node)^.Buffer;
    while P <> nil do
    begin
      If Byte(P^) = msgMessage
      then begin
	If num = I
	then begin
	  GetChild := P;
	  Exit
	end
	else Inc(num)
      end;
      P := NextNode(P)
    end;
    GetChild := nil
  end;
end;

function TMessageOutline.GetNumChildren(Node: Pointer): Integer;
var
  num: Integer;
  P: PChar;
  C: PCommand;
begin
  num := 0;
  If Node = @Self
  then begin
    C := First;
    while C <> nil do
    begin
      Inc(num);
      C := C^.Next
    end
  end
  else begin
    P := PCommand(Node)^.Buffer;
    while P <> nil do
    begin
      If Byte(P^) = msgMessage then Inc(num);
      P := NextNode(P)
    end;
  end;
  GetNumChildren := num
end;

function TMessageOutline.GetRoot: pointer;
begin
  GetRoot := @Self
end;

function TMessageOutline.GetText(Node: Pointer): String;
var
  s, txt: string;
  i: Word;
begin
  If Node = @Self
  then GetText := 'Messages'
  else
  case Byte(Node^) and not msgExpanded of
    msgCommand:
      GetText := PCommand(Node)^.Line^;
    msgFile:
      GetText := StrPas(PChar(Node) + 1);
    msgMessage:
      begin
	txt := {Relativate(} FileOfMessage(Node) {)} + '(';
	i := Word(pointer(PChar(Node) + 1)^);
	Str(i, s); txt := txt + s;
	i := Word(pointer(PChar(Node) + 3)^);
	If i <> 0
	then begin
	  Str(i, s);
	  txt := txt + ':' + s;
	end;
	txt := txt + '): ';
	txt := txt + StrPas(PChar(Node) + 5);
	GetText := txt
      end;
    else
      GetText := ''
  end;
end;

function TMessageOutline.HasChildren(Node: Pointer): Boolean;
begin
  HasChildren := (First <> nil) and
    (Node = @Self) or
    ((Byte(Node^) and not msgExpanded = msgCommand) and
     (GetNumChildren(Node) > 0))
end;

function TMessageOutline.IsExpanded(Node: Pointer): Boolean;
begin
  IsExpanded := (Node = @Self) or (Byte(Node^) and msgExpanded <> 0)
end;

function TMessageOutline.NextNode(Node: pointer): pointer;
begin
  case Byte(Node^) and not msgExpanded of
    msgCommand:
      NextNode := PCommand(Node)^.Next;
    msgEnd:
      NextNode := nil;
    msgMessage:
      NextNode := StrEnd(PChar(Node) + 5) + 1;
    else
      NextNode := StrEnd(PChar(Node) + 1) + 1;
  end;
end;

procedure TMessageOutline.Selected(I: Integer);
var
  Node: pointer;
  Open: TOpenFile;
begin
  Node := GetNode(I);
  If Node <> @Self then
  case Byte(Node^) and not msgExpanded of
    msgFile:
      with Open do
      begin
	Name := Relativate(FileOfMessage(Node));
	Row := 0;
	Col := 0;
	Message(Application, evCommand, cmOpenFile, @Open)
      end;
    msgMessage:
      with Open do
      begin
	Name := Relativate(FileOfMessage(Node));
	Row := Word(pointer(PChar(Node) + 1)^);
	Col := Word(pointer(PChar(Node) + 3)^);
	Message(Application, evBroadcast, cmOpenFile, @Open)
      end;
  end
end;

{ 
}

constructor TMessageWindow.Init(var Bounds: TRect; ATitle: string);
var
  R, CR: TRect;
  Delta: TPoint;
  Gate: PGGate;
  VSB: views.PScrollbar;
  SB: PScrollbar;
begin
  inherited Init(Bounds, ATitle, wnNoNumber);
  Flags := (Flags or wfHideClose) and not wfBackground;
  Palette := wpGrayWindow;
  { Adjust window size
  }
  GetClientRect(CR);
  Dec(CR.B.X, 17); {room for scrollbar}
  Delta.X := (CR.B.X - CR.A.X) mod CharSize.X;
  Delta.Y := (CR.B.Y - CR.A.Y) mod CharSize.Y;
  GrowTo(Size.X - Delta.X, Size.Y - Delta.Y);
(* Enable this if you prefer a text-mode scrollbar...
   Well, I don't think you do. But this is how it works:
  { Create a TV scrollbar, wrap it into a GGate, and insert it
  }
  R.Assign(0, 0, 1, (CR.B.Y - CR.A.Y) div CharSize.Y);
  VSB := New(views.PScrollbar, Init(R));
  Gate := NewGGate(VSB);
  Gate^.MoveTo(CR.B.X - Delta.X, CR.A.Y);
  Insert(Gate);
*)
  { Create a GV scrollbar, and associate an 'up scrollbar' with it
  }
  R.Assign(CR.B.X - Delta.X, CR.A.Y - 1, CR.B.X + 18 - Delta.X, CR.B.Y - Delta.Y + 1);
  SB := New(PScrollbar, Init(R));
  Insert(SB);
  VSB := New(PUpScrollbar, Init(SB));
  { Create a TV TMessageOutline object
  }
  R.Assign(0, 0, (CR.B.X - CR.A.X) div CharSize.X, (CR.B.Y - CR.A.Y) div CharSize.Y);
  MsgOutline := New(PMessageOutLine, Init(R, nil, VSB));
  { Wrap it into a GGate, and insert the 'up scrollbar'
  }
  Gate := NewGGate(MsgOutLine);
  Gate^.TGate^.InsertBefore(VSB, nil);
  Gate^.MoveTo(CR.A.X, CR.A.Y);
  Insert(Gate);
end;

procedure TMessageWindow.ChangeBounds(var Bounds: TRect);
var
  SaveBounds, CR: TRect;
begin
  { adjust the Bounds given such that the interior stays character-aligned
  }
  Frame^.GetBounds(SaveBounds);
  Frame^.SetBounds(Bounds);
  GetClientRect(CR);
  Frame^.SetBounds(SaveBounds);

  Dec(CR.B.X, 17);
  Dec(Bounds.B.X, (CR.B.X - CR.A.X) mod CharSize.X);
  Dec(Bounds.B.Y, (CR.B.Y - CR.A.Y) mod CharSize.Y);
  inherited ChangeBounds(Bounds)
end;

{$endif !Windows}

end.
