// Euclidean-TSP approximation by simulated annealing
// Written by Matthias K"oppe <mkoeppe@csmd.cs.uni-magdeburg.de>
//
// Besides the general iteration algorithm that comes from anneal.h,
// for the ChangeTwo neighbourhood a no-rejection implementation was
// programmed. It is not used here though, since it doesn't have
// performance benefits. The Lin2Opt neighbourhood showed best results.

#include "anneal.h"

int ChangeTwoLength(const Tour &tr, int i, int j, const Matrix<int> *distances = ::distances)
  // i,j are indices in the tour
{
  Tour tour = tr;
  int t = tour[i];
  tour[i] = tour[j];
  tour[j] = t;
  return length(tour, distances);
}

// Tour with a table of all neighbour-tour lengths
class Configuration {
public:
  Tour tour;
  int len;
  vector<int> neighbourlengths;
  Configuration(const Tour &t) :
    tour(t),
    len(length(t))
  {
    int i, j;
    for (i = 1; i<tour.size(); i++) {
      for (j = 0; j<i; j++) {
	neighbourlengths.push_back(ChangeTwoLength(tour, i, j));
      }
    }
  }
  int Length() { return len; }
  int NeighbourLength(int i, int j) {
    if (i==j) return len;
    if (i>j) return neighbourlengths[((i-1)*i)/2 + j];
    return neighbourlengths[((j-1)*j)/2 + i];
  }
};

void SimulatedAnnealingNoReject(SimulatedAnnealing &anneal, Tour &tour, int MaxIter)
{
  int k;
  int minlength = 100000000;
  int tourlength;
  Tour mintour(tour);
  for (k = 0; k<MaxIter; ) {
    tourlength = length(tour);
    if (tourlength < minlength) {
      minlength = tourlength;
      mintour = tour;
    }
    Configuration config(tour);
    
    cout << tourlength << ": ";
    double ProbSum = 0.0;
    vector<double> Probabilities;
    vector<int>::const_iterator n;
    for (n = config.neighbourlengths.begin();
	 n != config.neighbourlengths.end(); ++n) {
      double Prob = (*n)<=tourlength ? 1.0 : exp(((double) (tourlength - (*n))) / anneal.Temperature());      
//       cout << *n << ' ' << Prob << ", ";
      Probabilities.push_back(ProbSum += Prob);
    }
//     cout << endl;

    double Res = (((double) rand()) / (((double)RAND_MAX)+1)) * ProbSum;
    // search of Res in ordered interval list
    vector<double>::const_iterator p = Probabilities.begin();
    int i, j;
    for (i = 1; i<tour.size(); i++) {
      for (j = 0; j<i; j++, ++p) {
	if (Res <= *p) {
	  int t = tour[i];
	  tour[i] = tour[j];
	  tour[j] = t;
	  goto leave;
	}
      }
    }
  leave:
    // wait geometrically (prob = 1-ProbSum/Size) distributed time
    do {
      anneal.IncreaseTime(1); 
      k++;
      cout<<'+';
    } while ((((double) rand()) / (((double)RAND_MAX)+1))
	   * config.neighbourlengths.size() > ProbSum
	&& k<MaxIter);
    cout << endl;
  }

  // if we found better tour before, return that.

  if (minlength < tourlength)
    tour = mintour;
};


/////

int Scale;

double sc(int x)
{
  return x / (double) Scale;
}

double MinLength;

int main(int argc, char **argv)
{
  //ReadCities(cin);
  char name[80];
  strcpy(name, argv[1]);
  strcat(name, ".res");
  {
    ifstream res(name);
    if (res) {
      res >> MinLength;
      cout << "Best tour has a length of " << MinLength << '\n';
    }
    else MinLength = 100000000.0;
  }
  ifstream data(argv[1] /*"cont51.dat"*/);
  ReadCities(data);
  Scale = 1000;
  distances = CalcDistances(cities, Scale);
//   int i, j;
//   for (i = 0; i<cities.size(); i++) {
//     for (j = 0; j<cities.size(); j++)
//       cout << (*distances)[i][j] << '\t';
//     cout << '\n';
//   }

  int i;
//   Tour tour(cities.size());
//   cout << tour << "Length " << sc(length(tour)) << '\n';

  time_t seed;
  time(&seed);
  srand(seed);

  double factor = rand() / (RAND_MAX + 1.0);  //0.55;
  int temp = (int) (20 * (rand() / (RAND_MAX + 1.0))) * Scale; //10 * Scale;
  int iter = 80000;

  if (argc >=3) {
    iter = atoi(argv[2]);
    temp = (int) (atof(argv[3]) * Scale);
    factor = atof(argv[4]);
  }

  cout << "Starting at temperature " << sc(temp)
       << " and decrease factor " << factor
       << " with a seed of " << seed
       << '\n'; 

  //  srand(...);  

//   GeneralIteration(ChangeTwoNH(), LessOrEqual(), tour, iter);
//   cout << tour << "Length " << sc(length(tour)) << '\n';

  Tour tour2(cities.size());
  
  SimulatedAnnealing anneal(temp, iter * factor);

  //SimulatedAnnealingNoReject(anneal, tour2, iter);
  //GeneralIteration(ChangeTwoNH(), anneal, tour2, iter);
  GeneralIteration(Lin2OptNH(), anneal, tour2, iter);

  cout << tour2 << "Length " << sc(length(tour2)) << " Temp " <<
    sc(anneal.Temperature()) << '\n';


  // Post-iteration
  GeneralIteration(ChangeTwoNH(), LessOrEqual(), tour2, 20000);
  cout << tour2 << "Length " << sc(length(tour2))  << '\n';

  double dlen = sc(length(tour2));
  if (dlen < MinLength) {
    cout << "Improved best tour.\n";
    ofstream res(name);
    res << dlen << '\n';
    res << tour2 << '\n';
    int i;
    for (i = 0; i < argc; i++) res << argv[i] << ' ';
    res << '\n';
    res << "Temp: " << sc(temp) << '\n';
    res << "Factor: " << factor << '\n';
    res << "Seed: " << seed << '\n';
  }
} 




