// Symmetric-TSP solver
// Written by Matthias K"oppe <mkoeppe@csmd.cs.uni-magdeburg.de>
//
// Branch and bound, using the assignment-problem relaxation with
// original costs. As this is a poor relaxation, this implementation
// needs very much memory even on low-scale problems (cont26.dat). You
// can't expect results for larger problems.

#include "branch.h"

int AnzZimmer;

TNodePQ NodePQ;
int NewNodeCount = 0, DeleteNodeCount = 0;
char *NodeBlock;
char *NodePtr;

void ReduceMatrix(TIndexMatrix &M, int &sum)
{
  int i, j, min;
  sum = 0;
  for (i = 0; i <= M.Size; i++) {
    min = MAX_INT;
    for (j = 0; j <= M.Size; j++) 
      min = MIN(min, (int) M(i, j));
    if (min < MAX_INT/3) {
      for (j = 0; j<=M.Size; j++)
	M(i, j) -= min;
      sum += min;
    }
  }
  for (j = 0; j <= M.Size; j++) {
    min = MAX_INT;
    for (i = 0; i<=M.Size; i++) 
      min = MIN(min, (int) M(i, j));
    if (min < MAX_INT/3) {
      for (i = 0; i<=M.Size; i++)
	M(i, j) -= min;
      sum += min;
    }
  }
}

int Theta(TIndexMatrix &M, int i, int j) 
{
  int k, l, alpha, beta;
  alpha = MAX_INT;
  for (l=0; l<=M.Size; l++)
    if (l!=j) alpha = MIN(alpha, (int) M(i, l));
  beta = MAX_INT;
  for (k=0; k<=M.Size; k++)
    if (k!=i) beta = MIN(beta, (int) M(k, j));
  return alpha+beta;
}
  
int MaxTheta(TIndexMatrix &M, short int &i, short int &j)
{
  int k, l, Th, MaxTh = -1;
  for (k = 0; k<=M.Size; k++)
    for (l = 0; l<=M.Size; l++)
      if (M(k, l) == 0) {
	Th = Theta(M, k, l);
	if ((Th > MaxTh) && (Th < MAX_INT/3)) {
	  MaxTh = Th;
	  i = k; j = l;
	}
      }
  return MaxTh;
}

TMatrix *MakeMatrix() 
{
  TMatrix *Matrix = new TMatrix(AnzZimmer+1, AnzZimmer+1);
  int i, j;
  for (i = 0; i<=AnzZimmer; i++)
    for (j = 0; j<=AnzZimmer; j++)
      if (i==j) (*Matrix)[i][j] = 65535;
      else (*Matrix)[i][j]
	     = (int) rint(DoubleDist(cities[i], cities[j]) * Scale); 
  return Matrix;
}

TNode *BestRoute;
int BestRouteBound;
int MaxBound;
Tour BestTour(0);

TNode *NextRightNode(TNode *Node)
{
  while (Node->Up && (Node->Up->Right != Node)) Node = Node->Up;
  return Node->Up;
}

int FindNode(TNode *Node, int i)
{
  while (Node && (Node->i != i)) Node = NextRightNode(Node);
  return Node ? Node->j : -1;
}

int Valid(TNode *Node)
{
  Node = NextRightNode(Node);
  if (Node) {
    int i = Node->i;
    int j = Node->j;
    while ((j!=i) && (j!=-1)) j = FindNode(Node, j);
    return j==-1;
  }
  return 1;
}

void Check(TNode *Node)
  // Checks if Node is still needed and discards it and possibly
  // parent nodes to save memory.
{
  while (Node && !Node->LeftValid && !Node->RightValid) {
    TNode *Next = Node->Up;
    if (Next) {
      if (Node == Next->Left) {
	Next->Left = 0;
	Next->LeftValid = 0;
      }
      else {
	Next->Right = 0;
	Next->RightValid = 0;
      }
      delete Node;
    }
    Node = Next;
  }
}

Tour TourOfNode(TNode *Node) return tour(AnzZimmer+1)
{
  int i;
  Tour weg(AnzZimmer+1);
  for (i = 0; i<=AnzZimmer; i++) weg[i] = -1;
  TNode *N = Node->Up;
  int WasRightEnd = 1;
  while (N) {
    if (WasRightEnd) weg[N->i] = N->j;
    WasRightEnd = N->RightEnd;
    N = N->Up;
  }
  weg[Node->Matrix->zi[0]] = Node->Matrix->si[0];
  int j = 0;
  i = 0;
  do {
    tour[j++] = i;
    i = weg[i];
  } while (i);
}

void ShowRoute(TNode *Node)
{
  if (!BestRoute || BestRoute->Bound > Node->Bound) {
    if (BestRoute) {
      delete BestRoute->Matrix; // save mem
      BestRoute->Matrix = 0;
    }
    BestRoute = Node;
    BestRouteBound = Node->Bound;
    BestTour = TourOfNode(Node);
    cout << '\n' << BestTour
	 << " Length " << ((double) length(BestTour))/Scale;     
  }
  else {
    delete Node->Matrix; // save mem
    Node->Matrix = 0;
  }
}

void SplitNode(TNode *Node)
{
  int l, Theta, sum;
  short pi, pj;
  Theta = MaxTheta(*Node->Matrix, pi, pj);
  Node->i = Node->Matrix->zi[pi];
  Node->j = Node->Matrix->si[pj];
  if (Theta == -1) {
    Node->j = -1;
    for (l = 0; l<=Node->Matrix->Size; l++)
      if ((*Node->Matrix)(0, l) < MAX_INT/3)
	Node->j = Node->Matrix->si[l], pj = l;
    Node->i = Node->Matrix->zi[0], pi = 0;
    if (Node->j==-1) {
      if (Node->Up) {
	if (Node->RightEnd) Node->Up->RightValid = 0;
	else Node->Up->LeftValid = 0;
      }
      delete Node->Matrix;
      Node->Matrix = 0;
      return;
    }
    Theta = 0;
  }
  
  TIndexMatrix *LeftMatrix = new TIndexMatrix(*(Node->Matrix));
  (*LeftMatrix)(pi, pj) = 65535;
  ReduceMatrix(*LeftMatrix, sum);
  Node->Left = new TNode(Node->Bound+Theta, LeftMatrix, 0,
			 Node->i, Node->j, Node);
  if (!BestRoute
      || (Node->Left->Bound<BestRoute->Bound
	  && Node->Left->Bound<MaxBound))
    NodePQ.push(Node->Left);
  else {
    delete Node->Left;
    Node->Left = 0;
    Check(Node);
  }

  TIndexMatrix *RightMatrix = Node->Matrix;
  RightMatrix->DeleteRowCol(Node->i, Node->j);
  ReduceMatrix(*RightMatrix, sum);
  Node->Right = new TNode(Node->Bound+sum, RightMatrix, 1,
			  Node->i, Node->j, Node);
  if (!Valid(Node->Right)
      || (BestRoute
	  && (Node->Right->Bound >= BestRoute->Bound
	      ||Node->Right->Bound >= MaxBound))) {
    delete Node->Right; //->Matrix;
    Node->Right = 0;
    //Node->Right->Matrix = 0;
    Node->RightValid = 0;
    Check(Node);
  }
  else {
    if (Node->Right->Matrix->Size==0) {
      ShowRoute(Node->Right);
      Node->RightValid = 0;
      Check(Node);
    }
    else
      NodePQ.push(Node->Right);
  }
  Node->Matrix = 0;
}

TNode *BestNode(TNode *Node)
{
//   TNode *L = 0, *R = 0;
//   if (!Node->Left && Node->LeftValid) return Node;
//   if (Node->RightValid) R = BestNode(Node->Right);
//   if (Node->LeftValid) L = BestNode(Node->Left);
//   if (!L && !R) return 0;
//   if (!L) return R;
//   if (!R) return L;
//   if (L->Bound < R->Bound) return L;
//   else return R;

  if (NodePQ.empty()) return 0;
  TNode *result = NodePQ.top();
  NodePQ.pop();
  return result;
}
    
TNode *MakeRoot()
{
  TIndexMatrix *Matrix = new TIndexMatrix(AnzZimmer, MakeMatrix());
  int Bound;
  ReduceMatrix(*Matrix, Bound);
  return new TNode(Bound, Matrix, 1, -1, -1, 0);
}

void SplitNodes(TNode *Node)
{
  int GreatestShown = 0;
  TNode *Best = Node;
  do {
    SplitNode(Best);
    Best = BestNode(Node);
    if (!Best) return;
//     if (Best->Bound != GreatestShown) {
//       GreatestShown = Best->Bound;
//       cout << '\n' << Best->Bound
// 	   << " (new " << NewNodeCount
// 	   << ", delete " << DeleteNodeCount
// 	   << ", that is " << NewNodeCount-DeleteNodeCount
// 	   << ") " << flush;
//     }
//     else cout << '.' << flush;
  } while (!BestRoute
	   || Best->Bound < BestRouteBound
	   || NodePQ.LowerBoundThan(BestRouteBound));
}

int main(int argc, char **argv)
{
  char name[80];
  strcpy(name, argv[1]);
  strcat(name, ".res");
  {
    ifstream res(name);
    double MinLength;
    if (res) {
      res >> MinLength;
      cout << "Best tour has a length of " << MinLength << '\n';
    }
    else MinLength = 100000000.0;
    MaxBound = (int) ceil(MinLength * Scale) + 1;
  }

  ifstream data(argv[1]);
  ReadCities(data);
  distances = CalcDistances(cities, Scale);
  AnzZimmer = cities.size() - 1;

  // Node memory mgmt
#if defined(BREADTHFIRST)
  NodeBlock = (char *) malloc(70*1024*1024);
  NodePtr = NodeBlock;
#endif  

  TNode *Node = MakeRoot();
  BestRoute = 0;
  //  NodePQ.push(Node);
  
  SplitNodes(Node);

  //  Tour tour = TourOfNode(BestRoute);
  cout << '\n' << BestTour
       << " Length " << ((double) length(BestTour))/Scale; 
}

