/*---------------------------------------------------------------------------+
| This is the header file classes lookup, exp_T and Matrix                   |
|                                                                            |
|    exp_T   implementation of an expression tree with variable and          |
|            function lookup                                                 |
|    Matrix  implementation of a sparse matrix with symbolic entries         |
|                                                                            |
| The class lookup is a simple (and not very efficient) lookup table         |
+---------------------------------------------------------------------------*/

class lookup {
    char names[100][10];
    double val[100];
public:
    unsigned int nr;
    lookup() : nr(0) {}
    const char* name(unsigned int i) const { return (i<nr) ? names[i] : (char*)0; }
    double value(unsigned int i) const { return (i<nr) ? val[i] : 0; }
    double value (const char*) const;
    void insert(const char*, double);
    void init(const char* filename);
    bool is_var(const char*);
};

enum Operator {
    PLUS, MINUS, MULT, DIV, POWER, FUNCT, VALUE, VARIABLE, BAD
};

class func_T;
typedef double (*DFuncD)(double);

/*---------------------------------------------------------------------------+
| class exp_T
|
| An expression tree is a binary tree whose root is an operation and whose
|    left and right children are expression trees.
|
| The allowable operations are
|    PLUS, MINUS, MULT, DIV, POWER, FUNCT, VALUE, VARIABLE, BAD
|
| If the operation is FUNCT (i.e., a function), the left children is the
|    parameter which should be passed to the function and the right tree is
|    NULL and f is a pointer to the function
|
| If the operation is VARIABLE, the left and right children are NULL and
|    the name of the variable is stored in variable.
|
| If the operation is VALUE, the value is stored in v and the left and
|    right children are NULL
*/
class exp_T {
public:
    Operator    op;
    union {
        exp_T*  child[2];
        func_T* f;
        double  v;
        char    variable[10];
    } u;
    static lookup table;
    exp_T();
    exp_T(const char* filename) { if (filename) table.init(filename); }
    exp_T(lookup t) { table = t; }
    ~exp_T();

    double value();
    friend exp_T* BuildExpression(char** t);
    friend exp_T* BuildTerm(char** t);
    friend exp_T* BuildFactor(char** t);
    void show();
};

/*---------------------------------------------------------------------------+
| class func_T
|
| This class implements a function of one variable
|
| func is a pointer to a function with prototype
|        double func(double);
|
| The argument is an expression tree
*/
class func_T {
public:
    DFuncD          func;
    exp_T*          arg;
    ~func_T() { if (arg) delete arg; }
    double eval() { return func(arg->value()); }
};

/*---------------------------------------------------------------------------+
| class MatEntry
|
| This is an auxiliary class to implement a symbolic matrix entry.
|
| A symbolic matrix entry consists of a variable name and the row and
|    column in which that variable is stored in a matrix
*/
class MatEntry {
public:
    char variable[100];
    int row;
    int col;
    MatEntry(const char* name, int _row, int _col) : row(_row), col(_col)
        {
            strncpy(variable, name, 10);
        }
    MatEntry() : row(0), col(0) {}
};

/*---------------------------------------------------------------------------+
| class Matrix
|
| This implements a sparse matrix whose entries are variable names.  The
|    variable values are stored in a lookup table
*/
class Matrix {
    int nr;
    MatEntry* matrix;
public:
    int rows;
    int cols;
    Matrix(const char* filename, const char* initvalue);
    double** genMatrix();
    void update(const char* variable, double value);
    void show();
    static lookup table;
};

exp_T*      BuildExpression(char** t);
exp_T*      BuildTerm(char** t);
exp_T*      BuildFactor(char** t);
char*       funcName(DFuncD func);
DFuncD      getFunc(char* name);
char*       getToken(char* str = 0);




