/*---------------------------------------------------------------------------+
| This file contains two classes                                             |
|                                                                            |
|    exp_T   implementation of an expression tree with variable and          |
|            function lookup                                                 |
|    Matrix  implementation of a sparse matrix with symbolic entries         |
|                                                                            |
| The class lookup is a simple (and not very efficient) lookup table         |
+---------------------------------------------------------------------------*/
#pragma hdrstop
#include <iostream.h>
#include <iomanip.h>
#include <string.h>
#include <math.h>
#include <stdio.h>
#include <ctype.h>
#include "exp.h"

// lookup exp_T::table;

double x = 0;

/*---------------------------------------------------------------------------+
| exp_T::exp_T
|
| This is the default constructor which implements an empty tree
*/
exp_T::exp_T()
{
    u.child[0] = u.child[1] = 0;
    op = BAD;
}

/*---------------------------------------------------------------------------+
| exp_T::~exp_T
|
| This is the destructor for the tree.
*/
exp_T::~exp_T()
{
    switch (op) {
    case PLUS:
    case MINUS:
    case MULT:
    case DIV:
    case POWER:
        if (u.child[0])
            delete u.child[0];
        if (u.child[1])
            delete u.child[1];
        break;
    case FUNCT:
        if (u.f)
            delete u.f;
    default:
        break;
    }
}


/*---------------------------------------------------------------------------+
| getToken
|
| This is a utility function which retrieves the next token from a
|    character string.  If str is not NULL, the function returns the first
|    token in str.  If str is NULL, return the next token from the string
|    currently being processed
*/
char*
getToken(char* str)
{
    static char     lastchar = 0;
    static char*    next = 0;
    char*           token;
    if (!str) {
        str = next;
        if (str)
            *str = lastchar;
        else
            return 0;
    }
    while (isspace(*str))
        str++;
    token = str;
    switch (*str) {
    case '(':
    case ')':
    case '+':
    case '-':
    case '*':
    case '/':
    case '^':
    case '=':
        lastchar = *++str;
        *str = '\0';
        break;
    default:
        if (isdigit(*str)) {
            do {
                str++;
            } while (isdigit(*str));
            if (*str == '.') {
                str++;
                while (isdigit(*str))
                    str++;
            }
        } else if (isalpha(*str)) {
            do {
                str++;
            } while (isalpha(*str));
        }
        lastchar = *str;
        *str = '\0';
    }
    next = str;
    return token;
}

/*---------------------------------------------------------------------------+
| BuildExpression
|
| Given a pointer to the location of a token in a character string, build
|    an expression tree from the Expression starting at this position.
|
| An exception is thrown if the expression is malformed.
*/
exp_T*
BuildExpression(char** t)
{
    exp_T* term = BuildTerm(t);

    if (term) while (*t && ((**t == '+') || (**t == '-'))) {
        exp_T* express = new exp_T;
        if (**t == '+') {
            express->op = PLUS;
        } else {
            express->op = MINUS;
        }
        express->u.child[0] = term;
        *t = getToken();
        term = BuildTerm(t);
        if (term) {
            express->u.child[1] = term;
            term = express;
        } else {
            delete express;
            break;
        }
    }
    return term;
}

/*---------------------------------------------------------------------------+
| BuildTerm
|
| Given a pointer to the location of a token in a character string, build
|    an expression tree from the Term starting at this position.
|
| An exception is thrown if the Term is malformed.
*/
exp_T*
BuildTerm(char** t)
{
    exp_T* factor = BuildFactor(t);
    if (factor) {
        while (*t && ((**t == '^') || (**t == '*') || (**t == '/'))) {
            exp_T* term = new exp_T;
            switch (**t) {
            case '^':
                term->op = POWER;
                break;
            case '*':
                term->op = MULT;
                break;
            case '/':
                term->op = DIV;
                break;
            }
            term->u.child[0] = factor;
            *t = getToken();
            factor = BuildFactor(t);
            if (factor) {
                term->u.child[1] = factor;
                factor = term;
            } else {
                delete term;
                break;
            }
        }
    }
    return factor;
}


/*---------------------------------------------------------------------------+
| BuildFactor
|
| Given a pointer to the location of a token in a character string, build
|    an expression tree from the Factor starting at this position.
|
| An exception is thrown if the Factor is malformed.
*/
exp_T*
BuildFactor(char** t)
{
    exp_T* e = 0;

    if (**t == '(') {
        if ((*t = getToken())) {
            e = BuildExpression(t);
            if (!e || (**t != ')')) {
                throw "Error in expression\n";
            }
        }
        *t = getToken();
        return e;
    }
    if (**t == 'x') {
        e = new exp_T;
        e->op = VARIABLE;
        *t = getToken();
         return e;
    }
    if (isdigit(**t)) {
        double          val;
        e = new exp_T;
        e->op = VALUE;
        val = atof(*t);
        e->u.v = val;
        *t = getToken();
        return e;
    }
    if (isalpha(**t)) {
        DFuncD  func;
        if ((func = getFunc(*t))) {
            e = new exp_T;
            e->op = FUNCT;
            e->u.f = new func_T;
            e->u.f->func = func;
            *t = getToken();
            if (**t != '(') {
                throw "Error in function arg -- no open paren\n";
            }
            *t = getToken();
            e->u.f->arg = BuildExpression(t);
            if ((e->u.f->arg == 0) || (**t != ')')) {
                delete e;
                throw "Error in function arg -- no close paren\n";
            }
            *t = getToken();
            return e;
        }
    }
    throw "Error in factor\n";
}

/*---------------------------------------------------------------------------+
| exp_T::value
|
| This method returns the value of an expression tree.
*/
double
exp_T::value()
{
    double          a, b;
    switch (op) {
    case BAD:
        break;
    case VALUE:
        return u.v;
    case FUNCT:
        if (u.f->arg) {
            a = u.f->arg->value();
            if (u.f->arg->op != BAD)
                return ((u.f->func)(a));

            delete u.f->arg;
            u.f->arg = 0;
        }
        break;
//  case VARIABLE:
//        return table.value(u.variable);
    case PLUS:
    case MINUS:
    case MULT:
    case DIV:
    case POWER:
        if (u.child[0] && u.child[1]) {
            a = u.child[0]->value();
            b = u.child[1]->value();
            if ((u.child[0]->op == BAD) || (u.child[1]->op == BAD)) {
                delete u.child[0];
                delete u.child[1];
                u.child[0] = u.child[1] = 0;
                break;
            }
            switch (op) {
            case PLUS:
                return a + b;
            case MINUS:
                return a - b;
            case MULT:
                return a * b;
            case DIV:
                if (b)
                    return a / b;
                break;
            case POWER:
                return pow(a, b);
            default:
                break;
            }
        }
    }
    op = BAD;
    throw "Error in evaluation\n";
}

void
exp_T::show()
{
    static          level = 0;
    char            ch = ' ';
    level += 5;
    switch (op) {
    case PLUS:
        if (ch == ' ')
            ch = '+';
    case MINUS:
        if (ch == ' ')
            ch = '-';
    case DIV:
        if (ch == ' ')
            ch = '/';
    case MULT:
        if (ch == ' ')
            ch = '*';
        u.child[1]->show();
        cout << setw(level) << ch << endl;
//      printf("%*c\n", level, ch);
        u.child[0]->show();
        break;
    case VARIABLE:
        cout << setw(level) << ' ' << u.variable << endl;
//      printf("%*c%s\n", level, ' ', u.variable);
        break;
    case VALUE:
        cout << setw(level) << ' ' << setprecision(4) << u.v << endl;
//      printf("%*c%2.1f\n", level, ' ', u.v);
        break;
    case FUNCT:
        cout << setw(level) << ' ' << funcName(u.f->func) << endl;
//      printf("%*c%s\n", level, ' ', funcName(u.f->func));
        u.f->arg->show();
        break;
    }
    level -= 5;
}



