// -*- C++ -*-

#include "asmmacro.h"

typedef struct TMemBlock{
  unsigned size;
  TMemBlock *prev, *next, *nextfree;
  unsigned dummy[3]; // => n*16 size
  unsigned flags; // must be the last field in structure to distinguish between normal and smb blocks
} TMemBlock;

// SMALLBLOCKSIZE = maximum size of a "small" block in byte
#define SMALLBLOCKSIZE 28 // n*16 + 12, n=0,1,2,... ; 4 byte status flags
#define SMALLBLOCKGRAN 25 // 25 seems to be fastest (<25: longer block search; >10: longer map scans)
#define SMBCount  32*SMALLBLOCKGRAN
#define SMBMapSize ((4*SMALLBLOCKGRAN+15) & 0x0fff0)  // => TSmallMemBlock.TSMBEntry[].data para-aligned
// TSmallMemBlock: +0 - next; +4 - prev; +8 - free; +12 .. +n*16+11 - map; +n*16+12 - TSMBEntry.flags 
// +n*16+16 - TSMBEntry.data

#undef FREEEMPTYSMB // do not free smb's -> ~30% speedup

typedef struct TSMBEntry {
  unsigned flags; // mf_smb + entry number; must be the last field before data
  char data[SMALLBLOCKSIZE];
} TSMBEntry;

typedef struct TSmallMemBlock{
  TSmallMemBlock *next, *prev;
  unsigned free;
  unsigned char map[SMBMapSize];
  TSMBEntry blocklist[SMBCount];
} TSmallMemBlock;

static unsigned char smb_search_map[256];
static unsigned char smb_mask_map[256];

enum {
  mf_free, // block free
  mf_used, // block used
  mf_last, // last block at all
  mf_dirty, // block already freed
  mf_smb = 0x40000000 // smb pointer (2^31)
} MemFlags;

typedef struct TMemInfo {
  TMemBlock *Free;
  TSmallMemBlock *SMBStart;
} TMemInfo;

static unsigned ERamStart = 0;
static unsigned ERamSize  = 0;
static TMemInfo *MemInfo;

void memclean()
{
  TMemBlock *b;

  // use freelist

  b = (TMemBlock *)ERamStart;
  do {
    if (b->flags == mf_free) memfilll(((unsigned)(b))+sizeof(TMemBlock), 0, (b->size)>>2);
    b = b->next;
  } while (b);
}

unsigned initmem(unsigned start, unsigned size)
{
  TMemBlock *b;
  unsigned i;

  ERamStart = (start+15) & 0x0fffffff0; // start para-aligned
  ERamSize = (size-16) & 0x0fffffff0;

  // Nicht-konstante Verwaltungsinfos auch in Shared Memory!
  MemInfo = (TMemInfo *) ERamStart;
  ERamStart = (ERamStart + (sizeof(TMemInfo) + 15) & 0x0fffffff0);
  ERamSize = (ERamSize - sizeof(TMemInfo) - 16);
  MemInfo->Free = (TMemBlock *)ERamStart;
  MemInfo->SMBStart = 0;
  
  // Startblock anlegen 
  b = MemInfo->Free; //(TMemBlock *)ERamStart;
  b->size = ERamSize -2*sizeof(TMemBlock);
  b->prev = 0;
  b->next = (TMemBlock *)(ERamStart + ERamSize - sizeof(TMemBlock));
  b->flags= mf_free;
  b->nextfree = 0;

  // Endblock  anlegen
  b = (TMemBlock *)b->next;
  b->prev = (TMemBlock *)ERamSize;
  b->size = 0;
  b->next = 0;
  b->flags= mf_last;
  b->nextfree = 0;

  memclean(); 

  for (i=  0; i<  1; i++) smb_search_map[i] = 0;
  for (i=  1; i<  2; i++) smb_search_map[i] = 1;
  for (i=  2; i<  4; i++) smb_search_map[i] = 2;
  for (i=  4; i<  8; i++) smb_search_map[i] = 3;
  for (i=  8; i< 16; i++) smb_search_map[i] = 4;
  for (i= 16; i< 32; i++) smb_search_map[i] = 5;
  for (i= 32; i< 64; i++) smb_search_map[i] = 6;
  for (i= 64; i<128; i++) smb_search_map[i] = 7;
  for (i=128; i<192; i++) smb_search_map[i] = 6;
  for (i=192; i<224; i++) smb_search_map[i] = 5;
  for (i=224; i<240; i++) smb_search_map[i] = 4;
  for (i=240; i<248; i++) smb_search_map[i] = 3;
  for (i=248; i<252; i++) smb_search_map[i] = 2;
  for (i=252; i<254; i++) smb_search_map[i] = 1;
  for (i=254; i<255; i++) smb_search_map[i] = 0;

  for (i=  0; i<  1; i++) smb_mask_map[i] = 1;
  for (i=  1; i<  2; i++) smb_mask_map[i] = 2;
  for (i=  2; i<  4; i++) smb_mask_map[i] = 4;
  for (i=  4; i<  8; i++) smb_mask_map[i] = 8;
  for (i=  8; i< 16; i++) smb_mask_map[i] = 16;
  for (i= 16; i< 32; i++) smb_mask_map[i] = 32;
  for (i= 32; i< 64; i++) smb_mask_map[i] = 64;
  for (i= 64; i<128; i++) smb_mask_map[i] = 128;
  for (i=128; i<192; i++) smb_mask_map[i] = 64;
  for (i=192; i<224; i++) smb_mask_map[i] = 32;
  for (i=224; i<240; i++) smb_mask_map[i] = 16;
  for (i=240; i<248; i++) smb_mask_map[i] = 8;
  for (i=248; i<252; i++) smb_mask_map[i] = 4;
  for (i=252; i<254; i++) smb_mask_map[i] = 2;
  for (i=254; i<255; i++) smb_mask_map[i] = 1;


  return ERamStart; // frs Debugging
}

char *memalloc(unsigned);

char *smallalloc(unsigned size)
{
  struct TSmallMemBlock *b;
  unsigned pos, i, mask, map;
  unsigned *clear;

  if (!MemInfo->SMBStart) {
    MemInfo->SMBStart = (TSmallMemBlock *)memalloc(sizeof(TSmallMemBlock));
    //    memset(MemInfo->SMBStart, 0, sizeof(TSmallMemBlock));

    clear = (unsigned *)MemInfo->SMBStart;
    memset(clear, 0, 16+SMBMapSize);
    for (i=0; i<SMBCount; i++) {*clear = 0; clear+= (SMALLBLOCKSIZE+4)/4;};

    MemInfo->SMBStart->free = SMBCount;
//     MemInfo->SMBStart->next = 0; // already 0
//     MemInfo->SMBStart->prev = 0;
    if (!MemInfo->SMBStart) return 0;
  }
  b = MemInfo->SMBStart;
  while (b && (!b->free) && (b->next)) b = b->next;
  if ((!b->free) && (!b->next)) {
    b->next = (TSmallMemBlock *)memalloc(sizeof(TSmallMemBlock));
    if (!b->next) return 0;
    //    memset(b->next, 0, sizeof(TSmallMemBlock));

    clear = (unsigned *)b->next;
    memset(clear, 0, 16+SMBMapSize);
    for (i=0; i<SMBCount; i++) {*clear = 0; clear+= (SMALLBLOCKSIZE+4)/4;};

    b->next->free = SMBCount;
    b->next->prev = b;
    b->next->next = 0;
    b = b->next;
  }
  if (b) {
    i = 0;
    pos = 0;
    while ((i<SMBMapSize) && (b->map[i] == 0xff)) { 
      i++;
      pos +=8;
    }
    mask = smb_mask_map[b->map[i]];
    pos += smb_search_map[b->map[i]];

    b->map[i] |= mask;
//     cout << "smallalloc: map["<<i<<"] |= " << hex << mask << endl;
//     cout << "smb = " << hex << (unsigned)b << endl;
    b->free--;
    b->blocklist[pos].flags = mf_smb | pos;
    return (char *)(&b->blocklist[pos].data);
  }
}

char *memalloc(unsigned size)
{
  TMemBlock *b, *nb, *pf;
  char *ptr;
  unsigned bsize;

  if (size<=SMALLBLOCKSIZE) return smallalloc(size);
  
  size = (size+15) & 0xfffffff0; // round to paragraph size

  ptr = 0;

  pf = 0;
  b = (TMemBlock *)MemInfo->Free;
  while ( b && (b->size < size) && (b->nextfree) ) {
    pf = b;
    b = (TMemBlock *)b->nextfree;
  }
  if (!b) return 0; // out of memory
  if (b->size < size) return 0;
  bsize = b->size;
  if (bsize>size+sizeof(TMemBlock)) {
    // dann splitten wir halt den Speicherblock in zwei
    nb = (TMemBlock *)(((int)(b)) + sizeof(TMemBlock) + size);
    b->size = size;
    b->flags= mf_used;
    nb->next = b->next;
    b->next = nb;
    nb->prev = b;
    nb->flags = mf_free;
    nb->size = bsize - sizeof(TMemBlock) - size;
    nb->next->prev = nb;
    nb->nextfree = b->nextfree;
    b->nextfree = 0;
    if (pf) pf->nextfree = nb; else MemInfo->Free = nb;
    return (char *)(((int)(b))+sizeof(TMemBlock));
  } 
  else {
    b->flags = mf_used;
    if (pf) pf->nextfree = b->nextfree; else MemInfo->Free = b->nextfree;    
    return (char *)(((int)(b))+sizeof(TMemBlock));
  }
}

void memfree(char *ptr);

void smallfree(char *ptr, unsigned num)
{
  TSmallMemBlock *smb;
  char mask;

  num &= ~mf_smb;

//   cout << "smallfree: ptr = " << hex << (unsigned)ptr << " num = " << num << endl;
  smb = (TSmallMemBlock *) ( (unsigned) ptr - SMBMapSize - 4*sizeof(unsigned)
			     - sizeof(TSMBEntry) * num);
//   cout <<"smb = " << hex << (unsigned)smb << endl;
  mask = ~(1 << (num & 7));
  smb->map[num >> 3] &= mask;
  smb->free++;

#if (defined FREEEMPTYSMB)
  if (smb->free == SMBCount) {
    if (smb->prev) smb->prev->next = smb->next;
    else MemInfo->SMBStart = smb->next;
    if (smb->next) smb->next->prev = smb->prev;
    memfree((char *)smb);
  }
#endif

}

void memfree(char *ptr)
{
  TMemBlock *b, *nb, *pb, *fb, *pfb;
  b = (TMemBlock *)(((unsigned)(ptr))-sizeof(TMemBlock));

  if (((unsigned)b<ERamStart) || ((unsigned)b>ERamStart+ERamSize -1)) return; // pointer out of bounds

//   cout << "memfree (ptr = "<< hex << (unsigned)ptr << "): flags = " << b->flags << endl;

  if (b->flags & mf_smb) { // pointer belongs to a small memory block
    smallfree(ptr, b->flags);
    return;
  }

  if (b!= b->next->prev) return; // not a valid pointer
  if (b->flags != mf_used) return; // pointer not in use
  b->flags = mf_free;
  nb = b->next;
  if (nb && (nb->flags != mf_free)) nb = 0;
  pb = b->prev;
  if (pb && (pb->flags != mf_free)) pb = 0;

  if (pb && nb) {
    // nb and pb are free
    pb->size += b->size + nb->size + 2*sizeof(TMemBlock);
    b->flags = mf_dirty;
    nb->flags = mf_dirty;
    pb->next = nb->next;
    nb->next->prev = pb;
    pb->nextfree = nb->nextfree;
    b->nextfree = 0;
    nb->nextfree = 0;
    return;
  }

  if (nb) {
    fb = MemInfo->Free; pfb = 0;
    while (fb && (fb != nb)) {
      pfb = fb;
      fb = fb->nextfree;
    }
    if (pfb) pfb->nextfree = b; else MemInfo->Free = b;
    b->nextfree = nb->nextfree;
    nb->nextfree = 0;
    b->size += nb->size + sizeof(TMemBlock);
    b->next = nb->next;
    nb->next->prev = b;
    nb->flags = mf_dirty;
    return;
  }
  if (pb) {
    pb->size += b->size + sizeof(TMemBlock);
    pb->next = b->next;
    b->next->prev = pb;
    b->flags = mf_dirty;
    return;
  }

  fb = MemInfo->Free;
  pfb = 0;
  while (fb && (fb < b)) {
    pfb = fb;
    fb = fb->nextfree;
  }

  if (pfb) {
    b->nextfree = pfb->nextfree;
    pfb->nextfree = b;
  } 
  else {
    b->nextfree = MemInfo->Free;
    MemInfo->Free = b;
  } 

  // auf das 0-Setzen der freigegebenen Bloecke verzichten wir mal aus Geschwindigkeitsgrueden
  // durch das mf_dirty Flag laesst sich erkennen, ob der Block schonmal freigegeben wurde und 
  // mit einem angrenzenden (prev) Block verschmolzen wurde
}

void smbstate(unsigned &blocks, unsigned &used, unsigned &free, unsigned &blocksize)
{
  TSmallMemBlock *smb;

  blocks = used = free = 0;
  blocksize = SMALLBLOCKSIZE;
  
  smb = MemInfo->SMBStart;
  while (smb) {
    used += SMBCount - smb->free;
    free += smb->free;
    blocks ++;
    smb = smb->next;
  }
}

#if !defined(_LowLevel)
void checksmb()
{
  TSmallMemBlock *smb;
  unsigned i;
  
  i = 0;
  smb = MemInfo->SMBStart;
  while (smb) {
    cout << dec << i << ". smb at " << hex << (unsigned)smb << dec << " => " << smb->free <<
      " blocks free" << endl;
    i++;
    smb = smb->next;
  }
}
#endif

unsigned memfreesize()
{
  TMemBlock *b;
  unsigned size=0;

  b = (TMemBlock *)ERamStart;
  do { 
    if (b->flags==mf_free) size+=b->size;
    b = b->next;
  } while (b);
  return size;
}

unsigned memmaxsize()
{
  TMemBlock *b;
  unsigned size=0;
  
  b = (TMemBlock *)ERamStart;
  do {
    if ((b->flags == mf_free)&&(b->size > size)) size = b->size;
    b = b->next;
  } while (b);
  return size;
}
