"EssoLib, that is the library of semi-built-in functions"
"and syntax forms for Esso Scheme."
"Written by Matthias Koeppe <mkoeppe@csmd.cs.uni-magdeburg.de>"

"Type predicates"
(define (null? obj) (eq? (type obj) (quote null)))
(define (number? obj) (eq? (type obj) (quote number)))
(define (pair? obj) (eq? (type obj) (quote pair)))
(define (string? obj) (eq? (type obj) (quote string)))
(define (symbol? obj) (eq? (type obj) (quote symbol)))
(define (procedure? obj) (or (eq? (type obj) (quote lambda))
			     (eq? (type obj) (quote primitive))))
(define (vector? obj) (eq? (type obj) (quote vector)))
(define (input-port? obj) (eq? (type obj) (quote input-port)))
(define (output-port? obj) (eq? (type obj) (quote output-port)))
(define (char? obj) (eq? (type obj) (quote char)))
(define (boolean? obj) (eq? (type obj) (quote boolean)))

"Car derivatives"
(define (caar obj) (car (car obj)))
(define (cadr obj) (car (cdr obj)))
(define (cdar obj) (cdr (car obj)))
(define (cddr obj) (cdr (cdr obj)))
(define (caaar obj) (car (caar obj)))
(define (caadr obj) (car (cadr obj)))
(define (cadar obj) (car (cdar obj)))
(define (caddr obj) (car (cddr obj)))
(define (cdaar obj) (cdr (caar obj)))
(define (cdadr obj) (cdr (cadr obj)))
(define (cddar obj) (cdr (cdar obj)))
(define (cdddr obj) (cdr (cddr obj)))
(define (caaaar obj) (car (caaar obj)))
(define (caaadr obj) (car (caadr obj)))
(define (caadar obj) (car (cadar obj)))
(define (caaddr obj) (car (caddr obj)))
(define (cadaar obj) (car (cdaar obj)))
(define (cadadr obj) (car (cdadr obj)))
(define (caddar obj) (car (cddar obj)))
(define (cadddr obj) (car (cdddr obj)))
(define (cdaaar obj) (cdr (caaar obj)))
(define (cdaadr obj) (cdr (caadr obj)))
(define (cdadar obj) (cdr (cadar obj)))
(define (cdaddr obj) (cdr (caddr obj)))
(define (cddaar obj) (cdr (cdaar obj)))
(define (cddadr obj) (cdr (cdadr obj)))
(define (cdddar obj) (cdr (cddar obj)))
(define (cddddr obj) (cdr (cdddr obj)))

"Higher-order procs"
(define (map proc l)
  (if (null? l)
      ()
      (cons (apply proc (list (car l)))
	    (map proc (cdr l)))))

(define eqv? eq?)
(define (equal? a b)
  (and (eq? (type a) (type b))
       (cond ((string? a) (string=? a b))
	     ((pair? a) (and (equal? (car a) (car b))
			     (equal? (cdr a) (cdr b))))
	     (else (eqv? a b)))))

(define (member-aux eq-pred? obj l)
  (if (null? l)
      #f
      (if (eq-pred? obj (car l))
	  l
	  (member-aux eq-pred? obj (cdr l)))))
(define (memv obj l)
  (member-aux eqv? obj l))
(define (memq obj l)
  (member-aux eq? obj l))
(define (member obj l)
  (member-aux equal? obj l))

(define (assoc-aux eq-pred? obj al)
  (if (null? al)
      #f
      (if (eq-pred? obj (caar al))
	  (car al)
	  (assoc-aux eq-pred? obj (cdr al)))))
(define (assq obj al) 
  (assoc-aux eq? obj al))
(define (assv obj al)
  (assoc-aux eqv? obj al))
(define (assoc obj al)
  (assoc-aux equal? obj al))

(define (append l1 . lists)
  (if (null? lists)
      l1
      (if (null? l1) 
	  (apply append lists)
	  (cons (car l1)
		(apply append (cons (cdr l1) lists))))))

"User syntax forms"

(define (let->lambda bindings body)
  (cons (cons (quote lambda)
	      (cons (map car bindings)
		    body))
	(map cadr bindings)))

(define (eval-let frame bindings . body)
  (eval (let->lambda bindings body) frame))

(define (let*->lambda bindings body)
  (if (null? bindings)
      (cons (quote begin) body)
      (let->lambda (list (car bindings))
		   (list (let*->lambda (cdr bindings) body)))))

(define (eval-let* frame bindings . body)
  (eval (let*->lambda bindings body) frame))

"ESSO supports implicit forcing; we just apply a primitive,"
"and the promise is replaced by the promised value."
(define (force promise)
  (type promise)
  promise)
(define (eval-delay frame expression)
  (eval (list (quote make-promise)
	      (list (quote lambda) () expression))
	frame))

(define (eval-cond-aux frame clauses)
  (if (null? clauses)
      ()
      (if (eq? (caar clauses) (quote else))
	  (eval (cons (quote begin) (cdar clauses)) frame)
	  (if (eval (caar clauses) frame)
	      (eval (cons (quote begin) (cdar clauses)) frame)
	      (eval-cond-aux frame (cdr clauses))))))

(define (eval-cond frame . clauses)
  (eval-cond-aux frame clauses))

(define (eval-case-aux frame evaluated-key clauses)
  (if (null? clauses)
      ()
      (if (eq? (caar clauses) (quote else))
	  (eval (cons (quote begin) (cdar clauses)) frame)
	  (if (memv evaluated-key (caar clauses))
	      (eval (cons (quote begin) (cdar clauses)) frame)
	      (eval-case-aux frame evaluated-key (cdr clauses))))))

(define (eval-case frame key . clauses)
  (eval-case-aux frame (eval key frame) clauses))

(define (eval-sequence frame . seq)
  (eval (cons 'begin seq) frame))

"Quasiquotation; nested qqs not ok yet."
(define (eval-unquote-list-aux frame nesting l)
  (case (car l)
    ((unquote)
     (list (eval (cadr l) frame)))
    ((unquote-splicing) 
     (eval (cadr l) frame))
    (else (list l))))

(define (eval-quasiquote-aux frame nesting head tail)
  (cond ((null? tail) head)
	((pair? tail)
	 (if (pair? (car tail))
	     (eval-quasiquote-aux frame nesting
				  (append head
					  (eval-unquote-list-aux frame nesting
								 (car tail)))
				  (cdr tail))
	     (case (car tail)
	       ((unquote unquote-splicing)
		(append head
			(car (eval-unquote-list-aux frame nesting tail))))
	       (else
		(eval-quasiquote-aux frame nesting
				     (append head
					     (list (car tail)))
				     (cdr tail))))))
	(else (append head tail))))

(define (eval-quasiquote frame template)
  (eval-quasiquote-aux frame 1 () template))

"Install new syntax forms."
(set! syntax-alist (append (list (cons 'let eval-let)
				 (cons 'cond eval-cond)
				 (cons 'let* eval-let*)
				 (cons 'case eval-case)
				 (cons 'sequence eval-sequence)
				 (cons 'delay eval-delay)
				 (cons 'quasiquote eval-quasiquote))
			   syntax-alist))

"Check if task-force works correctly."
(define good-task-force #f)

"This test does not tell us if the file functions work task-forced."
(task-force (delay (set! good-task-force #t)))
  
"List operations"

(define (length list)
  (if (null? list) 
      0
      (+ 1 (length (cdr list)))))

(define (list-tail x k)
  (if (zero? k)
      x
      (list-tail (cdr x) (- k 1))))

(define (list-head-aux h t k)
  (if (zero? k)
      h
      (list-head-aux (append h (list (car t)))
		     (cdr t) k)))	 

(define (list-head x k)
  (list-head-aux () x k))

(define (list-ref x k)
  (car (list-tail x k)))

(define (reverse l)
  (if (or (null? l)
	  (null? (cdr l)))
      l
      (append (reverse (cdr l))
	      (list (car l)))))

(define (remove-if l pred?)
  (cond ((null? l) l)
	((pred? (car l)) (remove-if (cdr l) pred?))
	(else (cons (car l) (remove-if (cdr l) pred?)))))

(define (rac l)
  (if (null? (cdr l))
      (car l)
      (rac (cdr l))))

"Numerical operation"

(define (not bool) (if bool #f #t))
(define (zero? n) (= n 0))
(define (positive? n) (> n 0))
(define (negative? n) (< n 0))
(define (even? n) (= n (* (quotient n 2) 2)))
(define (odd? n) (not (even? n)))
(define (abs n) (if (negative? n) (- n) n))  

"Char functions"

(define (char=? c1 c2) (= (char->integer c1) (char->integer c2)))
(define (char<? c1 c2) (< (char->integer c1) (char->integer c2)))
(define (char>? c1 c2) (> (char->integer c1) (char->integer c2)))
(define (char<=? c1 c2) (<= (char->integer c1) (char->integer c2)))
(define (char>=? c1 c2) (>= (char->integer c1) (char->integer c2)))
(define (char-numeric? c) (and (char>=? c #\0) (char<=? c #\9)))
(define (char-whitespace? c) (if (memv c (quote (#\space #\newline))) #t #f))
(define (char-upper-case? c) (and (char>=? c #\A) (char<=? c #\Z)))
(define (char-lower-case? c) (and (char>=? c #\a) (char<=? c #\z)))
(define (char-alphabetic? c) (or (char-upper-case? c) (char-lower-case? c)))
(define (char-upcase c)
  (if (char-lower-case? c)
      (integer->char (- (char->integer c) 32))
      c))
(define (char-downcase c)
  (if (char-upper-case? c)
      (integer->char (+ (char->integer c) 32))
      c))
(define (char-ci=? c1 c2) (char=? (char-upcase c1) (char-upcase c2)))
(define (char-ci<? c1 c2) (char<? (char-upcase c1) (char-upcase c2)))
(define (char-ci>? c1 c2) (char>? (char-upcase c1) (char-upcase c2)))
(define (char-ci<=? c1 c2) (char<=? (char-upcase c1) (char-upcase c2)))
(define (char-ci>=? c1 c2) (char>=? (char-upcase c1) (char-upcase c2)))

"String functions"

(define (list->string-aux s pos l)
  (if (null? l)
      s
      (begin
	(string-set! s pos (car l))
	(list->string-aux s (+ pos 1) (cdr l)))))

(define (list->string l)
  (list->string-aux (make-string (length l)) 0 l))

(define (string . char-list)
  (list->string char-list))

(define (string-compare-aux char-predicate? char-equal? number-predicate? pos s1 l1 s2 l2)
  (cond ((or (>= pos l1) (>= pos l2))
	 (number-predicate? l1 l2))
	((char-predicate? (string-ref s1 pos) (string-ref s2 pos))
	 #t)
	((char-equal? (string-ref s1 pos) (string-ref s2 pos))
	 (string-compare-aux char-predicate? char-equal? number-predicate?
			     (+ pos 1) s1 l1 s2 l2))
	(else #f)))

(define (string-compare char-predicate? char-equal? number-predicate? s1 s2)
  (string-compare-aux char-predicate? char-equal? number-predicate?
		      0 s1 (string-length s1) s2 (string-length s2)))

(define (false? . dummy) #f)

'(define (string=? s1 s2) (string-compare false? char=? = s1 s2))
(define (string<? s1 s2) (string-compare char<? false? < s1 s2))
(define (string>? s1 s2) (string-compare char>? false? > s1 s2))
(define (string<=? s1 s2) (string-compare char<? char=? <= s1 s2))
(define (string>=? s1 s2) (string-compare char>? char=? >= s1 s2))
(define (string-ci=? s1 s2) (string-compare false? char-ci=? = s1 s2))
(define (string-ci<? s1 s2) (string-compare char-ci<? false? < s1 s2))
(define (string-ci>? s1 s2) (string-compare char-ci>? false? > s1 s2))
(define (string-ci<=? s1 s2) (string-compare char-ci<? char-ci=? <= s1 s2))
(define (string-ci>=? s1 s2) (string-compare char-ci>? char-ci=? >= s1 s2))

(define (string->list-aux s tail start end)
  (if (>= start end)
      tail
      (string->list-aux s (cons (string-ref s (- end 1))
				tail)
			start (- end 1))))

(define (string->list s)
  (string->list-aux s () 0 (string-length s)))

(define (string-copy s)
  (string-append s))

"Port handling"

(define (eof-object? obj) (eq? (type obj) (quote eof)))
(define (call-with-input-file name proc)
  (let* ((port (open-input-file name))
	 (result (proc port)))
    (close-input-port port)
    result))
(define (call-with-output-file name proc)
  (let* ((port (open-output-file name))
	 (result (proc port)))
    (close-output-port port)
    result))
"The `lazy' versions do not close the port; this will be done"
"automagically when no reference to this port exists any more."
(define (lazy-call-with-input-file name proc)
  (proc (open-input-file name)))
(define (lazy-call-with-output-file name proc)
  (proc (open-output-file name)))

"The `load' lambda will load a scheme file expression by expression,"
"evaluate each and return the last value."
(define (load-aux-aux port last)
  (let ((datum (eval (read port) the-global-frame)))
    (if (not (eof-object? datum))
	(load-aux-aux port datum)
	last)))
(define (load-aux port)
  (load-aux-aux port #f))
(define (load name)
  (call-with-input-file name load-aux))

"The `load-text' lambda will load a text file line by line."
(define (load-text-aux port l)
  (let ((line (read-line port)))
    (if (eof-object? line)
	l
	(load-text-aux port (append l (list line))))))
(define (load-text port)
  (load-text-aux port ()))
"The `lazy-load-text' lambda just returns a promise to deliver the following lines;"
"when forced, a pair of a line and a new promise is given."
'(define (lazy-load-text port)
  (let ((line (read-line port)))
    (if (eof-object? line)
	'()
	(cons line (delay (lazy-load-text port))))))
(define (lazy-load-text port)
  (make-port-read-promise port))

(define (newline . optional-port)
  (write-char #\newline . optional-port))

"Directory and path system."

(define (relative-path? name)
  (or (zero? (string-length name))
      (not (char=? (string-ref name 0) #\/))))
(define (slashify-path name)
  (if (or (zero? (string-length name))
	  (char=? (string-ref name (- (string-length name) 1)) #\/))
      name
      (string-append name "/")))
(define (file-directory-list-aux dir-list last-pos cur-pos str)
  (let ((slash (string-member str cur-pos #\/)))
    (if slash
	(file-directory-list-aux (append dir-list
					  (list (substring str last-pos slash)))
				 (+ 1 slash)
				 (+ 1 slash)
				 str)
	(append dir-list (list (substring str last-pos (string-length str)))))))
(define (file-directory-list name)
  (file-directory-list-aux () 0 0 name))
(define (condense-aux rev-result-list dotdot-count rev-dir-list)
  (cond ((null? rev-dir-list)
	 (if (zero? dotdot-count) rev-result-list #f))
	((zero? (string-length (car rev-dir-list)))
	 (if (null? (cdr rev-dir-list))
	     (if (zero? dotdot-count) (cons "" rev-result-list) #f)
	     (condense-aux rev-result-list dotdot-count (cdr rev-dir-list))))
	((string=? (car rev-dir-list) ".")
	 (condense-aux rev-result-list dotdot-count (cdr rev-dir-list)))
	((string=? (car rev-dir-list) "..")
	 (condense-aux rev-result-list (+ 1 dotdot-count) (cdr rev-dir-list)))
	((zero? dotdot-count)
	 (condense-aux (cons (car rev-dir-list) rev-result-list)
		       dotdot-count 
		       (cdr rev-dir-list)))
	(else (condense-aux rev-result-list (- dotdot-count 1) (cdr rev-dir-list)))))
(define (condense-directory-list dir-list)
  (condense-aux () 0 (reverse dir-list)))
(define (directory-list->name dir-list)
  (cond ((null? dir-list) "")
	((null? (cdr dir-list))
	 (if (zero? (string-length (car dir-list)))
	     "/"
	     (car dir-list)))
	(else (string-append (car dir-list) "/"
			     (directory-list->name (cdr dir-list))))))
(define (condense-path name)
  (directory-list->name (condense-directory-list (file-directory-list name))))
(define (file-name name)
  (rac (file-directory-list name)))
(define (file-extension name)
  (let* ((fname (file-name name))
	 (result (string-member fname 0 #\.)))
    (if result
	(substring fname (+ result 1) (string-length fname))
	"")))
(define (file-base name)
  (let ((fname (file-name name))
	(result (string-member fname 0 #\.)))
    (if result
	(substring fname 0 result)
	fname)))

"File/external variable system"

(define file-type-alist
  '(("scm" . scheme)
    ("scd" . scheme-data)
    ("txt" . text) ("cc" . text) ("h" . text)
    ("" . executable)))
(define (file-type name)
  (if (file-directory? name)
      'directory
      (if (file-executable? name)
	  'executable
	  (let ((result (assoc (file-extension name) file-type-alist)))
	    (if result (cdr result) 'unknown)))))

"Add directories to the search-directory-list only if they exist."
(define search-directory-list '())
(define (add-to-search-directory-list dir)
  (if (file-exists? dir)
      (set! search-directory-list (cons dir search-directory-list))))
(add-to-search-directory-list "/bin")
(add-to-search-directory-list "/usr/bin")
(define search-extension-list '("scm" "cof"))

"If task-force is not good (we hope that the (hopefully existent)"
"other thread has managed to compute that!), replace it by a normal"
"force.  That way, at least bounded input can be passed to an"
"external procedure."  
(define original-task-force task-force)
(if (not good-task-force) (set! task-force force))

"Output in some neat line-oriented external representation"
"This code here is not good enough for structured input/output of"
"external output; we just put out some of the contents in a very"
"raw fashion."
"Note that we make use of a non-standard feature of Esso Scheme:"
"Primitives like `write' return #f if the port is no longer writable."
"This way, if some low-level support makes the port non-writable, this"
"function will stop forcing the argument list and will return."
(define (write-to-executable-aux port l)
  (if (handle-port-read-promise l port)
      #t
      (case (type l)
	((number symbol boolean)
	 (and
	  (write l port)
	  (newline port)))
	((pair)
	 (and
	  (write-to-executable-aux port (car l))
	  (write-to-executable-aux port (cdr l))))
	((string)
	 (write-line l port))
	(else #t))))

(define wp 0)

(define (write-to-executable port l)
  (set! wp (+ 1 wp))
  (write-to-executable-aux port l)
  (close-output-port port)
  (set! wp (- wp 1))
)

"Input from some neat line-oriented external representation"
"Just read in as a list of lines; until we have a better idea"
"how to represent the Lisp structures in a semi-text-oriented"
"fashion."
(define (read-from-executable port)
  (lazy-load-text port))

"The set-car! detaches the parent thread from the in-port."
"That makes it possible for the child thread to get rid of this"
"port, and closing it thereby."
(define (load-executable name)
  (lambda arglist
    (let ((ports (run-executable name)))
      (if ports
	  (begin
	    (let ((inport (car ports)))
	      (task-force (delay (write-to-executable inport arglist))))
	    (set-car! ports #f)
	    (read-from-executable (cdr ports)))
	  #f))))

(define (load-file path)
  (case (file-type path)
    ((directory) (load-directory path))
    ((scheme) (load path))
    ((executable) (load-executable path))
    ((scheme-data) (call-with-input-file path read))
    ((text) (lazy-call-with-input-file path lazy-load-text))
    (else ())))

"Pretty printing"
"Not very sensible, yet."

(define (pretty-pair-aux x port)
  (cond ((null? x))
	((not (pair? x))
	 (begin
	   (write x port)
	   (newline port)))
	((string? (car x))
	 (begin
	   (write-line (car x) port)
	   (pretty-pair-aux (cdr x) port)))
	(else
	 (begin
	   (write (car x) port)
	   (newline port)
	   (pretty-pair-aux (cdr x) port)))))
(define (pretty-aux x port)
  (cond ((pair? x)
	 (pretty-pair-aux x port))
	(else (begin
		(write x port)
		(newline port))))
  #t)

"Help for novices"
(define help
  (string-append
   "Esso is an Extensible Shell with Scheme Orientation. You run commands\n"
   "by typing, for instance, (cd \"dir\") or (+ 7 4). You can read directories\n"
   "by typing the directory name. Type (procedures) for a list of defined\n"
   "procedures (that is, commands). Type (info) to learn about info."))
(define ? help) 

(define (foo) 
  "Install other system libraries"
  (load "feature.scm")
  "Install key features"
  (require 'info)
  (require 'frame)
  "Install info files"
  (set! info-file-list
	(remove-if ./ (lambda (s) (not (string=? (file-extension s) "sci")))))
)

"Require requires features feature."
(define feature-list #f)
(define (require feature)
  (if (not feature-list)
      (load "feature.scm"))
  (if (eq? bad-require-aux require)
      "Sorry, could not load `feature.scm'"
      (require feature)))
(define bad-require-aux require)

"Real tab defined in feature 'frame"
(define (tab x)
  (require 'frame)
  (if (eq? tab bad-tab-aux)
      "Sorry, could not load the tab feature"
      (tab x)))
(define bad-tab-aux tab)  

(define (info . x)
  (require 'info)
  "Install info files"
  (set! info-file-list
	(remove-if ./ (lambda (s) (not (string=? (file-extension s) "sci")))))
  (if (eq? tab bad-info-aux)
      "Sorry, could not load the info feature"
      (begin
	(set! info-file-list
	      (remove-if ./ (lambda (s) (not (string=? (file-extension s) "sci")))))
	(apply info x))))
(define bad-info-aux info)

"Local definitions"
(define terminal-open-expression
  '(begin
     "Lambdas using the-current-input/output-port"
     (define (current-input-port) the-current-input-port)
     (define (current-output-port) the-current-output-port)
     (define (with-input-from-file name thunk)
       (let ((save-input (current-input-port)))
	 (begin
	   (set! the-current-input-port (open-input-file name))
	   (let ((result (thunk)))
	     (begin
	       (close-input-port the-current-input-port)
	       (set! the-current-input-port save-input)
	       result)))))
     (define (with-output-to-file name thunk)
       (let ((save-output (current-output-port)))
	 (begin
	   (set! the-current-output-port (open-output-file name))
	   (let ((result (thunk)))
	     (begin
	       (close-output-port the-current-output-port)
	       (set! the-current-output-port save-output)
	       result)))))
     "The `interactive' lambda returns a promise to deliver lines"
     "from the current input port."
     (define (interactive)
       (delay (lazy-load-text (current-input-port))))
     "Directory and path system."
     (define current-directory "/")
     (define (expand-path name)
       (if (relative-path? name)
	   (condense-path (string-append (slashify-path current-directory) name))
	   (condense-path name)))
     (define (search-path-in-directory-list name directory-list)
       (if (null? directory-list)
	   #f
	   (let ((p (expand-path (string-append (slashify-path (car directory-list))
						name))))
	     (if (file-exists? p)
		 p
		 (search-path-in-directory-list name (cdr directory-list)))))) 
     (define (search-path-in-extension-list name extension-list)
       (if (null? extension-list) 
	   #f
	   (let* ((extname
		   (if (string=? "" (car extension-list))
		       name
		       (string-append name
				      (string-append "."
						     (car extension-list)))))
		  (path (if (relative-path? extname)
			    (search-path-in-directory-list extname (cons "" search-directory-list))
			    (search-path-in-directory-list extname '("")))))
	     (if path
		 path
		 (search-path-in-extension-list name (cdr extension-list))))))
     (define (search-path name)
       (search-path-in-extension-list name (cons "" search-extension-list)))
     (define (eval-external-symbol symbol)
       (let ((path (search-path (symbol->string symbol))))
	 (if path
	     (list (load-file path))
	     ())))
     (define (pretty x) (pretty-aux x (current-output-port)))
     "Esso specials"
     (define host-name "localhost")
     (define (terminal-name)
       (if (number? terminal-number)
	   (make-string 1 (integer->char (+ 49 terminal-number)))
	   terminal-number))
     (define (esso-input-prompt)
       (string-append "esso@"
		      host-name
		      "(" (terminal-name) ")"
		      ":"
		      current-directory
		      " > "))
     (define (esso-output-prompt) "====> ")
     (define (update-cwd))
     (define (chdir directory-name)
       (let ((new-cur-dir (expand-path directory-name)))
	 (if (file-directory? new-cur-dir)
	     (begin
	       (set! current-directory new-cur-dir)
	       (update-cwd)
	       current-directory)
	     #f)))
     (define cd chdir)
     (define (pwd) current-directory)
     (define (cat x) (lazy-call-with-input-file x lazy-load-text))
     (define (exit)
       (set! esso-running #f)
       "Have a nice day.")
     (define quit exit)
     (define ia interactive)
     "Sync with OS's ideas of some variables (if any)"
     (if (file-executable? "/bin/pwd")
	 (set! current-directory (car (/bin/pwd))))
     (if (file-executable? "/bin/hostname")
	 (set! host-name (car (/bin/hostname))))
     )
  )     

"Make local stuff available here"
(define terminal-number "console")
(eval terminal-open-expression)

"This is the Esso shell; loaded EssoLib."
