"Esso: Scheme-level profiling"
"Written by Matthias Koeppe <mkoeppe@csmd.cs.uni-magdeburg.de>"

"The list of profiled functions and their profiles."
(define profile-list '())

"Prof-data looks like this:"
'(procedure-symbol original-procedure number-of-calls cumulative-time)

"Called at the end of a profiled procedure"
(define (profile-e prof-data start-time result) 
  (set-car! (cddr prof-data) (+ 1 (caddr prof-data)))
  (set-car! (cdddr prof-data) (+ (- (system-time) start-time) (cadddr prof-data)))
  result)

(define profile-aux #f)
(define (profile procedure-symbol)
  (let ((procedure (eval procedure-symbol)))
    (if (procedure? procedure)
	(let ((prof-data (cons procedure-symbol
			       (cons procedure
				     (append '(0 0) '())))))
	  (begin
	    (set! profile-aux
		  (lambda arg-list
		    (profile-e prof-data (system-time) (procedure . arg-list))))
	    (eval `(set! ,procedure-symbol profile-aux))
	    (set! profile-list (cons prof-data profile-list))
	    #t))
	#f)))

(define (unprofile procedure-symbol)
  (let ((prof-data (assq procedure-symbol profile-list)))
    (if prof-data
	(begin
	  (set! profile-aux (cadr prof-data))
	  (eval `(set! ,procedure-symbol profile-aux))
	  #t)
	#f)))

'(define (profile-lambdas)
  (map (lambda (symbol)
	 (begin 
	   (if (and (not (assq symbol profile-list))
		    (eq? (type (eval symbol)) (quote lambda)))
	       (profile symbol))
	   #t))
       (bindings)))

(define (clear-profiles-aux l)
  (if (not (null? l))
      (begin
	(set-car! (cddar l) 0)
	(set-car! (cdddar l) 0)
	(clear-profiles-aux (cdr l)))))
(define (clear-profiles) (clear-profiles-aux profile-list) #t)
  
(define (print-profile-aux prof-data)
  (list (car prof-data) 
	(caddr prof-data)
	'cumulative
	(cadddr prof-data)
	'average
	(if (zero? (caddr prof-data))
	    0
	    (quotient (cadddr prof-data) (caddr prof-data)))))
(define (print-profiles)
  (pretty (map print-profile-aux profile-list)))

(define profiled-lambdas '(append assoc-aux call-with-input-file
  call-with-output-file cd chdir condense-aux condense-directory-list
  condense-path directory-list->name equal? eval-case eval-cond
  eval-delay eval-external-symbol eval-let eval-let* expand-path
  file-base file-directory-list file-directory-list-aux file-extension
  file-name file-type get-info good-procedure-name?
  index-info-file-aux info info-aux info-from-info-file-aux
  interactive lazy-call-with-input-file lazy-call-with-output-file
  lazy-load-text length let*->lambda let->lambda list->string
  list-head list-ref list-tail load load-executable load-file load-text
  map member-aux not-good-procedure-symbol? pretty procedure-symbol?
  procedures rac read-from-executable relative-path? remove-if reverse
  search-path search-path-in-directory-list
  search-path-in-extension-list slashify-path string string->list
  string-compare task-force variable-symbol? with-input-from-file
  with-output-to-file write-to-executable))

(define (profile-lambdas)
  (map profile profiled-lambdas)
  #t)

(define (unprofile-all)
  (map unprofile (map car profile-list))
  #t)

(set! info-file-list (cons "profile.sci" info-file-list))

"profile"