;;;
;;; Hilfsmittel
;;;

(define (lambda-and . x) 
  ;; Ein `and', das keine `special form' ist.
  (or (null? x)
      (and (car x) (apply lambda-and (cdr x)))))

(define (lambda-or . x)
  ;; Ein `or', das keine `special form' ist.
  (and (not (null? x))
       (or (car x) (apply lambda-or (cdr x)))))

(define (remove-if predicate? x)
  ;; Entfernt alle Elemente der Liste, f"ur die das Pr"adikat
  ;; erf"ullt ist.
  (cond ((null? x) x)
	((predicate? (car x)) (remove-if predicate? (cdr x)))
	(else (cons (car x) (remove-if predicate? (cdr x))))))

(define (first-that pred? l)
  ;; Ermittelt eine Teilliste, deren erstes
  ;; Element das Pr"adikat erf"ullt.
  (cond ((null? l) l)
	((pred? (car l)) l)
	(else (first-that pred? (cdr l)))))

(define (make-set elem-list)
  ;; L"a"st jedes Element nur einmal auftreten.
  (cond ((null? elem-list) elem-list)
	(else 
	 (cons (car elem-list)
	       (make-set 
		(remove-if (lambda (x) 
			     (equal? x (car elem-list)))
			   (cdr elem-list)))))))

(define (set-cons elem set)
  ;; F"ugt das Element nur hinzu, wenn es nicht schon da ist.
  (if (member elem set)
      set
      (cons elem set)))

(define (set-append . set-list)
  ;; F"ugt Elemente nur hinzu, wenn sie nicht schon da sind.
  (set-append-aux set-list '()))

(define (set-append-aux set-list result)
  (cond ((null? set-list) result)
	((null? (car set-list)) 
	 (set-append-aux (cdr set-list) result))
	((member (caar set-list) result)
	 (set-append-aux (cons (cdar set-list) (cdr set-list)) 
			 result))
	(else (set-append-aux (cons (cdar set-list) (cdr set-list)) 
			      (cons (caar set-list) result)))))

(define (set-minus set1 set2)
  ;; Entfernt alle Elemente der zweiten Menge aus der ersten Menge
  ;; und gibt das Ergebnis zur"uck.
  (remove-if (lambda (x) (member x set2)) set1))

(define (atom->string atom)
  (cond ((number? atom) (number->string atom))
	((symbol? atom) (symbol->string atom))
	((string? atom) atom)
	((char? atom) (string atom))))

;;;
;;; Interne Repr"asentation aussagenlogischer Formeln
;;;

;; Konstruktoren

;; Es werden keine Vereinfachungen gemacht, damit auch beliebig
;; unvereinfachte Ausdr"ucke repr"asentiert werden k"onnen.

(define (make-negation expr) (list 'not expr))
(define (make-conjunction . exprs) (cons 'and exprs))
(define (make-disjunction . exprs) (cons 'or exprs))
(define (make-implication expra exprb) (list 'imply expra exprb))
(define (make-equivalence expra exprb) (list 'equiv expra exprb))
(define (make-xor expra exprb) (list 'xor expra exprb))

;; Pr"adikate

(define (negation? expr) (and (pair? expr)
			      (eq? 'not (car expr))))
(define (conjunction? expr) (and (pair? expr)
				 (eq? 'and (car expr))))
(define (disjunction? expr) (and (pair? expr)
				 (eq? 'or (car expr))))
(define (implication? expr) (and (pair? expr)
				 (eq? 'imply (car expr))))
(define (equivalence? expr) (and (pair? expr)
				 (eq? 'equiv (car expr))))
(define (xor? expr) (and (pair? expr)
			 (eq? 'xor (car expr))))

;; Diese Definition versagt, sobald wir 
;; pr"adikatenlogische Ausdr"ucke als aussagenlogische
;; Atome verwenden wollen.
;(define logical-atom? atom?)
;; Wir verwenden deshalb diese Definition:
(define (logical-atom? expr) 
  (not (or (negation? expr)
	   (conjunction? expr)
	   (disjunction? expr)
	   (implication? expr)
	   (equivalence? expr)
	   (xor? expr))))

;; Selektoren f"ur die Teilausdr"ucke von
;; `implication', `equivalence', `xor'

(define left-subexpr cadr)
(define right-subexpr caddr)

;; Selektor f"ur die Liste der in `conjunction' oder `disjunction'
;; gebundenen Ausdr"ucke

(define subexpr-list cdr)

;; Selektor f"ur den Teilausdruck einer `negation'

(define subexpr cadr)

;;;
;;; Hilfsmittel f"ur externe Repr"asentationen
;;;

;; Entartete (null- oder einstellige) Konjunktionen 
;; und Disjunktionen m"us"-sen anders geklammert werden
;; als gew"ohnliche. Dazu verwenden wir diese Pr"adikate:

(define (multiop? expr)
  (or (conjunction? expr)
      (disjunction? expr)))

(define (void-multiop? expr)
  (and (multiop? expr)
       (null? (subexpr-list expr))))

(define (unary-multiop? expr)
  (and (multiop? expr)
       (pair? (subexpr-list expr))
       (null? (cdr (subexpr-list expr)))))

(define (operator-priority-logic expr)
  ;; Liefert die Priorit"at (St"arke der Bindung) des
  ;; Verkn"upfungssymboles in dem Ausdruck.
  (cond ((negation? expr) 10)
	((void-multiop? expr) 11)
	((unary-multiop? expr) 10)
	;; (9 ist reserviert f"ur Quantoren)
	((conjunction? expr) 8)
	((disjunction? expr) 6)
	((implication? expr) 4)
	((equivalence? expr) 4)
	((xor? expr) 4)
	(else 12)))

(define operator-priority operator-priority-logic)

;;; 
;;; Externe Repr"asentation: \TeX{}Form
;;;

(define (texform expr)
  ;; Liefert die \TeX{}-Form eines Ausdrucks.
  ;; Klammerung erfolgt, wenn die Priorit"at dies
  ;; erfordert.
  (let ((expr-priority (operator-priority expr)))
    (cond ((negation? expr) 
	   (string-append "\\neg " 
			  (texform-aux-open (subexpr expr) 
					    expr-priority)))
	  ((conjunction? expr) 
	   (texform-multiop "1" "\\wedge " expr expr-priority))
	  ((disjunction? expr) 
	   (texform-multiop "0" "\\vee " expr expr-priority))
	  ((implication? expr) 
	   (texform-binop "\\Rightarrow " expr expr-priority))
	  ((equivalence? expr)
	   (texform-binop "\\iff " expr expr-priority))
	  ((xor? expr)
	   (texform-binop "\\oplus " expr expr-priority))
	  (else (texform-atom expr)))))
	 
(define (texform-aux expr priority)
  (if (>= priority (operator-priority expr))
      (string-append "(" (texform expr) ")")
      (texform expr)))

(define (texform-aux-open expr priority)
  (if (> priority (operator-priority expr))
      (string-append "(" (texform expr) ")")
      (texform expr)))

(define (texform-atom expr)
  (string-append "\\mathit{"
		 (atom->string expr)
		 "}"))
		
(define (texform-binop opname expr priority)
  (string-append (texform-aux (left-subexpr expr) priority)
		 opname
		 (texform-aux (right-subexpr expr) priority)))
      
(define (texform-multiop neutralname opname expr priority)
  (cond ((void-multiop? expr) neutralname)
	((unary-multiop? expr) 
	 (texform-aux-open (car (subexpr-list expr))
			   priority))
	(else (string-append 
	       (texform-aux (car (subexpr-list expr))
			    priority)
	       (texform-list opname
			     (cdr (subexpr-list expr))
			     priority)))))
			  
(define (texform-list opname oplist priority)
  (if (null? oplist)
      ""
      (string-append opname
		     (texform-aux (car oplist) priority)
		     (texform-list opname (cdr oplist) priority))))

;;; 
;;; Externe Repr"asentation: TextForm
;;;

(define (textform expr)
  ;; Liefert die Text-Form eines Ausdrucks.
  ;; Klammerung erfolgt, wenn die Priorit"at dies
  ;; erfordert.
  (let ((expr-priority (operator-priority expr)))
    (cond ((negation? expr) 
	   (string-append "!" 
			  (textform-aux-open (subexpr expr) 
					     expr-priority)))
	  ((conjunction? expr) 
	   (textform-multiop "1" " && " expr expr-priority))
	  ((disjunction? expr) 
	   (textform-multiop "0" " || " expr expr-priority))
	  ((implication? expr) 
	   (textform-binop " => " expr expr-priority))
	  ((equivalence? expr)
	   (textform-binop " <=> " expr expr-priority))
	  ((xor? expr)
	   (textform-binop " ^^ " expr expr-priority))
	  (else (textform-atom expr)))))
	 
(define (textform-aux expr priority)
  (if (>= priority (operator-priority expr))
      (string-append "(" (textform expr) ")")
      (textform expr)))

(define (textform-aux-open expr priority)
  (if (> priority (operator-priority expr))
      (string-append "(" (textform expr) ")")
      (textform expr)))

(define (textform-atom expr)
  (atom->string expr))
		
(define (textform-binop opname expr priority)
  (string-append (textform-aux (left-subexpr expr) priority)
		 opname
		 (textform-aux (right-subexpr expr) priority)))

(define (textform-multiop neutralname opname expr priority)
  (cond ((void-multiop? expr) neutralname)
	((unary-multiop? expr) 
	 (textform-aux-open (car (subexpr-list expr))
			    priority))
	(else (string-append 
	       (textform-aux (car (subexpr-list expr))
			     priority)
	       (textform-list opname
			      (cdr (subexpr-list expr))
			      priority)))))

(define (textform-list opname oplist priority)
  (if (null? oplist)
      ""
      (string-append opname
		     (textform-aux (car oplist) priority)
		     (textform-list opname (cdr oplist) priority))))

;;; 
;;; Konjunktive Normalform
;;;

(define positive-literal? logical-atom?)

(define (negative-literal? expr) 
  (and (negation? expr) 
       (logical-atom? (subexpr expr))))

(define (contrary? literal1 literal2)
  (or (and (positive-literal? literal1) 
	   (negative-literal? literal2))
      (and (positive-literal? literal2)
	   (negative-literal? literal1))))

(define (literal? expr)
  (or (positive-literal? expr)
      (negative-literal? expr)))

(define (clause? expr) 
  ;; Ermittelt, ob eine Klausel vorliegt.
  ;; Auch ein einzelnes Literal wird als Klausel angesehen. 
  (or (literal? expr)
      (and (disjunction? expr)
	   (apply lambda-and (map literal? (subexpr-list expr))))))

(define (clause->list clause)
  ;; Liefert eine Liste aller Literale in der Klausel.
  (if (disjunction? clause)
      (subexpr-list clause)
      (list clause)))

(define (literal-atom literal) 
  ;; Liefert das Atom des Literals.
  (if (negation? literal) 
      (subexpr literal)
      literal))

(define (negate-literal literal)
  ;; Negiert das Literal.
  (if (negative-literal? literal)
      (literal-atom literal)
      (make-negation (literal-atom literal))))

(define (canonicalize expr) 
  ;; Liefert einen "aquivalenten Ausdruck, der nur Konjunktionen,
  ;; Disjunktionen und Negationen verwendet. Die anderen 
  ;; "aquivalenten Umformungen behandeln nur diese drei 
  ;; Verkn"upfungen.
  (cond ((logical-atom? expr) 
	 (canonicalize-atom expr))
	((negation? expr) 
	 (make-negation (canonicalize (subexpr expr))))
	((conjunction? expr) 
	 (apply make-conjunction 
		(map canonicalize (subexpr-list expr))))
	((disjunction? expr) 
	 (apply make-disjunction 
		(map canonicalize (subexpr-list expr))))
	((implication? expr) 
	 (make-disjunction 
	  (make-negation (canonicalize (left-subexpr expr)))
	  (canonicalize (right-subexpr expr))))
	((equivalence? expr) 
	 (let ((canonicalized-left 
		(canonicalize (left-subexpr expr)))
	       (canonicalized-right 
		(canonicalize (right-subexpr expr))))
	   (make-disjunction 
	    (make-conjunction canonicalized-left 
			      canonicalized-right)
	    (make-conjunction 
	     (make-negation canonicalized-left)
	     (make-negation canonicalized-right)))))
	((xor? expr) 
	 (let ((canonicalized-left 
		(canonicalize (left-subexpr expr)))
	       (canonicalized-right 
		(canonicalize (right-subexpr expr))))
	   (make-disjunction 
	    (make-conjunction canonicalized-left 
			      (make-negation canonicalized-right))
	    (make-conjunction (make-negation canonicalized-left)
			      canonicalized-right))))
	(else expr)))

(define (canonicalize-atom expr) expr)

(define (normalize-canonical expr) 
  ;; Bringt einen kanonischen Ausdruck in Pr"a-KNF, das hei"st,
  ;; der Ausdruck ist eine Konjunktion von Disjunktionen, aber
  ;; m"oglicherweise noch nicht abgeflacht und vereinfacht.
  (cond ((logical-atom? expr) expr)
	((negation? expr) (normalize-negation expr))
	((conjunction? expr) (normalize-conjunction expr))
	((disjunction? expr) (normalize-disjunction expr))
	(else expr)))

(define (normalize-negation expr)
  (cond ((logical-atom? (subexpr expr)) expr)
	((negation? (subexpr expr)) 
	 ;; Doppelte Verneinung entfernen
	 (normalize-canonical (subexpr (subexpr expr))))
	((conjunction? (subexpr expr)) 
	 ;; de Morgan
	 (normalize-canonical 
	  (apply make-disjunction 
		 (map make-negation 
		      (subexpr-list (subexpr expr))))))
	((disjunction? (subexpr expr))
	 ;; de Morgan
	 (normalize-canonical
	  (apply make-conjunction
		 (map make-negation
		      (subexpr-list (subexpr expr))))))
	(else expr)))

(define (normalize-disjunction expr)
  ;; Hier werden Disjunktionen von Konjunktionen 
  ;; mit dem Distributivgesetz "`ausmultipliziert"'.
  (if (null? (subexpr-list expr))
      expr
      (let ((normalized-left 
	     (normalize-canonical (car (subexpr-list expr)))))
	(cond ((null? (cdr (subexpr-list expr))) 
	       (make-disjunction normalized-left))
	      ((conjunction? normalized-left)
	       (let ((right (cdr (subexpr-list expr))))
		 (apply make-conjunction
			(map (lambda (ex) 
			       (normalize-disjunction 
				(apply make-disjunction 
				       (cons ex right))))
			     (subexpr-list normalized-left)))))
	      ((and (conjunction? (cadr (subexpr-list expr)))
		    (null? (cddr (subexpr-list expr))))
	       (apply make-conjunction
		      (map (lambda (ex)
			     (normalize-disjunction
			      (make-disjunction normalized-left 
						ex)))
			   (subexpr-list 
			    (cadr (subexpr-list expr))))))
	      (else (make-disjunction 
		     normalized-left
		     (normalize-disjunction 
		      (apply make-disjunction 
			     (cdr (subexpr-list expr))))))))))

(define (normalize-conjunction expr)
  (let ((normalized-subexpr-list
	 (map normalize-canonical (subexpr-list expr))))
    (apply make-conjunction normalized-subexpr-list)))

(define (flatten expr)
  ;; Flacht Pr"a-KNF-Ausdr"ucke ab.
  (cond ((conjunction? expr) 
	 (apply make-conjunction 
		(flatten-list conjunction? (subexpr-list expr))))
	((disjunction? expr)
	 (apply make-disjunction 
		(flatten-list disjunction? (subexpr-list expr))))
	((negation? expr)
	 (make-negation (flatten (subexpr expr))))
	(else expr)))

(define (flatten-list predicate? expr-list)
  (cond ((null? expr-list) expr-list)
	((predicate? (car expr-list)) 
	 (append (flatten-list predicate? 
			       (subexpr-list (car expr-list)))
		 (flatten-list predicate? 
			       (cdr expr-list))))
	(else (cons (flatten (car expr-list))
		    (flatten-list predicate? 
				  (cdr expr-list))))))

(define (remove-duplicates expr)
  ;; Entfernt aus allen Klauseln mehrfach auftretende Literale;
  ;; entfernt "uberfl"ussige (immer wahre) Klauseln.
  (if (conjunction? expr)
      (apply make-conjunction 
	     (remove-duplicates-clause-list (subexpr-list expr)))
      expr))

(define (remove-duplicates-clause-list clause-list)
  (cond ((null? clause-list) clause-list)
	((obsolete-literal-list? (clause->list (car clause-list)))
	 (remove-duplicates-clause-list (cdr clause-list)))
	(else (cons (apply make-disjunction 
			   (remove-duplicates-list 
			    (clause->list
			     (car clause-list))))
		    (remove-duplicates-clause-list 
		     (cdr clause-list))))))

(define (obsolete-literal-list? literal-list)
  ;; Ermittelt, ob die Liste komplement"are Literale enth"alt.
  ;; (Eine Disjunktion der Literale der Liste ist dann immer wahr.)
  (and (pair? literal-list)
       (or (member (negate-literal (car literal-list))
		 (cdr literal-list))
	   (obsolete-literal-list? (cdr literal-list)))))

(define (remove-duplicates-list literal-list)
  ;; Entfernt wiederholte Vorkommen von Literalen in der Liste.
  (cond ((null? literal-list) literal-list)
	((member (car literal-list) (cdr literal-list))
	 (remove-duplicates-list (cdr literal-list)))
	(else (cons (car literal-list) 
		    (remove-duplicates-list (cdr literal-list))))))

(define (normalize expr)
  ;; Bringt einen Ausdruck in die KNF.
  (remove-duplicates 
   (flatten (normalize-canonical (canonicalize expr)))))

;;;
;;; Resolution
;;;

(define (theorem? axioms hypothesis)
  ;; Ermittelt, ob die Hypothese ein Satz des Axiomensystems ist.
  (not 
   (not 
    (contradiction? 
     (normalize 
      (make-conjunction axioms (make-negation hypothesis)))))))

(define (contradiction? expr)
  ;; F"uhrt auf dem KNF-Ausdruck Resolution durch,
  ;; bis ein Widerspruch abgeleitet wurde oder sicher
  ;; ist, da"s kein Widerspruch abgeleitet werden kann.
  (resolvent-diagonal? (subexpr-list expr) '()))

(define (resolvent-diagonal? clauses-to-come clauses-done)
  ;; Geht iterativ alle Elemente der ersten Klauselmenge
  ;; durch und resolviert sie mit allen Elementen der zweiten
  ;; Klauselmenge; die Resolventen werden der ersten Klauselmenge
  ;; hinzugef"ugt, und das verwendete Element der ersten
  ;; Klauselmenge wird in die zweite Menge verschoben.
  ;; Diese Diagonalisierung entspricht einer Breitensuche,
  ;; da die gefundenen Resolventen an die erste Klauselmenge
  ;; \emph{angeh"angt} wird.
  (and (not (null? clauses-to-come))
       (let* ((res-list (resolvent-list (car clauses-to-come)
					clauses-done
					'()))
	      (new-clauses-done (set-cons (car clauses-to-come)
					  clauses-done))
	      (new-clauses-to-come 
	       (set-minus (set-append clauses-to-come
				      res-list)
			  new-clauses-done)))
	 ;; Ist die leere Disjunktion abgeleitet?
	 ;; Sonst mu"s weiter iteriert werden.
	 (or (found-answer? res-list)
	     (resolvent-diagonal? new-clauses-to-come
				  new-clauses-done)))))

(define (found-answer? res-list)
  ;; Ermittelt, ob die leere Disjunktion abgeleitet wurde.
  (member (make-disjunction) res-list))

(define (resolvent-list clause clause-list res-list)
  ;; Resolviert iterativ alle Elemente der Liste mit der Klausel,
  ;; sofern m"og"-lich. Das Ergebnis ist eine Liste aller
  ;; Resolventen.
  (cond ((null? clause-list) res-list)
	(else (resolvent-list 
	       clause
	       (cdr clause-list)
	       (set-append (resolve-all clause (car clause-list))
			   res-list)))))

(define (resolve-all clause1 clause2)
  ;; Ermittelt eine Liste aller Resolventen, die sich durch
  ;; Auswahl eines Paars komplement"arer Literale der angegebenen
  ;; Klauseln ergeben.
  (map (lambda(l) (apply make-disjunction l))
       (resolve-all-list (clause->list clause1) '()
			 (clause->list clause2) '())))

(define (resolve-all-list list1-to-come list1-done list2 res-list)
  (cond ((null? list1-to-come) res-list)
	((member (negate-literal (car list1-to-come)) list2)
	 (resolve-all-list (cdr list1-to-come) 
			   (cons (car list1-to-come) list1-done)
			   list2
			   (cons 
			    (join-literal-lists 
			     (append list1-to-come list1-done)
			     list2 
			     (car list1-to-come) 
			     (negate-literal (car list1-to-come)))
			    res-list)))
	(else 
	 (resolve-all-list (cdr list1-to-come)
			   (cons (car list1-to-come) list1-done)
			   list2
			   res-list))))

(define (join-literal-lists list1 list2 literal1 literal2)
  ;; Verbindet die Literal-Listen. Zuvor werden die 
  ;; angegebenen Literale jeweils aus den Listen entfernt.
  (set-append 
   (remove-if (lambda (x) (equal? x literal1)) list1)
   (remove-if (lambda (x) (equal? x literal2)) list2)))

(define (resolve clause1 clause2)
  ;; Ermittelt eine Resolvente der beiden Klauseln.
  ;; Wird nicht verwendet.
  (car (resolve-all clause1 clause2)))

(define (resolvable? clause1 clause2)
  ;; Ermittelt, ob die Klauseln resolvierbar sind.
  ;; Wird nicht verwendet.
  (not (null? (resolve-all clause1 clause2))))


