{
   *********************************************************************
   Version: 1998.06.09
   Copyright (C) 1997, 1998 Gertjan Schouten

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   *********************************************************************

   Modified by Matthias K"oppe <mkoeppe@cs.uni-magdeburg.de> ":
   Added support for 1-bpp and 4-bpp bitmaps.
}

unit bitmaps;

interface

uses
   objects, colors;

const
   //        redpos       greenpos     bluepos
   ptRGBl = ( 0 shl 8) + (8 shl 16) + (16 shl 24);
   ptBGRl = (16 shl 8) + (8 shl 16) + ( 0 shl 24);
   ptRGBw = ( 0 shl 8) + (5 shl 16) + (10 shl 24);
   ptBGRw = (10 shl 8) + (5 shl 16) + ( 0 shl 24);

   ptRGB8 = ptRGBl + 8; { 00000000 bbbbbbbb gggggggg rrrrrrrr }
   ptRGB6 = ptRGBl + 6; { 00000000 00bbbbbb 00gggggg 00rrrrrr ** VGA palet layout ** }
   ptRGB5 = ptRGBw + 5; {                   0bbbbbgg gggrrrrr }

   ptBGR8 = ptBGRl + 8; { 00000000 rrrrrrrr gggggggg bbbbbbbb }
   ptBGR6 = ptBGRl + 6; { 00000000 00rrrrrr 00gggggg 00bbbbbb }
   ptBGR5 = ptBGRw + 5; {                   0rrrrrgg gggbbbbb }

   // pixel formats

   pfBGR15 = ptBGRw + 2; { 0rrrrrgg gggbbbbb }
   pfBGR24 = ptBGRl + 3; { rrrrrrrr gggggggg bbbbbbbb }
   pfBGR32 = ptBGRl + 4; { 00000000 rrrrrrrr gggggggg bbbbbbbb }

type
   TPaletType = Longint;
   TEncodeColor = Function(rgbcolor: longint):longint;
   TPalette = array[0..255] of longint;
   PPalette = ^TPalette;

   PBitmap = ^TBitmap;

   TBitmap = object
   private
      FData:pointer;
      FWidth:word;
      FHeight:word;
      FSize:longint;
      FBytesPerPixel:word;
      FBitsPerPixel:word;
      FEncodeColor:TEncodeColor;
      FPalette:pointer;
      procedure putPixels1(Source, Dest: Pointer; Count: Longint);
      procedure putPixels4(Source, Dest: Pointer; Count: Longint);
      procedure putPixels8(Source, Dest: Pointer; Count: Longint);
      procedure putPixels15(Source, Dest: Pointer; Count: Longint);
      procedure putPixels24(Source, Dest: Pointer; Count: Longint);
      procedure putPixels32(Source, Dest: Pointer; Count: Longint);
   public
      constructor Create;
      destructor Destroy;virtual;
      procedure Empty;
      function LoadFromFile(Filename:string):boolean;
      function GetData:pointer;
      function GetPalette:PPalette;
      function GetWidth:longint;
      function GetHeight:longint;
      function GetSize:longint;
      function HasPalette:boolean;
      procedure ScaleTo(AWidth, AHeight: word);
      procedure SetWidth(Value: word);
      procedure SetHeight(Value: word);
      procedure SetBytesPerPixel(Value: word);
      function GetBytesPerPixel:word;
      procedure SetEncodeColor(ec: TEncodeColor);
      procedure SetPalette(APalette: Pointer; Count:Word; PaletType:TPaletType; RecordSize:Word);
      procedure PutPixels(X, Y: Integer; Count: Longint; Buffer: Pointer; PixelFormat:Longint);
      function Clone:PBitmap;
   end ;

   PBitmapIO = ^TBitmapIO;

   TBitmapIO = object
   protected
      FBitmap: PBitmap;
      FFrame: integer;
      FFileExt: string[4];
      FFrameCount: integer;
      FNext: PBitmapIO;
   public
      function LoadFromFile(Filename:string):boolean;
      function LoadFromStream(Stream:PStream):boolean;
      procedure SetFrame(Value:integer);
      procedure SetNext(Value:PBitmapIO);
      function GetFileExt:string;
      function GetBitmap:PBitmap;
      function GetFrameCount:integer;
      function GetNext:PBitmapIO;
      procedure SetBitmap(Value:PBitmap);
      constructor Create;
      function LoadHeader(Stream:PStream):boolean; virtual;
      function LoadImage(Stream:PStream):boolean; virtual;
      destructor Destroy; virtual;
   end ;

Function GetImageFormat(FileExt: string):PBitmapIO;
Procedure SetBytesPerPixelAddr(Value:pointer);

implementation

type
   pByte = ^Byte;
   pInteger = ^Integer;
   pWord = ^Word;
   pLongint = ^Longint;
   ByteArray = array[0..0] of byte;
   WordArray = array[0..0] of word;
   LongintArray = array[0..0] of Longint;

const
   BytesPerPixel_Addr : pword = nil;

function pixel15to24(color: word):longint; Assembler;
asm
   xorl    %eax,%eax
   movw    color,%ax
   shll    $6,%eax
   shrw    $6,%ax
   shll    $3,%eax
   shlb    $3,%ah
end ;

constructor TBitmap.create;
begin
empty;
end ;

destructor TBitmap.destroy;
begin
empty;
end ;

function TBitmap.LoadFromFile(Filename:string):boolean;
var i:integer; ImageFormat:PBitmapIO;
begin
result := false;
i := length(FileName);
while (FileName[i] <> '.') and (i > 0) do
   dec(i);
if (i = 0) then exit;
ImageFormat := getImageFormat(Copy(FileName, i, Length(FileName)));
if not(ImageFormat = nil) then begin
   ImageFormat^.setBitmap(@self);
   result := ImageFormat^.LoadFromFile(Filename);
   end ;
end ;

procedure TBitmap.empty;
begin
if (FData <> nil) and (FSize > 0) then
   Freemem(FData, FSize);
if (FPalette <> nil) then
   Freemem(FPalette, 256 * SizeOf(Longint));
FData := nil;
FPalette := nil;
FWidth := 0;
FHeight := 0;
FSize := 0;
If BytesPerPixel_Addr <> nil then
   setBytesPerPixel(BytesPerPixel_Addr^)
else setBytesPerPixel(0);
end ;

function TBitmap.getData:pointer;
begin
Result := FData;
end ;

function TBitmap.getPalette: PPalette;
begin
if FPalette = nil then Result := @DefaultPalette
else Result := FPalette;
end ;

function TBitmap.getWidth:longint;
begin
Result := FWidth;
end ;

function TBitmap.getHeight:longint;
begin
Result := FHeight;
end ;

function TBitmap.GetSize:Longint;
begin
Result := FSize;
end ;

function TBitmap.HasPalette:boolean;
begin
Result := FPalette <> nil;
end ;

procedure TBitmap.ScaleTo(AWidth, AHeight: word);
var
   xs, ys, yi, xi, x, y: longint;
   newimage: pointer;
   newsize:longint;
   esi, edi: pointer;
begin
if FData = nil then begin
   setWidth(AWidth);
   setHeight(AHeight);
   exit;
   end ;
xi := (fWidth shl 16) div aWidth;
yi := (fHeight shl 16) div aHeight;
NewSize := 4 + AWidth * AHeight * fBytesPerPixel;
GetMem(NewImage, NewSize);
PWord(NewImage)^ := AWidth;
PWord(NewImage + 2)^ := AHeight;
edi := NewImage + 4;
ys := 0;
for y := 0 to AHeight - 1 do begin
   esi := (fData + 4) + (ys shr 16) * (fWidth * fBytesPerPixel);
   xs := 0;
   for x := 0 to AWidth - 1 do begin
         case fBytesPerPixel of
            1: ByteArray(edi^)[x] := ByteArray(esi^)[xs shr 16];
            2: WordArray(edi^)[x] := WordArray(esi^)[xs shr 16];
            4: LongintArray(edi^)[x] := LongintArray(esi^)[xs shr 16];
         end ;
      xs := xs + xi;
      end ;
   edi := edi + (fBytesPerPixel * AWidth);
   ys := ys + yi;
   end ;
Freemem(FData, FSize);
FData := NewImage;
FWidth := AWidth;
FHeight := AHeight;
FSize := NewSize;
end ;

procedure TBitmap.setWidth(Value:word);
var
   newImage: pointer;
   y, newSize: longint;
begin
if Value = FWidth then exit;
if (FHeight = 0) then fWidth := Value
else if Value = 0 then begin
   if FData <> Nil then
      freemem(FData, FSize);
   FData := nil;
   FSize := 0;
   FWidth := Value;
   end
else begin
   newSize := 4 + Value * fHeight * fBytesPerPixel;
   GetMem(newImage, newSize);
   pword(newImage)^ := Value;
   pword(newImage+2)^ := FHeight;
   if FData <> nil then begin
      for y := 0 to fHeight-1 do begin
         move(pointer(FData + 4 + (FBytesPerPixel * FWidth * y))^,
              pointer(NewImage + 4 + (FBytesPerPixel * Value * y))^,
              (FBytesPerPixel * FWidth));
         end ;
      Freemem(FData, FSize);
      end ;
   FData := newImage;
   FSize := NewSize;
   FWidth := Value;
   end ;
end ;

procedure TBitmap.setHeight(Value:word);
var newImage: pointer; newSize: longint;
begin
if Value = FHeight then exit;
if (FWidth = 0) then FHeight := Value
else if Value = 0 then begin
   if FData <> Nil then
      freemem(FData, fSize);
   FData := nil;
   FSize := 0;
   FHeight := Value;
   end
else begin
   NewSize := 4 + fWidth * Value * fBytesPerPixel;
   Getmem(NewImage, NewSize);
   if FData <> nil then begin
      move(FData^, NewImage^, FSize);
      Freemem(FData, FSize);
      end;
   pword(NewImage)^ := FWidth;
   pword(NewImage + 2)^ := Value;
   FData := NewImage;
   FSize := NewSize;
   FHeight := Value;
   end ;
end ;

procedure TBitmap.setBytesPerPixel(Value: word);
begin
if (FBytesPerPixel <> Value) and (Value > 0) and (Value < 5) then begin
   FBytesPerPixel := Value;
   FBitsPerPixel := Value * 8;
      case FBytesPerPixel of
         1: FEncodeColor := @EncodeColor8;
         2: FEncodeColor := @EncodeColor15;
         4: FEncodeColor := @EncodeColor24;
      end ;
   if FData <> nil then
      Freemem(FData, FSize);
   FData := Nil;
   FSize := 0;
   if (FWidth <> 0) and (FHeight <> 0) then begin
      FSize := 4 + FWidth * FHeight * FBytesPerPixel;
      Getmem(FData, FSize);
      pword(FData)^ := FWidth;
      pword(FData + 2)^ := FHeight;
      end ;
   end ;
end ;

procedure TBitmap.SetEncodeColor(ec: TEncodeColor);
begin
  FEncodeColor := ec
end;

function TBitmap.getBytesPerPixel:word;
begin
result := FBytesPerPixel;
end ;

procedure TBitmap.setPalette(APalette: Pointer; Count: Word; PaletType: TPaletType; RecordSize: Word);
var
   i: longint;
   r, g, b, m: byte;
   rshr, gshr, bshr: word;
begin
if (FPalette = nil) then
   Getmem(FPalette, 256 * SizeOf(Longint));
if (PaletType > 0) then begin
   m := 8 - byte(PaletType);
   rshr := (PaletType shr 8) - m;
   gshr := (PaletType shr 16) - m;
   bshr := (PaletType shr 24) - m;
   m := 255 shl m;
   for i := 0 to Count - 1 do begin
      r := (Longint(APalette^) shr rshr) and m;
      g := (Longint(APalette^) shr gshr) and m;
      b := (Longint(APalette^) shr bshr) and m;
      LongintArray(fPalette^)[i] := (r shl 16) or (g shl 8) or (b);
      APalette := APalette + RecordSize;
      end ;
   end
else move(APalette^, FPalette^, Count * SizeOf(Longint));
// if (FBytesPerPixel = 1) and (FOptions and bmDither = 0) then begin
//    for i := 0 to Count - 1 do
//       setRGBPalette(i, LongintArray(FPalette^)[i]);
//    end ;
end ;

// PUTPIXELS1: by Matthias K"oppe <mkoeppe@cs.uni-magdeburg.de> ":

Procedure TBitmap.putPixels1(Source, Dest: Pointer; Count: Longint);
var i,j,c,b: longint;
begin
  for i := 0 to count div 8 - 1 do begin
    b := byte(source^);
    for j := 0 to 7 do begin
      c := fEncodeColor(LongintArray(fPalette^)[(b shr 7) and 1]);
      move(c, dest^, fBytesPerPixel);
      dest := dest + fBytesPerPixel;
      b := b shl 1;
    end;
    Source := Source + 1;
  end;
  b := byte(source^);
  for j := 0 to count and 7 - 1 do begin
    c := fEncodeColor(LongintArray(fPalette^)[(b shr 7) and 1]);
    move(c, dest^, fBytesPerPixel);
    dest := dest + fBytesPerPixel;
    b := b shl 1;
  end;
end;

// PUTPIXELS4: by Matthias K"oppe <mkoeppe@cs.uni-magdeburg.de> ":

Procedure TBitmap.putPixels4(Source, Dest: Pointer; Count: Longint);
var i,c:longint;
begin
  for i := 0 to count div 2 - 1 do begin
    c := fEncodeColor(LongintArray(fPalette^)[byte(source^) shr 4]);
    move(c, dest^, fBytesPerPixel);
    dest := dest + fBytesPerPixel;
    c := fEncodeColor(LongintArray(fPalette^)[byte(source^) and 15]);
    move(c, dest^, fBytesPerPixel);
    dest := dest + fBytesPerPixel;
    Source := Source + 1;
  end ;
  if (count and 1 = 1) then begin
    c := fEncodeColor(LongintArray(fPalette^)[byte(source^) shr 4]);
    move(c, dest^, fBytesPerPixel);
  end ;
end ;

Procedure TBitmap.putPixels8(Source, Dest: Pointer; Count: Longint);
var i,c:longint;
begin
for i := 0 to count - 1 do begin
   c := FEncodeColor(LongintArray(FPalette^)[byte(source^)]);
   move(c, dest^, FBytesPerPixel);
   dest := dest + FBytesPerPixel;
   Source := Source + 1;
   end ;
end ;

Procedure TBitmap.putPixels15(Source, Dest: Pointer; Count: Longint);
var i,c:longint;
begin
for i := 0 to count-1 do begin
   c := FEncodeColor(Pixel15to24(word(Source^)));
   move(c, dest^, FBytesPerPixel);
   dest := dest + FBytesPerPixel;
   Source := Source + 2;
   end ;
end ;

Procedure TBitmap.putPixels24(Source, Dest: Pointer; Count: Longint);
var i,c:longint;
begin
for i := 0 to count-1 do begin
   c := FEncodeColor(longint(Source^));
   move(c, Dest^, fBytesPerPixel);
   Dest := Dest + fBytesPerPixel;
   Source := Source + 3;
   end ;
end ;

Procedure TBitmap.putPixels32(Source, Dest: Pointer; Count: Longint);
var i,c:longint;
begin
for i := 0 to count-1 do begin
   c := FEncodeColor(longint(source^));
   move(c, dest^, FBytesperpixel);
   dest := dest + FBytesperpixel;
   Source := Source + 4;
   end ;
end ;

// PUTPIXELS:
// some mods by Matthias K"oppe <mkoeppe@cs.uni-magdeburg.de>":

procedure TBitmap.PutPixels(X, Y: Integer; Count: Longint; Buffer: Pointer; PixelFormat: Longint);
var ofs:pointer;
begin
ofs := FData + 4 + (x + y * FWidth) * FBytesPerPixel;
if PixelFormat and 255 = FBitsPerPixel then
   move(Buffer^, ofs^, Count * FBytesPerPixel)
else begin
      case Pixelformat and 255 of
         1: putpixels1(Buffer, Ofs, Count);
         4: putpixels4(Buffer, Ofs, Count);
         8: putpixels8(Buffer, Ofs, Count);
         15: putpixels15(Buffer, Ofs, Count);
         24: putpixels24(Buffer, Ofs, Count);
         32: putpixels32(Buffer, Ofs, Count);
      end ;
   end ;
end ;

function TBitmap.Clone:PBitmap;
begin
result := new(PBitmap, create);
result^.setBytesPerPixel(FBytesPerPixel);
result^.setWidth(FWidth);
result^.setHeight(FHeight);
if (FSize = result^.getSize) then begin
   if (FSize > 0) then
      move(FData^, result^.getData^, FSize);
   if HasPalette then
      result^.setPalette(FPalette, 256, 0, 0);
   end
else begin
   dispose(Result, Destroy);
   result := nil;
   end ;
end ;

var
   ImageFormats:PBitmapIO;

constructor TBitmapIO.Create;
begin
FFrame := 0;
FBitmap := Nil;
FFileExt := '';
FFrameCount := 0;
FNext := ImageFormats;
ImageFormats := @self;
end ;

procedure TBitmapIO.setNext(Value:PBitmapIO);
begin
FNext := Value;
end ;

function TBitmapIO.getFileExt:string;
begin
result := fFileExt;
end ;

function TBitmapIO.getBitmap:PBitmap;
begin
result := fBitmap;
end ;

function TBitmapIO.getFrameCount:integer;
begin
result := fFrameCount;
end ;

function TBitmapIO.getNext:PBitmapIO;
begin
result := fNext;
end ;

function TBitmapIO.LoadFromFile(Filename:string):boolean;
var Stream:PStream;
begin
result := false;
Stream := new(PBufStream, Init(Filename, stOpenRead, $FFFF));
if Stream^.Status = stOk then
   result := LoadFromStream(Stream);
Dispose(Stream, Done);
end ;

function TBitmapIO.LoadFromStream(Stream:PStream):boolean;
begin
result := false;
if FBitmap <> nil then begin
   setFrame(0);
   LoadHeader(Stream);
   if FBitmap^.getData <> nil then
      result := LoadImage(Stream);
   end ;
end ;

procedure TBitmapIO.setFrame(Value:integer);
begin
fFrame := Value;
end ;

procedure TBitmapIO.setBitmap(Value:PBitmap);
begin
fBitmap := Value;
end ;

function TBitmapIO.LoadHeader(Stream:PStream):boolean;
begin
result := false;
end ;

function TBitmapIO.LoadImage(Stream:PStream):boolean;
begin
result := false;
end ;

destructor TBitmapIO.Destroy;
begin
end ;

Function GetImageFormat(FileExt: string):PBitmapIO;
var i:integer;
begin
for i := 1 to Length(FileExt) do
   FileExt[i] := UpCase(FileExt[i]);
result := ImageFormats;
while not(result = nil) do begin
   if not(pos(FileExt, result^.getFileExt) = 0) then break;
   result := result^.getNext;
   end ;
end ;

Procedure setBytesPerPixelAddr(Value:pointer);
begin
BytesPerPixel_Addr := Value;
end ;

begin
ImageFormats := Nil;
end .
