unit GvVirt;

{ Graphics Vision Virtual Desktop,
  Copr. 1996 Matthias K"oppe

  $Id: gvvirt.pas 1.3 1999/02/10 17:40:22 mkoeppe Exp $
}

{$ifndef FPK}
{$A+,B-,F+,G+,I-,O+,P-,Q-,R-,S-,T-,V+,X+}
{$endif}

interface

uses Objects, Drivers, Views, GvViews, GvBitmap, GvApp;

type
  PVirtualDragRect = ^TVirtualDragRect;
  TVirtualDragRect = object(TDragRect)
    Visible: TRect;
    constructor Init(var Bounds: TRect; var AVisible: TRect);
    procedure Draw; virtual;
    procedure SizeLimits(var Min, Max: TPoint); virtual;
  end;

  PVirtualDesktop = ^TVirtualDesktop;
  TVirtualDesktop = object(TNewDesktop)
    Virt: TRect;
    DragRect: PGView;
    constructor Init(var Bounds: TRect; Bitmap: string;
      FactorX, FactorY: Integer);
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure ScrollTo(P: TPoint);
    procedure SetFactor(FactorX, FactorY: Integer);
    procedure SizeLimits(var Min, Max: TPoint); virtual;
  private
    procedure ValidVirt(var R: TRect; var Delta: TPoint);
  end;

  PGroupImage = ^TGroupImage;
  TGroupImage = object(TGView)
    Group: PGGroup;
    constructor Init(var Bounds: TRect; AGroup: PGGroup);
    procedure ChangeBounds(var Bounds: TRect); virtual;
    procedure Draw; virtual;
    procedure GetGroupBounds(var Bounds: TRect); virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

  PDesktopImage = ^TDesktopImage;
  TDesktopImage = object(TGroupImage)
    LeftStyle: Word;
    RightStyle: Word;
    constructor Init(var Bounds: TRect);
    procedure Draw; virtual;
    procedure GetGroupBounds(var Bounds: TRect); virtual;
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

  PInteriorWindow = ^TInteriorWindow;
  TInteriorWindow = object(TWindow)
    Interior: PGView;
    constructor Init(var Bounds: TRect; ATitle: TTitleStr; ANumber: Integer);
    procedure InitInterior; virtual;
    procedure ChangeBounds(var Bounds: TRect); virtual;
  end;

  PDesktopImageWindow = ^TDesktopImageWindow;
  TDesktopImageWindow = object(TInteriorWindow)
    constructor Init(var Bounds: TRect; ATitle: TTitleStr);
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure InitInterior; virtual;
    procedure SizeLimits(var Min, Max: TPoint); virtual;
  end;

implementation

{$ifdef FPK}
uses Bgi, ExtGraph, GvFPK;
{$else}
{$ifdef Windows}
uses WinGr, ExtGraph;
{$else}
uses Gr, MetaGr, MyMouse, ExtGraph;
{$endif}
{$endif}

constructor TVirtualDragRect.Init(var Bounds: TRect; var AVisible: TRect);
begin
  inherited Init(Bounds);
  GrowMode := gfGrowHiX + gfGrowHiY;
  Visible := AVisible;
  Visible.Move(- Origin.X, - Origin.Y)
end;

procedure TVirtualDragRect.Draw;
var
  i, x, d: Integer;
begin
  SetWriteMode(XorPut);
  SetColor(15);
  Rectangle(0, 0, Size.X - 1, Size.Y - 1);
  Rectangle(1, 1, Size.X - 2, Size.Y - 2);
  Rectangle(2, 2, Size.X - 3, Size.Y - 3);
  { this produces errors since the dotted line is not clipping-conforming
  }
  (* SetLineStyle(DottedLn, 0, NormWidth); *)
  x := 0;
  d := (Visible.B.X - Visible.A.X) div 3;
  For i := 1 to Size.X div d do
  begin
    Inc(x, d);
    VertLine(x, 3, Size.Y - 4)
  end;
  x := 0;
  d := (Visible.B.Y - Visible.A.Y) div 3;
  For i := 1 to Size.Y div d do
  begin
    Inc(x, d);
    HoriLine(3, x, Size.X - 4)
  end;
  (* SetLineStyle(SolidLn, 0, NormWidth); *)
  SetWriteMode(NormalPut)
end;

procedure TVirtualDragRect.SizeLimits(var Min, Max: TPoint);
begin
  Min.X := 0; Min.Y := 0;
  Max.X := MaxInt;
  Max.Y := MaxInt
end;

{ ----------------------------------------------------------------------
}

constructor TVirtualDesktop.Init(var Bounds: TRect; Bitmap: string;
  FactorX, FactorY: Integer);
begin
  inherited Init(Bounds, Bitmap);
  SetFactor(FactorX, FactorY);
end;

procedure TVirtualDesktop.HandleEvent(var Event: TEvent);
var
  Delta: TPoint;
  R, DR: TRect;
  Speed: Integer;
begin
  inherited HandleEvent(Event);
  {$ifndef Windows}
  If MouseInstalled then
  If (Event.What = evMouseMove) or (Event.What = evTimer)
  then begin
    If GetShiftState and kbLeftShift <> 0 then Speed := 32 else
    if GetShiftState and kbCtrlShift <> 0 then Speed := 16 else
    Speed := 8;
    Delta.X := 0; Delta.Y := 0;
    If Size.X < Virt.B.X - Virt.A.X
    then begin
      If MouseWhere.X = 0 then Delta.X := Speed else
      if MouseWhere.X = SizeX - 1 then Delta.X := - Speed;
    end;
    If Size.Y < Virt.B.Y - Virt.A.Y
    then begin
      If MouseWhere.Y = 0 then Delta.Y := Speed else
      if MouseWhere.Y = SizeY - 1 then Delta.Y := - Speed;
    end;
    If (Delta.X <> 0) or (Delta.Y <> 0)
    then begin
      GetBounds(R);
      If DragRect = nil
      then begin
	DragRect := New(PVirtualDragRect, Init(Virt, R));
	GOwner^.InsertBefore(DragRect, @Self)
      end;
      DragRect^.GetBounds(DR);
      ValidVirt(DR, Delta);
      DragRect^.Locate(DR)
    end
    else begin
      If DragRect <> nil
      then begin
	DragRect^.GetBounds(R);
	Dispose(DragRect, Done);
	DragRect := nil;
	ScrollTo(R.A);
      end
    end
  end
  {$endif !Windows}
end;

procedure TVirtualDesktop.ScrollTo(P: TPoint);
var
  Delta: TPoint;

	procedure Relocate(P: PGView); {$ifndef FPK}far;{$endif}
	var
	  R: TRect;
	begin
	  P^.GetBounds(R);
	  R.Move(Delta.X, Delta.Y);
	  P^.SetBounds(R)
	end;

begin
  Delta.X := P.X - Virt.A.X;
  Delta.Y := P.Y - Virt.A.Y;
  ValidVirt(Virt, Delta);
  ForEach(@Relocate);
  ScrollGroup(@Self, Delta)
end;

procedure TVirtualDesktop.SetFactor(FactorX, FactorY: Integer);
var
  R: TRect;
begin
  GetBounds(Virt);
  Virt.Grow(((FactorX - 1) * (Virt.B.X - Virt.A.X)) div 2,
    ((FactorY - 1) * (Virt.B.Y - Virt.A.Y)) div 2);
  R.Copy(Virt);
  R.Move(- Origin.X, - Origin.Y);
  Background^.SetBounds(R); {omit checking}
  DrawView
end;

procedure TVirtualDesktop.SizeLimits(var Min, Max: TPoint);
begin
  Min.X := 0; Min.Y := 0;
  Max.X := MaxInt;
  Max.Y := MaxInt
end;

procedure TVirtualDesktop.ValidVirt(var R: TRect; var Delta: TPoint);
var
  Bounds: TRect;
begin
  GetBounds(Bounds);
  If Delta.X + R.A.X > Bounds.A.X then Delta.X := Bounds.A.X - R.A.X;
  If Delta.Y + R.A.Y > Bounds.A.Y then Delta.Y := Bounds.A.Y - R.A.Y;
  If Delta.X + R.B.X < Bounds.B.X then Delta.X := Bounds.B.X - R.B.X;
  If Delta.Y + R.B.Y < Bounds.B.Y then Delta.Y := Bounds.B.Y - R.B.Y;
  R.Move(Delta.X, Delta.Y);
end;

{ ----------------------------------------------------------------------
}

constructor TGroupImage.Init(var Bounds: TRect; AGroup: PGGroup);
begin
  inherited Init(Bounds);
  Options := Options or ofBufferOne;
  GrowMode := gfGrowHiX + gfGrowHiY;
  Group := AGroup;
end;

procedure TGroupImage.ChangeBounds(var Bounds: TRect);
{ Keep the ratio of the imaged group
}
var
  X, Y: Integer;
  GroupBounds: TRect;
begin
  GetGroupBounds(GroupBounds);
  X := LongDiv(LongMul(Bounds.B.Y - Bounds.A.Y,
    GroupBounds.B.X - GroupBounds.A.X),
    GroupBounds.B.Y - GroupBounds.A.Y);
  Y := LongDiv(LongMul(Bounds.B.X - Bounds.A.X,
    GroupBounds.B.Y - GroupBounds.A.Y),
    GroupBounds.B.X - GroupBounds.A.X);
  If Bounds.B.X < Bounds.A.X + X
  then Bounds.B.X := Bounds.A.X + X;
  If Bounds.B.Y < Bounds.A.X + Y
  then Bounds.B.Y := Bounds.A.X + Y;
  inherited ChangeBounds(Bounds)
end;

procedure TGroupImage.Draw;
var
  GroupBounds: TRect;
  P: PGView;
  R: TRect;
  Pt: TPoint;

	procedure Scale(var P: TPoint);
	begin
	  P.X := LongDiv(LongMul(P.X - GroupBounds.A.X, Size.X), GroupBounds.B.X - GroupBounds.A.X);
	  P.Y := LongDiv(LongMul(P.Y - GroupBounds.A.Y, Size.Y), GroupBounds.B.Y - GroupBounds.A.Y);
	end;

begin
  GetGroupBounds(GroupBounds);
  P := Group^.Last;
  while P <> nil do
  begin
    If P^.GetState(sfVisible)
    then begin
      P^.GetBounds(R);
      Pt := R.A;
      Scale(R.A); Scale(R.B);
      If not P^.Valid(cmHeyYou)
      then begin
	Inc(Pt.Y, 20);
	Scale(Pt);
	If P^.GetState(sfActive)
	then SetFillStyle(SolidFill, P^.GetColor(7))
	else SetFillStyle(SolidFill, P^.GetColor(6));
	Bar(R.A.X, R.A.Y, R.B.X - 1, Pt.Y - 1);
	SetFillStyle(SolidFill, P^.GetColor(1));
	Bar(R.A.X, Pt.Y + 1, R.B.X - 1, R.B.Y - 1);
	HoriLine(R.A.X, Pt.Y, R.B.X - 1)
      end else
      if not P^.GetState(sfTransparent)
      then begin
	SetFillStyle(SolidFill, P^.GetColor(1));
	RBar(R);
      end;
      SetColor(0);
      RRect(R);
    end;
    P := P^.PrevView
  end;
end;

procedure TGroupImage.GetGroupBounds(var Bounds: TRect);
begin
  Group^.GetExtent(Bounds)
end;

procedure TGroupImage.HandleEvent(var Event: TEvent);
const
  TitleBuf: TTitleStr = '';
var
  R, GroupBounds: TRect;
  Wh: TPoint;
  P: PGView;

	procedure BackScale(var P: TPoint);
	begin
	  P.X := LongDiv(LongMul(P.X, GroupBounds.B.X - GroupBounds.A.X), Size.X) + GroupBounds.A.X;
	  P.Y := LongDiv(LongMul(P.Y, GroupBounds.B.Y - GroupBounds.A.Y), Size.Y) + GroupBounds.A.Y;
	end;

begin
  If Event.What = evPositionalCtx
  then begin
    GetGroupBounds(GroupBounds);
    MakeLocal(Event.Where, Wh);
    BackScale(Wh);
    P := Group^.First;
    while P <> nil do
    begin
      If P^.GetState(sfVisible) and not P^.Valid(cmHeyYou)
      then begin
	P^.GetBounds(R);
	If R.Contains(Wh)
	then begin
	  TitleBuf := PWindow(P)^.GetTitle(80);
	  Event.What := evNothing;
	  Event.InfoPtr := @TitleBuf;
	  Exit
	end
      end;
      P := P^.NextView
    end
  end;
  inherited HandleEvent(Event);
  If Event.What = evTimer
  then DrawView;
end;

{ ----------------------------------------------------------------------
}

constructor TDesktopImage.Init(var Bounds: TRect);
begin
  inherited Init(Bounds, Desktop);
  LeftStyle := $3333;
  RightStyle := $3333;
end;

procedure TDesktopImage.Draw;
var
  R, GroupBounds: TRect;

	procedure Scale(var P: TPoint);
	begin
	  P.X := LongDiv(LongMul(P.X - GroupBounds.A.X, Size.X), GroupBounds.B.X - GroupBounds.A.X);
	  P.Y := LongDiv(LongMul(P.Y - GroupBounds.A.Y, Size.Y), GroupBounds.B.Y - GroupBounds.A.Y);
	end;

begin
  inherited Draw;
  GetGroupBounds(GroupBounds);
  Group^.GetExtent(R);
  Scale(R.A);
  Scale(R.B);
  SetWriteMode(XorPut);
  SetColor(White);
  SetLineStyle(UserBitLn, LeftStyle, NormWidth);
  HoriLine(R.A.X - 1, R.A.Y - 1, R.B.X);
  VertLine(R.A.X - 1, R.A.Y - 1, R.B.Y);
  SetLineStyle(UserBitLn, RightStyle, NormWidth);
  HoriLine(R.A.X - 1, R.B.Y, R.B.X);
  VertLine(R.B.X, R.A.Y - 1, R.B.Y);
  SetLineStyle(SolidLn, 0, NormWidth);
  SetWriteMode(NormalPut)
end;

procedure TDesktopImage.GetGroupBounds(var Bounds: TRect);
begin
  Bounds.Copy(PVirtualDesktop(Group)^.Virt);
  Bounds.Move(-Group^.Origin.X, -Group^.Origin.Y)
end;

procedure TDesktopImage.HandleEvent(var Event: TEvent);
var
  GroupBounds: TRect;
  Wh: TPoint;

	procedure BackScale(var P: TPoint);
	begin
	  P.X := LongDiv(LongMul(P.X, GroupBounds.B.X - GroupBounds.A.X), Size.X);
	  P.Y := LongDiv(LongMul(P.Y, GroupBounds.B.Y - GroupBounds.A.Y), Size.Y);
	end;

begin
  If (Event.What = evMouseDown) or (Event.What = evMouseUp)
  then begin
    GetGroupBounds(GroupBounds);
    MakeLocal(Event.Where, Wh);
    BackScale(Wh);
    Wh.X := Group^.Size.X div 2 - Wh.X;
    Wh.Y := Group^.Size.Y div 2 - Wh.Y;
    PVirtualDesktop(Group)^.ScrollTo(Wh);
  end;
  inherited HandleEvent(Event);
  If Event.What = evTimer
  then begin
    LeftStyle := (LeftStyle shl 1) or (LeftStyle shr 15);
    RightStyle := (RightStyle shr 1) or (RightStyle shl 15)
  end
end;

{ ----------------------------------------------------------------------
}

constructor TInteriorWindow.Init(var Bounds: TRect;
  ATitle: TTitleStr; ANumber: Integer);
begin
  inherited Init(Bounds, ATitle, ANumber);
  InitInterior;
  If Interior <> nil
  then begin
    Insert(Interior);
    Flags := Flags and not wfBackground
  end;
  ChangeBounds(Bounds)
end;

procedure TInteriorWindow.ChangeBounds(var Bounds: TRect);
var
  Delta: TPoint;
  IBounds, NBounds, SBounds: TRect;
begin
  If Interior <> nil
  then begin
    Delta.X := Bounds.B.X - Bounds.A.X - Size.X;
    Delta.Y := Bounds.B.Y - Bounds.A.Y - Size.Y;
    If (Delta.X <> 0) or (Delta.Y <> 0)
    then begin
      Lock;
      GetBounds(SBounds);
      SetBounds(Bounds);
      with Interior^ do
      begin
	CalcBounds(IBounds, Delta);
	NBounds.Copy(IBounds);
	ChangeBounds(IBounds);
	GetBounds(IBounds);
      end;
      SetBounds(SBounds);
      Dec(Bounds.B.X, NBounds.B.X - IBounds.B.X);
      Dec(Bounds.B.Y, NBounds.B.Y - IBounds.B.Y);
      inherited ChangeBounds(Bounds);
      Interior^.SetBounds(IBounds);
      Unlock;
      Exit;
    end
  end;
  inherited ChangeBounds(Bounds);
end;

procedure TInteriorWindow.InitInterior;
begin
end;

{ ----------------------------------------------------------------------
}

constructor TDesktopImageWindow.Init(var Bounds: TRect; ATitle: TTitleStr);
begin
  inherited Init(Bounds, ATitle, wnNoNumber);
  Options := Options and not (ofSelectable + ofTileable);
  Flags := Flags and not wfZoom;
end;

procedure TDesktopImageWindow.HandleEvent(var Event: TEvent);
var
  Ev: TEvent;
begin
  If Event.What = evMouseDown
  then begin
    inherited HandleEvent(Event);
    If Event.What = evNothing
    then begin
      GetEvent(Ev);
      If (Ev.What = evCommand) and (Ev.Command = cmClose)
      then Close
      else PutEvent(Ev)
    end
  end
  else
    inherited HandleEvent(Event)
end;

procedure TDesktopImageWindow.InitInterior;
var
  R: TRect;
begin
  GetClientRect(R);
  Interior := New(PDesktopImage, Init(R))
end;

procedure TDesktopImageWindow.SizeLimits(var Min, Max: TPoint);
begin
  Min.x := 100;
  Min.y := 80;
  Max.x := 300;
  Max.y := 240;
end;

end.
