#ifndef _ColorSpace_H_ 
#define _ColorSpace_H_

#include "border/BorderSpace.h"

template <class K>
class ColorSpace : public BorderSpace {
    protected:
        ColorSpace(); // Not implemented.

    public:
        /* constructing a ColorSpace instance */ 
        ColorSpace(const Ideal<K>&, int max_degree);
        /* destructor */
        virtual ~ColorSpace();

        /* determine whether or not the system of equations is feasible */
        virtual int isVarietyEmpty();
};

/* constructing a ColorSpace instance */ 
template <class K>
ColorSpace<K>::ColorSpace(const Ideal<K>& inI, int degree)
    : BorderSpace<K>(inI, degree)
{
}

/* default destructor */
template <class K>
ColorSpace<K>::~ColorSpace()
{
}

template <class K>
int
ColorSpace<K>::isVarietyEmpty()
{
    //std::cout << "Computing Border Space ..." << std::endl;
    /* construct linear system associated with this ideal */
    int codim_deg = -1;
    int iteration = 0;
    LinearSystem<K> *ls = construct_system();
    //print_linear_system(*ls);
    ls->rank_reverse();
    ++num_iterations;
    if (ls->get_codim(0,1) == 0) { 
        //std::cout << "The polynomial '1' is in the ideal." << std::endl;
        delete ls;
        return 0;
    }
    int curr_degree = 2;
    int num_lte_2_mons = calculate_num_lte_d_monomials(num_vars,2);
    int num_lte_3_mons = calculate_num_lte_d_monomials(num_vars,3);
    int num_lte_1_mons = calculate_num_lte_d_monomials(num_vars,1);
    std::vector<int> fixed(num_vars);
    while (1) {
        ls->setNumCols(num_lte_3_mons);
        codim_deg = ls->get_codim(0,num_lte_d_mons);
        int prev_rank = 0;
        int deg_iteration = 0;
        while (true) {
            double start = tic();
            int rank_tmp = ls->getNumRows();
            int col_lb = 0;
            plus_linear_system(*ls, prev_rank, ls->getNumRows(), col_lb, num_lte_d_mons);
            //print_linear_system(*ls);
            //std::cout << "Num new polynomials = " << ls->getNumRows()-rank_tmp << std::endl;
            prev_rank = rank_tmp;
            ls->rank_reverse(prev_rank);
            ++num_iterations;
            int new_codim_deg = ls->get_codim(0,num_lte_d_mons);
            if (l) {
                std::cout << "Iter = " << iteration <<  " D = " << curr_degree;
                std::cout << " N = " << ls->getNumCols() << " Dim = " << ls->getNumRows();
                std::cout << " Codim = " << new_codim_deg << " Time = " << toc(start) << "s";
                std::cout << " NZ = " << ls->getNumNonZeros() << std::endl;
            }
            ++iteration;
            //std::cout << "Num non-zeros = " << ls->getNumNonZeros() << std::endl;
            if (ls->get_codim(0,1) == 0) {
                //std::cout << "The polynomial '1' is in the ideal." << std::endl;
                delete ls;
                return 0;
            }
            if (new_codim_deg == codim_deg) { break; }
            codim_deg = new_codim_deg;
            ++deg_iteration;
        }
        //std::cout << "Linear System:\n";
        if (codim_deg == ls->get_codim(0,num_lte_d_sub_one_mons)) { break; }
        ++curr_degree;
    }
    if (max_degree != -1 && curr_degree > max_degree) { codim_deg = -1; }
    //print_linear_system(*ls);

    /* clean up */
    delete ls;
    return codim_deg;
}


template <class K>
LinearSystem<K>*
ColorSpace<K>::construct_system()
{
    /* variables dealing with monomials */
    int num_cols = calculate_num_lte_d_monomials(num_vars, curr_degree);

    // Construct the linear system.
    LinearSystem<K> *ls = new LinearSystem<K>(num_cols);

    std::vector<TermGenerator<K>*> gens;
    int row = 0;
    for (int i = 0; i < I.getNumGens(); i++) {
        for (int j = 0; j < I[i].getNumTerms(); j++) {
            ls->addEntry(row, I[i][j].mon.get_index(), I[i][j].coeff);
        }
        ++row;
    }

    //ls->print();
    return ls;
}

template <class K>
void
ColorSpace<K>::plus_linear_system(LinearSystem<K>& ls, Index row_start, Index row_end, 
                                    Index col_start, Index col_end)
{
    Index row = ls.getNumRows();
    ls.setNumRows(row+num_vars*(row_end-row_start));
    Monomial mon(num_vars);
    for (Index i = row_start; i < row_end; ++i) {
        if (ls.begin(i) == ls.end(i)) { continue; }
        if (ls.last(i).index() >= col_end || ls.last(i).index() < col_start) { continue; }
        for (typename LinearSystem<K>::RowIter ri = ls.begin(i); ri != ls.end(i); ++ri) {
            mon.set_index(ri.index());
            for (Index n = 0; n < num_vars; ++n) {
                //if (mon[n] != 1) {
                ++mon[n];
                ls.addEntry(row+n, mon.get_index(), ri.coeff());
                --mon[n];
                //}
                //else {
                //ls.addEntry(row+n, mon.get_index(), ri.coeff());
                //}
            }
        }
        row += num_vars;
    }
}

template <class K>
void
ColorSpace<K>::print_linear_system(const LinearSystem<K>& ls)
{
    Term<K> term(num_vars); 
    for (Index i = 0; i < ls.getNumRows(); ++i) {
        for (typename LinearSystem<K>::RowIter ri = ls.begin(i); ri != ls.end(i); ++ri) {
            if (ri != ls.begin(i)) { std::cout << "+"; }
            term.mon.set_index(ri.index());
            term.coeff = ri.coeff();
            term.print();
        }
        if (ls.begin(i) != ls.end(i)) { std::cout << "\n"; }
    }
}

#endif
