#ifndef IDEAL_BUILDER_H
#define IDEAL_BUILDER_H

#include "Ideal.h"
#include "Graph.h"

using namespace std;

class IsomorphismIdealBuilder {
    public:
        /* default constructor */
        IsomorphismIdealBuilder() {};
        /* destructor */
        ~IsomorphismIdealBuilder();

        /* construct an isomorphism ideal */
        template <class K>
        Ideal<K>* buildIsomorphismIdeal(const Graph& g1, const Graph& g2, bool optimised, bool symmetric);

    protected:

        /* construct an isomorphism ideal */
        template <class K>
        Ideal<K>* buildIsomorphismIdeal(const Graph& g1, const Graph& g2);

        /* construct an isomorphism ideal */
        template <class K>
        Ideal<K>* buildIsomorphismIdeal(const Graph& g1, const Graph& g2, bool optimised);

        /* construct an isomorphism ideal */
        template <class K>
        Ideal<K>* buildIsomorphismIdealSymm(const Graph& g, bool optimised);
};

template <class K>
Ideal<K> *
IsomorphismIdealBuilder::buildIsomorphismIdeal(
            const Graph& g1, const Graph& g2,
            bool optimised, bool symmetric)
{
    if (symmetric) { return buildIsomorphismIdealSymm<K>(g1, optimised); }
    else { return buildIsomorphismIdeal<K>(g1, g2, optimised); }
}

template <class K>
Ideal<K> *
IsomorphismIdealBuilder::buildIsomorphismIdeal(const Graph& g1, const Graph& g2)
{
    if (g1.getNumVertices() != g2.getNumVertices()) {
        std::cerr << "ERROR: Graph size mismatch.\n";
        exit(1);
    }
    int n = g1.getNumVertices();
    int num_vars = n*n;
    Ideal<K>* I = new Ideal<K>(num_vars);
    /* create a vector of size num_vertices, initialized to 0 */

    /* add polynomials to the ideal: x_i^2 - x_i */
    for (int i = 0; i < num_vars; ++i) {
        Polynomial<K> p(num_vars);
        Term<K> term(num_vars);
        term.mon[i] = 1;
        term.coeff = -1;
        p.addeq(term);
        term.mon[i] = 2;
        term.coeff = 1;
        p.addeq(term);
        I->addPoly(p);
    }

    for (int i = 0; i < n; ++i) {
        Polynomial<K> p1(num_vars);
        Polynomial<K> p2(num_vars);
        for (int j = 0; j < n; ++j) {
            Term<K> term1(num_vars);
            term1.coeff = 1;
            term1.mon[i*n+j] = 1;
            p1.addeq(term1);
            Term<K> term2(num_vars);
            term2.coeff = 1;
            term2.mon[i+j*n] = 1;
            p2.addeq(term2);
        }
        Term<K> term(num_vars);
        term.coeff = -1;
        p1.addeq(term);
        p2.addeq(term);
        I->addPoly(p1);
        I->addPoly(p2);
    }

    //g1.print_adj_list();
    //g2.print_adj_list();
    for (int i = 0; i < n; ++i) {
        for (int j = 0; j < n; ++j) {
            std::cerr << "(" <<i+1<<","<<j+1<<"): ";
            Polynomial<K> p(num_vars);
            const std::vector<int> aj = g2.getAdjList()[j];
            for (int k = 0; k < (int) aj.size(); ++k) {
                Term<K> term(num_vars);
                term.coeff = 1;
                term.mon[i*n+aj[k]] = 1;
                p.addeq(term);
                std::cerr << "+P("<<i+1<<","<<aj[k]+1<<")";
            }
            const std::vector<int> ai = g1.getAdjList()[i];
            for (int k = 0; k < (int) ai.size(); ++k) {
                Term<K> term(num_vars);
                term.coeff = -1;
                term.mon[ai[k]*n+j] = 1;
                p.addeq(term);
                std::cerr << "-P("<<ai[k]+1<<","<<j+1<<")";
            }
            I->addPoly(p);
            std::cerr << std::endl;
        }
    }

    /* return graph k-coloring ideal */
    return I;    
}

template <class K>
Ideal<K> *
IsomorphismIdealBuilder::buildIsomorphismIdeal(const Graph& g1, const Graph& g2, bool optimised)
{
    if (g1.getNumVertices() != g2.getNumVertices()) {
        std::cerr << "ERROR: Graph size mismatch.\n";
        exit(1);
    }
    int n = g1.getNumVertices();
    // The array indices is a map from (i,j) into a variable index in the ideal.
    int *indices = new int[n*n];
    int index = 0;
    for (int i = 0; i < n; ++i) {
        for (int j = 0; j < n; ++j) {
            if (!optimised || g1.getDegree(i) == g2.getDegree(j)) {
                indices[i*n+j] = index;
                index++;
            }
            else {
                indices[i*n+j] = -1;
            }
        }
    }
    int num_vars = index;
    Ideal<K>* I = new Ideal<K>(num_vars);
    /* create a vector of size num_vertices, initialized to 0 */

    /* add polynomials to the ideal: x_i^2 - x_i */
    for (int i = 0; i < num_vars; ++i) {
        Polynomial<K> p(num_vars);
        Term<K> term(num_vars);
        term.mon[i] = 1;
        term.coeff = -1;
        p.addeq(term);
        term.mon[i] = 2;
        term.coeff = 1;
        p.addeq(term);
        I->addPoly(p);
    }

    for (int i = 0; i < n; ++i) {
        Polynomial<K> p(num_vars);
        for (int j = 0; j < n; ++j) {
            if (indices[i*n+j] != -1) {
                Term<K> term(num_vars);
                term.coeff = 1;
                term.mon[indices[i*n+j]] = 1;
                p.addeq(term);
                std::cerr << "+P("<<i+1<<","<<j+1<<")";
            }
        }
        Term<K> term(num_vars);
        term.coeff = -1;
        p.addeq(term);
        std::cerr << "-1" << std::endl;
        I->addPoly(p);
    }

    for (int j = 0; j < n; ++j) {
        Polynomial<K> p(num_vars);
        for (int i = 0; i < n; ++i) {
            if (indices[i*n+j] != -1) {
                Term<K> term(num_vars);
                term.coeff = 1;
                term.mon[indices[i*n+j]] = 1;
                p.addeq(term);
                std::cerr << "+P("<<i+1<<","<<j+1<<")";
            }
        }
        Term<K> term(num_vars);
        term.coeff = -1;
        p.addeq(term);
        std::cerr << "-1" << std::endl;
        I->addPoly(p);
    }

    //g1.print_adj_list();
    //g2.print_adj_list();
    for (int i = 0; i < n; ++i) {
        for (int j = 0; j < n; ++j) {
            std::cerr << "(" <<i+1<<","<<j+1<<"): ";
            Polynomial<K> p(num_vars);
            const std::vector<int> aj = g2.getAdjList()[j];
            for (int k = 0; k < (int) aj.size(); ++k) {
                if (indices[i*n+aj[k]] != -1) {
                    Term<K> term(num_vars);
                    term.coeff = 1;
                    term.mon[indices[i*n+aj[k]]] = 1;
                    p.addeq(term);
                    std::cerr << "+P("<<i+1<<","<<aj[k]+1<<")";
                }
            }
            const std::vector<int> ai = g1.getAdjList()[i];
            for (int k = 0; k < (int) ai.size(); ++k) {
                if (indices[ai[k]*n+j] != -1) {
                    Term<K> term(num_vars);
                    term.coeff = -1;
                    term.mon[indices[ai[k]*n+j]] = 1;
                    p.addeq(term);
                    std::cerr << "-P("<<ai[k]+1<<","<<j+1<<")";
                }
            }
            I->addPoly(p);
            std::cerr << std::endl;
        }
    }

    /* return graph k-coloring ideal */
    return I;    
}



template <class K>
Ideal<K> *
IsomorphismIdealBuilder::buildIsomorphismIdealSymm(const Graph& g, bool optimised)
{
    int n = g.getNumVertices();
    int *indices = new int[n*n];
    int index = 0;
    for (int i = 0; i < n; ++i) {
        for (int j = i; j < n; ++j) {
            if (!optimised || g.getDegree(i) == g.getDegree(j)) {
                indices[i*n+j] = index;
                indices[j*n+i] = index;
                index++;
            }
            else {
                indices[i*n+j] = -1;
                indices[j*n+i] = -1;
            }
        }
    }
    int num_vars = index;

    Ideal<K>* I = new Ideal<K>(num_vars);
    /* create a vector of size num_vertices, initialized to 0 */

    /* add polynomials to the ideal: x_i^2 - x_i */
    for (int i = 0; i < num_vars; ++i) {
        Polynomial<K> p(num_vars);
        Term<K> term(num_vars);
        term.mon[i] = 1;
        term.coeff = -1;
        p.addeq(term);
        term.mon[i] = 2;
        term.coeff = 1;
        p.addeq(term);
        I->addPoly(p);
    }

    for (int i = 0; i < n; ++i) {
        Polynomial<K> p(num_vars);
        for (int j = 0; j < n; ++j) {
            if (indices[i*n+j] != -1) {
                Term<K> term(num_vars);
                term.coeff = 1;
                term.mon[indices[i*n+j]] = 1;
                p.addeq(term);
                std::cerr << "+P("<<i+1<<","<<j+1<<")";
            }
        }
        Term<K> term(num_vars);
        term.coeff = -1;
        p.addeq(term);
        std::cerr << "-1" << std::endl;
        I->addPoly(p);
    }

    //g.print_adj_list();
    for (int i = 0; i < n; ++i) {
        for (int j = i+1; j < n; ++j) {
            std::cerr << "(" <<i+1<<","<<j+1<<"): ";
            Polynomial<K> p(num_vars);
            const std::vector<int> aj = g.getAdjList()[j];
            for (int k = 0; k < (int) aj.size(); ++k) {
                if (indices[i*n+aj[k]] != -1) {
                    Term<K> term(num_vars);
                    term.coeff = 1;
                    term.mon[indices[i*n+aj[k]]] = 1;
                    p.addeq(term);
                    std::cerr << "+P("<<i+1<<","<<aj[k]+1<<")";
                }
            }
            const std::vector<int> ai = g.getAdjList()[i];
            for (int k = 0; k < (int) ai.size(); ++k) {
                if (indices[ai[k]*n+j] != -1) {
                    Term<K> term(num_vars);
                    term.coeff = -1;
                    term.mon[indices[ai[k]*n+j]] = 1;
                    p.addeq(term);
                    std::cerr << "-P("<<ai[k]+1<<","<<j+1<<")";
                }
            }
            I->addPoly(p);
            std::cerr << std::endl;
        }
    }

    /* return graph k-coloring ideal */
    return I;    
}
#endif
