#include <fstream>
#include <iostream>
#include <iomanip>
#include <sstream>
#include <getopt.h>
using namespace std;

#include "shared/F.h"
#include "shared/Graph.h"
#include "shared/GraphBuilder.h"
#include "isomorphism/IsomorphismIdealBuilder.h"
#include "nulla/Nullstellensatz.h"

/* forward declaration */
void print_usage();
void process_command_line(int argc, char *argv[], std::string& filename1, std::string& filename2);
template <class T>
void constructIdeal(Graph& g1, Graph& g2);

// Global variables (yuck!).
int f = 0; // The field.
int alpha_deg = -1; // The degree for running NulLA.
bool check = false;
bool print = false;
bool symmetric = false; // Is the permutation matrix symmetric.
bool optimised = false; // Do we optimise?

int main(int argc, char* argv[])
{
    std::string filename1;
    std::string filename2;
    process_command_line(argc, argv, filename1, filename2);

    GraphBuilder gb;
    Graph* g1 = gb.buildGraph(filename1.c_str());
    Graph* g2 = 0;  
    if (!filename2.empty()) { gb.buildGraph(filename2.c_str()); }
    else { g2 = g1; }
    switch (f) {
        case 0: {
            constructIdeal<Q>(*g1, *g2);
            break;
        }
        case 2: {
            constructIdeal<F2>(*g1, *g2);
            break;
        }
        case 4: {
            constructIdeal<F4>(*g1, *g2);
            break;
        }
        default: {
            FP::initialise(f);
            constructIdeal<FP>(*g1, *g2);
            break;
        }
    }

    return 0;
}

template <class T>
void constructIdeal(Graph& g1, Graph& g2)
{
    IsomorphismIdealBuilder cib;
    Ideal<T>* ideal;
    ideal = cib.buildIsomorphismIdeal<T>(g1, g2, optimised, symmetric);
    ideal->print();
    if (alpha_deg >= 0) {
        assert(&g1 == &g2); // Check g1 and g2 are the same.
        for (int i = 0; i < g1.getNumVertices(); ++i) {
            Nullstellensatz<T> nulla(*ideal,alpha_deg);
            Term<T> t(ideal->getNumVars());
            Polynomial<T> p(ideal->getNumVars());
            t.mon[i+i*g1.getNumVertices()] = 1;
            p.addeq(t);
            ideal->addPoly(p);
            //ideal->print();
            bool isInfeasible = nulla.isVarietyEmpty();
            if (isInfeasible) {
                if (print) { nulla.print_certificate(); }
                if (check) { nulla.check_certificate(); }
                cout << "System of equations is NOT feasible.\n";
            } else {
                cout << "System of equations MAY BE feasible... try a higher degree for alpha.\n";
            }
            ideal->popPoly();
        }
    }
    delete ideal;
}

void print_usage()
{
	std::cout << "Usage: isomorphism [options] <filename>\n";
	std::cout << "Options:\n";
	std::cout << "\t-f, --field\tThe field characteristic of the ideal. Default is 0.\n";
}

void process_command_line(  
            int argc, char *argv[],
            std::string& filename1,
            std::string& filename2)
{
	char c;
	int option_index(0);

	/* add more options here */
	struct option long_options[] =
	{
		{"field", optional_argument, 0, 'f'},
        {"degree", required_argument, 0, 'd'},
		{"symmetric", 0, 0, 's'},
        {"print", 0, 0, 'p'},
        {"check", 0, 0, 'c'},
        {"optimize", 0, 0, 'O'},
		{"help", optional_argument, 0, 'h'}
	};

	while ((c =
		getopt_long(argc, argv, "pcsOf:d:", long_options, &option_index)) != -1) {
		switch (c) {
			case 'f': /* externally generated graph */
				if (optarg) { f = atoi(optarg); }
				break;
			case 's': 
				symmetric = true;
				break;
			case 'O': 
				optimised = true;
				break;
			case 'd': 
                if (!optarg) { print_usage(); }
				else { alpha_deg = atoi(optarg); }
				break;
            case 'p':
                print = true;
                break;
            case 'c':
                check = true;
                break;
			case 'h': 
				print_usage();
				break;
		}
	}

    if (optind == argc) { 
        std::cerr << "ERROR: expected filename.\n";
        print_usage();
        exit(1);
    }
    filename1 = argv[optind];
    ++optind;
    if (optind != argc) { filename2 = argv[optind];}
}
