#ifndef _Nullstellensatz_H_ 
#define _Nullstellensatz_H_

#include "Index.h"
#include "MyTime.h"
#include "Ideal.h" 
#include "LinearSystem.h" 
#include "LinearSystemF2.h"
#include "TermGenerator.h"
#include "MonExpEnum.h"
#include "Util.h"
#include "Q.h"

#define _NULLA_EQUALITY

class NullstellensatzInterface {
    public:
        NullstellensatzInterface() {}
        virtual ~NullstellensatzInterface() {}

        /* determine whether or not the system of equations is feasible */
        virtual bool isVarietyEmpty() = 0;

        /* print certificate if linear system is feasible */
        virtual void print_certificate() const = 0;

        /* print certificate if linear system is feasible */
        virtual void check_certificate() const = 0;

        virtual void set_output_level(int level) = 0;
    private:
        
};

template <class K>
class Nullstellensatz : public NullstellensatzInterface {
    protected:
        Nullstellensatz(); // Not implemented.

        /* ideal to test for feasibility */
        const Ideal<K>& I;

        /* degree of Nullstellensatz certificate */
        const int max_alpha_deg;
        const int min_alpha_deg;
        int alpha_deg;

        /* number of variables */
        int num_vars;

        /* linear system */
        LinearSystem<K>* ls;

        int l;
    
        /** construct the linear system associated with this ideal and degree. */
        void construct_linear_system();

        /** construct the linear system associated with this ideal and degree. */
        void construct_linear_system_opt();

        /* The following prints a plus sign if v is rational and negative. */
        void printplus(const K& v) const;

    public:
        /* constructing a Nullstellensatz instance */ 
        Nullstellensatz(const Ideal<K>&, int degree);
        /* constructing a Nullstellensatz instance */ 
        Nullstellensatz(const Ideal<K>&, int min_degree, int max_degree);
        /* destructor */
        virtual ~Nullstellensatz();

        /* determine whether or not the system of equations is feasible */
        bool isVarietyEmpty();

        /* print certificate if linear system is feasible */
        virtual void print_certificate() const;

        /* print certificate if linear system is feasible */
        virtual void check_certificate() const;

        virtual void set_output_level(int level) { l = level; }
};

/* constructing a Nullstellensatz instance */ 
template <class K>
Nullstellensatz<K>::Nullstellensatz(const Ideal<K>& inI, int degree)
    : I(inI), max_alpha_deg(degree), min_alpha_deg(0), alpha_deg(0), num_vars(I.getNumVars()), ls(0), l(1)
{
}

/* constructing a Nullstellensatz instance */ 
template <class K>
Nullstellensatz<K>::Nullstellensatz(const Ideal<K>& inI, int min_degree, int max_degree)
    : I(inI), max_alpha_deg(min_degree), min_alpha_deg(min_degree), alpha_deg(min_degree),
      num_vars(I.getNumVars()), ls(0), l(1)
{
}

/* default destructor */
template <class K>
Nullstellensatz<K>::~Nullstellensatz()
{
    delete ls;
}

template <class K>
bool
Nullstellensatz<K>::isVarietyEmpty()
{
    for (alpha_deg = min_alpha_deg; alpha_deg <= max_alpha_deg; ++alpha_deg) {
        double start1 = tic();
        if (l) {
            std::cout << "Computing for degree " << alpha_deg << " ... ";
            std::cout << std::flush;
        }
        construct_linear_system();
        if (l) {
            std::cout << " (" << ls->getNumNonZeros() << ") " << std::flush;
            std::cout << toc(start1) << " " << std::flush;
        }
        double start2 = tic();
        /* if consistent -> certificate exists -> variety is EMPTY */
        bool result = ls->isConsistent();
        if (l) {
            std::cout << " (" << ls->getNumNonZeros() << ") " << std::flush;
            std::cout << toc(start2) << "s." << std::endl;
        }
        if (result) { return true; }
    }
    return false;
}

template <class K>
void
Nullstellensatz<K>::construct_linear_system()
{
    // Delete previous linear system.
    delete ls; ls = 0;

    /* variables dealing with monomials */
    int c_index = 0;

    // Construct the list of monomial generators.
    std::vector<TermGenerator<K>*> gens;
    for (int i = 0; i < I.getNumGens(); ++i) {
        int degree = I[i].degree();
        if (alpha_deg < degree) { continue; }
        for (int j = 0; j < I[i].getNumTerms(); ++j) {
            gens.push_back(new TermGenerator<K>(I[i][j], 0, alpha_deg-degree, c_index));
        }
        c_index += calculate_num_lte_d_monomials(num_vars, alpha_deg-degree);
    }

    // Construct the linear system.
    ls = new LinearSystem<K>(c_index+1);

    // Construct the heap of monomial generators.
    make_heap(gens.begin(), gens.end(), mg_cmp);
    int row = 0;
    Monomial prev(num_vars);
    // While there are still monomial generators on the heap.
    while (!gens.empty()) {
        TermGenerator<K>& term_gen = *(gens.front());
        const Monomial& curr = term_gen.get_mon();
        if (!(prev == curr)) { ++row; prev = curr; }
        ls->addEntry(row, term_gen.get_mon_index(), term_gen.get_coeff());
        pop_heap(gens.begin(), gens.end(), mg_cmp);
        if (term_gen.next_mon()) {
            push_heap(gens.begin(), gens.end(), mg_cmp);
        } else {
            gens.pop_back();
            delete &term_gen;
        }
    }
    // The first row of the linear system always corresponds to the constant row.
    ls->addRHSEntry(0, -1);
}

template <class K>
void
Nullstellensatz<K>::construct_linear_system_opt()
{
    // Delete previous linear system.
    delete ls; ls = 0;

    /* variables dealing with monomials */
    int c_index = 0;

    // Construct the list of monomial generators.
    Monomial constant(I.getNumVars());
    std::vector<Monomial> mons;
    std::vector<TermGenerator<K>*> gens;
    for (int i = 0; i < I.getNumGens(); ++i) {
        int degree = I[i].degree();
        for (int j = 0; j < I[i].getNumTerms(); ++j) {
            if (I[i][j].degree() == degree) {
                bool is_divisable = false;
                for (int k = 0; k < (int) mons.size(); ++k) {
                    if (mons[k].divides(I[i][j].mon)) { is_divisable = true; break; }
                }
                if (!is_divisable) {
                    mons.push_back(I[i][j].mon);
                    break;
                }
            }
        }
        if ((int) mons.size() != i+1) { mons.push_back(constant); }
        //mons[i].print(); std::cout << " ";
        if (alpha_deg < degree) { continue; }
        for (int j = 0; j < I[i].getNumTerms(); ++j) {
            gens.push_back(new TermGenerator<K>(I[i][j], 0, alpha_deg-degree, c_index, i));
        }
        c_index += calculate_num_lte_d_monomials(num_vars, alpha_deg-degree);
    }

    // Construct the linear system.
    ls = new LinearSystem<K>(c_index+1);

    // Construct the heap of monomial generators.
    make_heap(gens.begin(), gens.end(), mg_cmp);
    int row = 0;
    Monomial prev(num_vars);
    // While there are still monomial generators on the heap.
    while (!gens.empty()) {
        TermGenerator<K>& term_gen = *(gens.front());
        const Monomial& mult = term_gen.get_mon_mult();
        bool is_divisable = false;
        for (int i = 0; i < term_gen.get_poly_index(); ++i) {
            if (!mons[i].is_constant() && mons[i].divides(mult)) {
                is_divisable = true;
                //mons[i].print(); std::cout << " divides ";
                //mult.print(); std::cout << "\n";
                break;
            }
        }
        if (!is_divisable) {
            const Monomial& curr = term_gen.get_mon();
            if (!(prev == curr)) { ++row; prev = curr; }
            ls->addEntry(row, term_gen.get_mon_index(), term_gen.get_coeff());
        }
        pop_heap(gens.begin(), gens.end(), mg_cmp);
        if (term_gen.next_mon()) {
            push_heap(gens.begin(), gens.end(), mg_cmp);
        } else {
            gens.pop_back();
            delete &term_gen;
        }
    }
    // The first row of the linear system always corresponds to the constant row.
    ls->addRHSEntry(0, -1);
}

template <class K>
void
Nullstellensatz<K>::print_certificate() const
{
    const std::vector<K>& sol = ls->getSolution();
    MonExpEnum mon_exp_enum(num_vars, 0);

   /* printing in maple format */
    std::cout << "cert := ";

    /* We find out which ideal generators are used in the certificate. */
    int c_index = 0;
    std::vector<bool> isGenUsed(I.getNumGens());
    for (int i = 0; i < I.getNumGens(); ++i) {
        isGenUsed[i] = false;
        int degree = I[i].degree();
        if (alpha_deg>=degree) {
            int num_mons = calculate_num_lte_d_monomials(num_vars, alpha_deg-degree);
            for (int j = c_index; j < c_index+num_mons; ++j) {
                if (sol[j] != 0) { isGenUsed[i] = true; break; }
            }
            c_index += num_mons;
        }
    }

    c_index = 0;
    bool isFirstPoly = true;
    for (int i = 0; i < I.getNumGens(); ++i) {
        int degree = I[i].degree();
        if (isGenUsed[i]) {
            if (!isFirstPoly) { printf(" + "); } else { isFirstPoly = false; }
            printf("(");
            bool isFirstMon = true;
            for (int d = 0; d <= alpha_deg-degree; ++d) {
                mon_exp_enum.reset_deg(d);
                do {
                    const Monomial& next = mon_exp_enum.get_mon();
                    /* add this coefficient to relevant generators */
                    if (sol[c_index] != 0) {
                        if (!isFirstMon) { printplus(sol[c_index]); }
                        else { isFirstMon = false; }
                        if (next.is_constant()) {
                            std::cout << sol[c_index];
                        }
                        else {
                            if (sol[c_index] != 1 && sol[c_index] != -1) {
                                std::cout << sol[c_index] << "*";
                            }
                            next.print();
                        }
                    }
                    c_index++;
                } while (mon_exp_enum.next_mon());
            }
            printf(")*(");
            I[i].print();
            printf(")");
        }
        else if (alpha_deg >= degree) {
            c_index += calculate_num_lte_d_monomials(num_vars, alpha_deg-degree);
        }
    }
    std::cout << ";\n";
}

template <class K>
void
Nullstellensatz<K>::check_certificate() const
{
    std::cout << "Checking certificate..." << std::flush;
    const std::vector<K>& sol = ls->getSolution();
    MonExpEnum mon_exp_enum(num_vars, 0);

    Polynomial<K> poly(num_vars);
    Polynomial<K> poly_tmp(num_vars);

    int c_index = 0;
    for (int i = 0; i < I.getNumGens(); ++i) {
        int degree = I[i].degree();
        for (int d = 0; d <= alpha_deg-degree; ++d) {
            mon_exp_enum.reset_deg(d);
            do {
                /* add this coefficient to relevant generators */
                if (sol[c_index] != 0) {
                    Term<K> term(sol[c_index], mon_exp_enum.get_mon());
                    poly_tmp = I[i];
                    poly_tmp.muleq(term);
                    poly.addeq(poly_tmp);
                }
                c_index++;
            } while (mon_exp_enum.next_mon());
        }
    }
    if (poly.getNumTerms() != 1 || !poly[0].mon.is_constant()) {
        std::cerr << "\nERROR: Certificate is invalid!.\n";
        poly.print(); std::cout << std::endl;
        return;
    }
    std::cout << "ok.\n";
}

template <class K>
inline void
Nullstellensatz<K>::printplus(const K& ) const
{
    std::cout << "+";
}

template <>
inline void
Nullstellensatz<Q>::printplus(const Q& v) const
{
    if (v >= 0) { std::cout << "+"; }
}

#endif
