#ifndef _F4_H_
#define _F4_H_

#include <inttypes.h>
#include <iostream>
#include <cassert>

class F4
{
	public:
		F4();
        F4(int);
		F4(const F4& x);
		F4& operator=(int);
		F4& operator=(const F4& x);

		bool operator==(const F4& x) const; 
		bool operator!=(const F4& x) const;

		F4 operator+() const;
		F4 operator-() const;
		F4 operator+(const F4& x) const;
		F4 operator-(const F4& x) const;
		F4 operator*(const F4& x) const;
		F4 operator/(const F4& x) const;
		void operator+=(const F4& x);
		void operator-=(const F4& x);
		void operator*=(const F4& x);
		void operator/=(const F4& x);

		static void add(const F4& x1, const F4& x2, F4& x);
		static void sub(const F4& x1, const F4& x2, F4& x);
		static void mul(const F4& x1, const F4& x2, F4& x);
		static void div(const F4& x1, const F4& x2, F4& x);

		friend std::ostream& operator<<(std::ostream& os, const F4& x);
		friend std::istream& operator>>(std::istream& is, F4& x);

	private:
        static uint8_t mul_lookup[4][4];
        static uint8_t div_lookup[4][4];
		uint8_t v;
};

inline F4::F4() { v = 0; }
inline F4::F4(int i) { if (i<0) { i *= -1; } v = i%4; } 
inline F4::F4(const F4& x) { v = x.v; }
inline F4& F4::operator=(const F4& x) { v = x.v; return *this; }
inline F4& F4::operator=(int i) { if (i<0) { i*= -1; } v = i%4; return *this; }

inline bool F4::operator==(const F4& x) const { return v == x.v; }
inline bool F4::operator!=(const F4& x) const { return v != x.v; }

inline F4 F4::operator+() const { return *this; }
inline F4 F4::operator-() const { return *this; }
inline F4 F4::operator+(const F4& x) const { return F4(v ^ x.v); }
inline F4 F4::operator-(const F4& x) const { return F4(v ^ x.v); }
inline F4 F4::operator*(const F4& x) const { return F4(mul_lookup[v][x.v]); }
inline F4 F4::operator/(const F4& x) const { assert(x.v); return F4(div_lookup[v][x.v]); }

inline void F4::operator+=(const F4& x) { v ^= x.v; }
inline void F4::operator-=(const F4& x) { v ^= x.v; }
inline void F4::operator*=(const F4& x) { v = mul_lookup[v][x.v]; }
inline void F4::operator/=(const F4& x) { assert(x.v); v = div_lookup[v][x.v]; }

inline void F4::add(const F4& x1, const F4& x2, F4& x) { x.v=x1.v^x2.v; }
inline void F4::sub(const F4& x1, const F4& x2, F4& x) { x.v=x1.v^x2.v; }
inline void F4::mul(const F4& x1, const F4& x2, F4& x) { x.v=mul_lookup[x1.v][x2.v]; }
inline void F4::div(const F4& x1, const F4& x2, F4& x) { assert(x2.v); x.v=div_lookup[x1.v][x2.v]; }

inline std::ostream& operator<<(std::ostream& os, const F4& x) { os << (int) x.v; return os; } 
inline std::istream& operator>>(std::istream& is, F4& x) {
	is >> x.v;
	x.v%=4;
	return is;
}

#endif
