#ifndef _FP_H_
#define _FP_H_
#include <cassert>
#include <iostream>

typedef unsigned int UINT;
typedef int INT;

class FP
{
	public:
		static void initialise(UINT _p);
	private:
		static UINT p;
		static UINT* inverse;
		static UINT compute_inverse(UINT b);

	public:
		FP();
		FP(const FP& x);
		FP(UINT);
		FP(INT);
		FP& operator=(const FP& x);

		bool operator==(const FP& x) const;
		bool operator!=(const FP& x) const;

		FP operator+() const;
		FP operator-() const;

		void operator++();
		void operator--();

		FP operator+(const FP& x) const;
		FP operator-(const FP& x) const;
		FP operator*(const FP& x) const;
		FP operator/(const FP& x) const;
		void operator+=(const FP& x);
		void operator-=(const FP& x);
		void operator*=(const FP& x);
		void operator/=(const FP& x);

		static void add(const FP& x1, const FP& x2, FP& x);
		static void sub(const FP& x1, const FP& x2, FP& x);
		static void mul(const FP& x1, const FP& x2, FP& x);
		static void div(const FP& x1, const FP& x2, FP& x);

        static int getFieldCharacteristic();

		friend std::ostream& operator<<(std::ostream& os, const FP& x);
		friend std::istream& operator>>(std::istream& is, FP& x);

	private:
		UINT v;
};

inline FP::FP() { v=0; }
inline FP::FP(UINT _v) {  v=_v%p; }
inline FP::FP(INT _v) { if (_v>=0) { v=_v%p; } else { v=(_v%(INT)p)+p; } }
inline FP::FP(const FP& x) { v=x.v; }
inline FP& FP::operator=(const FP& x) { v=x.v; return *this; }

inline bool FP::operator==(const FP& x) const { return v == x.v; }
inline bool FP::operator!=(const FP& x) const { return v != x.v; }

inline FP FP::operator+() const { return FP(v); }
inline FP FP::operator-() const { return FP(p-v); }

inline void FP::operator++() { ++v; v%=p; }
inline void FP::operator--() { if (v) {--v; } else { v=p-1; } }

inline FP FP::operator+(const FP& x) const { return FP(v+x.v); }
inline FP FP::operator-(const FP& x) const { return FP(v+p-x.v); }
inline FP FP::operator*(const FP& x) const { return FP(v*x.v); }
inline FP FP::operator/(const FP& x) const { assert(x.v); return FP(v*inverse[x.v]); }

inline void FP::operator+=(const FP& x) { v += x.v; v %= p; }
inline void FP::operator-=(const FP& x) { v += p-x.v; v %= p; }
inline void FP::operator*=(const FP& x) { v *= x.v; v %= p; }
inline void FP::operator/=(const FP& x) { assert(x.v); v *= inverse[x.v]; v %= p; }

inline void FP::add(const FP& x1, const FP& x2, FP& x) { x.v=(x1.v+x2.v)%p; }
inline void FP::sub(const FP& x1, const FP& x2, FP& x) { x.v=(x1.v+p-x2.v)%p; }
inline void FP::mul(const FP& x1, const FP& x2, FP& x) { x.v=(x1.v*x2.v)%p; }
inline void FP::div(const FP& x1, const FP& x2, FP& x)
{ assert(x2.v); x.v=(x1.v*inverse[x2.v])%p; }

inline int FP::getFieldCharacteristic() { return p; }

inline std::ostream& operator<<(std::ostream& os, const FP& x) { os << x.v; return os; } 
inline std::istream& operator>>(std::istream& is, FP& x) {
    int v;
	is >> v;
	if (v >= 0) { x.v = v % x.p; } else { x.v = v % x.p + x.p; }
	return is;
}

#endif
