#include "GraphBuilder.h"
#include "Graph.h"

#include <string>
#include <iostream>
#include <fstream> 
#include <sstream>
#include <assert.h>

using namespace std;

GraphBuilder::GraphBuilder()
{
}

/* destructor */
GraphBuilder::~GraphBuilder()
{
}

Graph *GraphBuilder::buildGraph(const char* filename)
{
	/* the argument is the filename */
	return get_file_instance(filename);
}

Graph *GraphBuilder::get_file_instance(const char *fn)
{
	/* vector to store edges as they are created */
	vector<Graph::Edge> edges;
	Graph::Edge e;

	int num_v, num_e;
	string line;
	/* the argument is the filename */
	ifstream in(fn);

	/* check if graph file opened sucessfully */	
	if (!in) {
		cout << "ERROR: Unable to open file " << fn << endl;
		exit(1);
	}

	/*
	 * According to DIMACS format, lines with comments are preceded by 
	 * a 'c' character, and there will be a line of the form p edge n m
	 * where n = # of vertices and m = # of edges. Last follows the edges
	 * of the form e 5 11, indicating an edge between 5 and 11.
	 */
	while(!in.eof()) {
		getline(in, line);
		if (get_NumVertAndNumEdgefromDIMACSformat(line, num_v, num_e)) {
			break;
		}
	}

	/* get edges out of file and add them to the graph */
	while(!in.eof()) {
		getline(in, line);
		/* only add edges of the form e i j */
		if (line.length() != 0 && line[0] == 'e') {
			parse_edge(line, &e);
			edges.push_back(e);
		}
	}
	in.close();

	/* create a new graph instance */
	assert(num_e == (int) edges.size());
	Graph *G = new Graph(num_v, edges);

	return G;
}

bool GraphBuilder::get_NumVertAndNumEdgefromDIMACSformat(const string &s,
	int &num_v, int &num_e)
{
	/* if the line begins with a 'c', it is a comment */
	if (s.length() == 0 || s[0] == 'c') { return false; }

	/* if the line begins with a 'p edge', this line contains num_v and num_e */
	int ret = sscanf(s.c_str(), "p edge %d %d", &num_v, &num_e);
	if (ret != 2) {
		cout << "Error: Graph file improperly formed at line = " << s << endl;
		exit(1);
	}
	return true;
}

void GraphBuilder::parse_edge(const string &s, Graph::Edge *e)
{
	/* s is of the form e i j */
	int ret = sscanf(s.c_str(), "e %d %d", &e->i, &e->j);
	if (ret != 2) {
		cout << "Error: Graph file improperly formed at line = " << s << endl;
		exit(1);
	}
	--e->i;
	--e->j;

	if (e->i < 0 || e->j < 0) {
		cout << "Error: Vertices are assumed to be non-negative.\n";
		cout << "Error: Edge (" << e->i << ", " << e->j << ") improperly formed" << endl;
		exit(1);
	}
}

Graph*
GraphBuilder::buildGraph(int num_v, float p)
{
	/* vector to store edges as they are created */
	vector<Graph::Edge> edges;
	Graph::Edge e;

    for (int i = 0; i < num_v; ++i) {
        for (int j = i+1; j < num_v; ++j) {
            if (p >= (float) rand() / RAND_MAX) {
                e.i = i; e.j = j;    
                edges.push_back(e);
            }
        }
    }   
    return new Graph(num_v, edges);
}
