#ifndef IDEAL_H
#define IDEAL_H

#include "Polynomial.h"
#include "F.h"
#include <string>
#include <vector>

class IdealAPI
{
public:
    IdealAPI() {}
    virtual ~IdealAPI() {}

    virtual void print() const = 0;
    virtual void print_magma() const = 0;
    virtual void print_m2() const = 0;
    virtual void print_cocoa() const = 0;
};


template <class K>
class Ideal : public IdealAPI {
	private:
		/** vectors for storing polynomial information */
		std::vector<Polynomial<K> > polys;	

		/** number of variables in the polynomial */
		const int n;	

	public:
		/** Default constructor */
		Ideal(int num_vars);
		/** Destructor */
		virtual ~Ideal();

		/** Get the ith polynomial generator of the ideal. */
		const Polynomial<K>& operator[](int i) const;

		/** Add a polynomial generator. */
		void addPoly(const Polynomial<K>& p);
        /** Removes the last polynomial. **/
        void popPoly();

		/** Get the number of generators for the ideal */
		int getNumGens() const;

		/** Get the number of variables in the ideal */
		int getNumVars() const;

		/** Print out the ideal. */
		void print() const;
		void print_magma() const;
		void print_cocoa() const;
		void print_m2() const;
};

/** Default constructor */
template <class K> inline
Ideal<K>::Ideal(int num_vars) : n(num_vars) {}

/** Destructor */
template <class K> inline
Ideal<K>::~Ideal() {}

/** Get the ith polynomial generator of the ideal. */
template <class K> inline
const Polynomial<K>& Ideal<K>::operator[](int i) const { return polys[i]; } 

/** Add a polynomial generator. */
template <class K> inline
void Ideal<K>::addPoly(const Polynomial<K>& p) { polys.push_back(p); }

/** Remove the last polynomia. */
template <class K> inline
void Ideal<K>::popPoly() { polys.erase(polys.end()--); }

/** Get the number of generators for the ideal */
template <class K> inline
int Ideal<K>::getNumGens() const { return (int) polys.size(); }

/** Get the number of variables in the ideal */
template <class K> inline
int Ideal<K>::getNumVars() const { return n; }

/** Print out the ideal. */
template <class K>
void Ideal<K>::print() const
{
    std::cout << "FIELD=F" << getFieldChar<K>() << "\n";
    std::cout << "NUMBER_OF_VARIABLES=" << n << "\n";
	for (size_t i = 0; i < polys.size(); ++i) {
		polys[i].print();
        std::cout << "\n";
	}
    std::cout << std::flush;
}

/** Print out the ideal. */
template <class K>
void Ideal<K>::print_cocoa() const
{
    std::cout << "Use R::=";
    if (getFieldChar<K>() == 0) { std::cout << "Q"; }
    else { std::cout << "Z/(" << getFieldChar<K>() << ")"; }
    std::cout << "[x[0.." << n << "]], DegRevLex;" << "\n";
    std::cout << "I:=Ideal(" << "\n";
	for (size_t i = 0; i < polys.size(); ++i) {
		polys[i].print();
        if (i < polys.size()-1) { std::cout << ","; }
        std::cout << "\n";
	}
    std::cout << ");\n";
    std::cout << "Time T := GBasis(I);\n";
    std::cout << "I.GBasis;\n";
    std::cout << std::flush;
}

/** Print out the ideal. */
template <class K>
void Ideal<K>::print_magma() const
{
    std::cout << "R<";
    for (int i = 0; i < n; ++i) {
        std::cout << "x_" << i;
        if (i < n-1) { std::cout << ","; }
    } 
    std::cout << "> := PolynomialRing(";
    if (getFieldChar<K>() == 0) { std::cout << "RationalField()"; }
    else { std::cout << "GF(" << getFieldChar<K>() << ")"; }
    std::cout << "," << n << ",\"grevlex\");\n";
    std::cout << "time G := GroebnerBasis([" << "\n";
	for (size_t i = 0; i < polys.size(); ++i) {
		polys[i].print_magma();
        if (i < polys.size()-1) { std::cout << ","; }
        std::cout << "\n";
	}
    std::cout << "]);\n";
    std::cout << "G;\n";
    std::cout << std::flush;
}

/** Print out the ideal. */
template <class K>
void Ideal<K>::print_m2() const
{
    std::cout << "R = ";
    if (getFieldChar<K>() == 0) { std::cout << "QQ"; }
    else { std::cout << "ZZ/" << getFieldChar<K>(); }
    std::cout << "[x_0..x_" << n << ",MonomialOrder=>GRevLex];\n";
    std::cout << "I = ideal(" << "\n";
	for (size_t i = 0; i < polys.size(); ++i) {
		polys[i].print_magma();
        if (i < polys.size()-1) { std::cout << ","; }
        std::cout << "\n";
	}
    std::cout << ");\n";
    std::cout << "time gens gb I;\n";
    std::cout << "quit;\n";
    std::cout << std::flush;
}


#endif
