#ifndef _IdealFactory_H_
#define _IdealFactory_H_

#include <string>
#include <iostream>
#include "Ideal.h"

class IdealFactory {
    protected:
        const char COMMENT;
        template <class K> Ideal<K>* buildIdeal(std::istream& in);

	public:
		IdealFactory();

        IdealAPI* buildIdeal(const char* filename);
};

template <class K>
Ideal<K>*
IdealFactory::buildIdeal(std::istream& in)
{
	std::string line;

	/* Strip comments. */
	while (in.peek() == COMMENT) { getline(in, line); }

	/* Find the number of variables. */
    int num_vars = 0;
	getline(in, line);
	if (sscanf(line.c_str(), "NUMBER_OF_VARIABLES=%d", &num_vars) != 1) {
		std::cerr << "ERROR! Number of variables not found.\n"; 
		exit(1);
	}

	Ideal<K>* ideal = new Ideal<K>(num_vars);

	/* Strip comments. */
	while (in.peek() == COMMENT) { getline(in, line); }

	while (!getline(in, line).eof()) {
		Polynomial<K> poly(num_vars);
		const char* curr = line.c_str();
		const char* end = curr+line.size();

		Term<K> term(num_vars);
		// Skip initial whitespace.
		while (curr != end && isspace(*curr)) { ++curr; }
		// Parse the polynomial string.
		while (curr != end)
		{
			term.mon.set_constant(); // Reset temporary monomial.
			// Find the end of the next monomial.
		   const char* mon_end = std::min(std::find(curr+1, end, '+'), std::find(curr+1, end, '-'));

			// Parse the monomial string.  First, find coefficient.
			int num = 1;
			int den = 1;
            int sign = 1;
			if (*curr == '-') { sign = -1; ++curr; } else if (*curr == '+') { ++curr; }
			sscanf(curr, "%d/%d", &num, &den);
			term.coeff = sign*num;
            assert(den >= 0);
			if (den != 1) { term.coeff /= den; }

			// Find the first x variable.
			curr = std::find(curr, mon_end, 'x');
			while (curr !=  mon_end) {
				int e = 1; // The exponent of the variable.
				int index = -1; // The index of the variable.
				// Retrieve the variable index and exponent.
				// Note that the exponent is only read if it exists.
				sscanf(curr, "x[%d]^%d", &index, &e);
				if (index == -1) {
					std::cerr << "ERROR: badly formed polynomial string.\n";
					std::cerr << "Got to here ^" << curr << "\n";
					exit(1);
				}
				if (index < 0 || index >= num_vars) {
                    std::cerr << "ERROR: Variable index out of range: ";
                    std::cerr << index << " > " << num_vars << ".\n";
					std::cerr << "Got to here ^" << curr << "\n";
                    exit(1);
                }
                if (e <= 0) { 
                    std::cerr << "ERROR: Only positive exponents accepted.\n";
					std::cerr << "Got to here ^" << curr << "\n";
                    exit(1);
                }
				term.mon[index] = e; // Set the exponent.
				// Find the next x variable.
				curr = std::find(curr+1, mon_end, 'x');
				// Note that finding the next 'x' variable may skip over badly
				// formed input without issuing an error message.
			}
			poly.addeq(term);
			curr = mon_end;
		}
		ideal->addPoly(poly);

		/* Strip comments. */
		while (in.peek() == COMMENT) { getline(in, line); }
	}
    //ideal->print();
    return ideal;
}

#endif
