#ifndef _MonomialGenerator_H
#define _MonomialGenerator_H

#include "Monomial.h"
#include "MonExpEnum.h"
#include <vector>

class MonomialGenerator
{
	public:
		/** Constructor. */
		MonomialGenerator(
							 const Monomial& m,
							 int min_degree,
							 int max_degree,
							 int offset,
                             int poly = -1);
		/** Constructor. */
		MonomialGenerator(
							 const Monomial& m,
							 const std::vector<int>& degrees,
							 int offset);
		/** Constructor. */
		MonomialGenerator(
							 const Monomial& m,
							 const std::vector<int>& degrees,
							 const std::vector<int>& offsets);

		/** Updates the next monomial. */
		/** False if there are no more monomials to generate. */
		bool next_mon();

		/** Return the next monomial. */
		const Monomial& get_mon() const;

		/** Return the next monomial mulitplier. */
		const Monomial& get_mon_mult() const;

		/** Get the degree of the monomial. */
		int get_mon_degree() const;

		/** Get the index of the monomial. */
		int get_mon_index() const;

        /** Get the index of the associated polynomial. */
        int get_poly_index() const;
        
		/** Get the indices of the monomial. */
		const std::vector<int>& get_mon_indices() const;

		/** Degree reverse lexicographic ordering on the monomials. */
		bool operator<(const MonomialGenerator& mg) const;

	private:
		/// The degrees of the generated monomials.
		std::vector<int> degrees;
		/// The current degree of the gerenated monomials.
		int curr_deg;
		/// The degree of the monomial.
		int mon_deg;
        /// The index of the polynomial.
        int poly;
		/// The index of the generated monomial.
		std::vector<int> mon_indices;
		/// The original monomial.
		const Monomial orig_mon;
		/// Enumerator of the mulipliers.
		MonExpEnum enumerator;
		/// The next monomial.
		Monomial mon;
};

bool mg_cmp(const MonomialGenerator* mg1, const MonomialGenerator* mg2);

inline const Monomial&
MonomialGenerator::get_mon() const
{
	return mon;
}

inline const Monomial&
MonomialGenerator::get_mon_mult() const
{
    return enumerator.get_mon();
}

inline int
MonomialGenerator::get_mon_degree() const
{
	return mon_deg;
}

inline const std::vector<int>&
MonomialGenerator::get_mon_indices() const
{
	return mon_indices;
}

inline int
MonomialGenerator::get_mon_index() const
{
	assert(mon_indices.size() == 1);
	return mon_indices[0];
}

inline int
MonomialGenerator::get_poly_index() const
{
	return poly;
}

inline bool
MonomialGenerator::operator<(const MonomialGenerator& mg) const
{
	if (mon_deg < mg.mon_deg) { return true; }
	if (mg.mon_deg < mon_deg) { return false; }
	return (mg.mon < mon);
}

#endif
