#include "MonomialSparse.h"
#include <assert.h>
#include <cstring>
#include <iostream>

using namespace std;

std::vector<std::vector<int> > MonomialSparse::index_map;

MonomialSparse::MonomialSparse(int num_vars)
	: n(num_vars)
{
}

MonomialSparse::MonomialSparse(const std::vector<int>& v)
	: n(v.size())
{
	for (int i = 0; i < n; ++i) {
		if (v[i]) { exp.push_back(Exp(i, v[i])); }
	}
}

MonomialSparse::MonomialSparse(int num_vars, const int* v)
	: n(num_vars)
{
	for (int i = 0; i < n; ++i) {
		if (v[i]) { exp.push_back(Exp(i, v[i])); }
	}
}

MonomialSparse::MonomialSparse(const MonomialSparse& m)
	: n(m.n)
{
	*this = m;
}

MonomialSparse&
MonomialSparse::operator=(const MonomialSparse& m)
{
	assert(n = m.n);
	exp = m.exp;
	return *this;
}

MonomialSparse::~MonomialSparse()
{
}

void
MonomialSparse::mul(
				const MonomialSparse& m1,
				const MonomialSparse& m2,
				MonomialSparse& res)
{
	assert(m1.n == m2.n && m1.n == res.n);
	const vector<Exp>& exp1 = m1.exp;
	const vector<Exp>& exp2 = m2.exp;
	vector<Exp>& exp = res.exp;
	exp.clear();

	vector<Exp>::const_iterator it1 = exp1.begin();
	vector<Exp>::const_iterator it2 = exp2.begin();
	while (it1 != exp1.end() && it2 != exp2.end())
	{
		if (it1->i < it2->i) {
			exp.push_back(*it1); ++it1;
		} else if (it2->i < it1->i) {
			exp.push_back(*it2); ++it2;
		} else {
			exp.push_back(Exp(it1->i, it1->e+it2->e));
			++it1; ++it2;
		}
	}

	exp.insert(exp.end(), it1, exp1.end());
	exp.insert(exp.end(), it2, exp2.end());
}


void 
MonomialSparse::mul(const MonomialSparse& m)
{
	assert(n == m.n);

	vector<Exp>::iterator it1 = exp.begin();
	vector<Exp>::const_iterator it2 = m.exp.begin();
	while (it1 != exp.end() && it2 != m.exp.end()) {
		if (it1->i < it2->i) {
			++it1;
		} else if (it2->i < it1->i) {
			it1 = exp.insert(it1, *it2); ++it2; ++it1;
		} else {
			it1->e += it2->e;
			++it1; ++it2;
		}
	}

	exp.insert(exp.end(), it2, m.exp.end());
}

bool
MonomialSparse::divides(const MonomialSparse& m) const
{
	assert(n == m.n);

	vector<Exp>::const_iterator it1 = exp.begin();
	vector<Exp>::const_iterator it2 = m.exp.begin();
	while (it1 != exp.end() && it2 != m.exp.end()) {
		if (it1->i < it2->i) { return false; }
		else if (it2->i < it1->i) { ++it2; }
		else {
			if (it1->e > it2->e) { return false; }
			++it1; ++it2;
		}
	}
    if (it1 != exp.end()) { return false; }
    return true;
}

void
MonomialSparse::swap(MonomialSparse& m)
{
	assert(n == m.n);
	exp.swap(m.exp);
}

void
MonomialSparse::print() const
{
	for (size_t i = 0; i < exp.size(); ++i) {
		if (i != 0) { std::cout << "*"; }
		if (exp[i].e != 1) { printf("x[%d]^%d",exp[i].i,exp[i].e); }
		else { printf("x[%d]",exp[i].i); }
	}
}

void
MonomialSparse::print_magma() const
{
	for (size_t i = 0; i < exp.size(); ++i) {
		if (i != 0) { std::cout << "*"; }
		if (exp[i].e != 1) { printf("x_%d^%d",exp[i].i,exp[i].e); }
		else { printf("x_%d",exp[i].i); }
	}
}

void
MonomialSparse::print_debug() const
{
	size_t index = 0;
	for (int i = 0; i < n; ++i) {
		if (index < exp.size() && exp[index].i == i) {
			cout << " " << exp[index].e;
			++index;
		}
		else { cout << " 0"; }
	}
}
