#ifndef _MonomialSparse_H_
#define _MonomialSparse_H_

#include "Index.h"
#include "Exponent.h"
#include "Util.h"
#include <assert.h>
#include <vector>
#include <string>
#include <iostream>

class MonomialSparse
{
	public:
		/** Constructor. */
		explicit MonomialSparse(int num_vars);
		/** Constructor Dense. */
		MonomialSparse(const std::vector<int>& v);
		/** Constructor Dense. */
		MonomialSparse(int num_vars, const int *v);
		/** Copy constructor. */
		MonomialSparse(const MonomialSparse& m);
		/** Assignment operator. */
		MonomialSparse& operator=(const MonomialSparse& m);
		/** Destructor. */
		~MonomialSparse();

		/** Swap the contents of two monomials. */
		void swap(MonomialSparse& m);

		/** Set all exponents to zero. */
		void set_constant();
		/** Are all exponents zero. */
		bool is_constant() const;

		/** Print monomial. */
		void print() const;
		void print_magma() const;
		/** Print monomial. */
		void print_debug() const;

		/** Less than comparison operator. */
		bool operator<(const MonomialSparse& m) const;

		/** Equality comparison operator. */
		bool operator==(const MonomialSparse& m);

        /** Comparison. */
        static int cmp(const MonomialSparse& m1, const MonomialSparse& m2);

		/** Multiply two monomials m1 and m2. Result stored in res.*/
		static void mul(const MonomialSparse& m1,
							 const MonomialSparse& m2,
							 MonomialSparse& res);

        /** Multiply this monomial by m */
        void mul(const MonomialSparse& m);

        // True is this monomial divides m.
        bool divides(const MonomialSparse& m) const;

        /** Returns the degree of the monomial. */
        int degree() const;

		/** Get the number of variables. */
		int get_num_vars() const;

		/** Get the number of non-zero entries. */
		int get_num_non_zeros() const;

        /** build the map to construct indices **/
        static void build_index_map(int num_vars, int max_deg);
        /** Get the index representation of the monomial. */
        Index get_index() const;
        /** Set the monomial so that it equals the given index. */
        void set_index(Index i);

	private:
        /* index map */
        static std::vector<std::vector<int> > index_map;

		int n; // The number of variables.
		// An exponent.
		struct Exp {
			public:
				Exp(int index, int exponent) : i(index), e(exponent) {}
                Exp(const Exp& exp) : i(exp.i), e(exp.e) {}
                const Exp& operator=(const Exp& exp) { i=exp.i; e=exp.e; return *this; }
				int i; int e;
		};
		std::vector<Exp> exp;

	public:
		// TODO: What follows is a quick and dirty implementation so that the
		// MonomialSparse class can work instead of MonomialDense without
		// changing any code. Preferably an iterator class is used instead.
		class Var {
			public:
				void operator++();
				void operator--();
				void operator=(int e);
				void operator+=(int e);
				void operator-=(int e);
				operator int();
			private:
				Var(int _index, std::vector<Exp>::iterator _it,
									 std::vector<Exp>& exp);
				int index;
				std::vector<Exp>::iterator it;
				std::vector<Exp>& exp;
			friend class MonomialSparse;
		};

		class ConstVar {
			public:
				operator int();
			private:
				ConstVar(int _index, std::vector<Exp>::const_iterator _it,
									 const std::vector<Exp>& exp);
				int index;
				std::vector<Exp>::const_iterator it;
				const std::vector<Exp>& exp;
			friend class MonomialSparse;
		};


		/** Get the ith exponent. */
		Var operator[](int i);
		/** Get the ith exponent. */
		ConstVar operator[](int i) const;

	protected:
		MonomialSparse(); // Not implemented.
};

inline void
MonomialSparse::set_constant()
{
	exp.clear();
}

inline bool
MonomialSparse::is_constant() const
{
	return exp.empty();
}

inline int
MonomialSparse::degree() const
{
    int d = 0;
    for (size_t i = 0; i < exp.size(); ++i) { d += exp[i].e; }
    return d;
}

inline int
MonomialSparse::get_num_vars() const
{
	return n;
}

inline int
MonomialSparse::get_num_non_zeros() const
{
	return exp.size();
}

// Returns true if *this < m reverse lexicographically.
inline bool
MonomialSparse::operator<(const MonomialSparse& m) const
{
	size_t i = 0;
	while (i < exp.size() && i < m.exp.size()) {
		if (exp[i].i < m.exp[i].i) { return false; }
		if (exp[i].i > m.exp[i].i) { return true; }
		if (exp[i].e < m.exp[i].e) { return true; }
		if (exp[i].e > m.exp[i].e) { return false; }
		++i;
	}
	if (exp.size() < m.exp.size()) { return true; }
	if (exp.size() > m.exp.size()) { return false; }
	return false; // Monomial are the same.
}

inline bool
MonomialSparse::operator==(const MonomialSparse& m)
{
	assert(n = m.n);
	if (exp.size() != m.exp.size()) { return false; }
	for (size_t i = 0; i < exp.size(); ++i) {
		if (exp[i].i != m.exp[i].i || exp[i].e != m.exp[i].e) { return false; }
	}
	return true; // Monomials are equal.
}

// Returns a negative number if m1<m2, positive if m1>m2, and 0 if m1==m2.
// Lexicographic sorting.
inline int
MonomialSparse::cmp(const MonomialSparse& m1, const MonomialSparse& m2)
{
    int res;
	size_t i = 0;
	while (i < m1.exp.size() && i < m2.exp.size()) {
        res = m2.exp[i].i - m1.exp[i].i;
        if (res) { return res; }
        res = m1.exp[i].e - m2.exp[i].e;
        if (res) { return res; }
		++i;
	}
    res = (int) m1.exp.size() - (int) m2.exp.size();
	return res;
}

inline
void
MonomialSparse::build_index_map(int num_vars, int max_deg)
{
    index_map.resize(max_deg);
    for (int d = 0; d < max_deg; ++d) {
        index_map[d].resize(num_vars);
        for (int i = 0; i < num_vars; ++i) {
            index_map[d][i] = calculate_num_lte_d_monomials(i, d);
        }
    }
}

#if 0
// Index the monomials in a deg lex order.
inline
Index
MonomialSparse::get_index() const
{
    assert(!index_map.empty());
    Index index = 0;
    int d = degree();
    index += calculate_num_lte_d_monomials(n,d-1);
    int i = 0;
    for (size_t it = 0; it < exp.size(); ++it) {
        while (i < exp[it].i) {
            index += calculate_num_lte_d_monomials(n-1-i,d-1);
            ++i;
        }
        int e = exp[it].e;
        if (e < d) { index += calculate_num_lte_d_monomials(n-1-i,d-e-1); }
        d -= e; 
        ++i;
        if (d == 0) { break; }
    }
    return index;
}

// Index the monomials in a deg lex order.
inline
void
MonomialSparse::set_index(Index index)
{
    set_constant();
    int d = 0;
    while (index >= calculate_num_lte_d_monomials(n,d)) { ++d; }
    if (d == 0) { return; }
    index -= calculate_num_lte_d_monomials(n,d-1);
    for (Index i = 0; i < n; ++i) {
        if (d == 0) { break; }
        int e = d;
        while (e >= 0 && index >= calculate_num_lte_d_monomials(n-1-i,d-e)) {
            --e;
        }
        if (e < d) { index -= calculate_num_lte_d_monomials(n-1-i,d-e-1); }
        d -= e; 
        if (e != 0) { exp.push_back(Exp(i,e)); }
    }
}
#endif

#if 1
// Index the monomials in a deg rev lex order.
inline
Index
MonomialSparse::get_index() const
{
    Index index = 0;
    int d = degree();
    index += calculate_num_lte_d_monomials(n,d-1);
    for (size_t it = 0; it < exp.size(); ++it) {
        int i = exp[it].i;
        int e = exp[it].e;
        index += calculate_num_lte_d_monomials(n-1-i,d)
                -calculate_num_lte_d_monomials(n-1-i,d-e);
        d -= e; 
        if (d == 0) { break; }
    }
    return index;
}

// Index the monomials in a deg rev lex order.
inline
void
MonomialSparse::set_index(Index index)
{
    set_constant();
    int d = 0;
    while (index >= calculate_num_lte_d_monomials(n,d)) { ++d; }
    if (d == 0) { return; }
    index -= calculate_num_lte_d_monomials(n,d-1);
    for (Index i = 0; i < n; ++i) {
        if (d == 0) { break; }
        int e = 0;
        while (e < d && index >= calculate_num_lte_d_monomials(n-1-i,d)
                -calculate_num_lte_d_monomials(n-1-i,d-e-1)) { ++e; }
        index -= calculate_num_lte_d_monomials(n-1-i,d)
                -calculate_num_lte_d_monomials(n-1-i,d-e);
        d -= e; 
        if (e != 0) { exp.push_back(Exp(i,e)); }
    }
}
#endif

/////////////////////////////////////////////////////////////////////
// Below are the functions of the MonomialSparse iterators. //
/////////////////////////////////////////////////////////////////////

inline MonomialSparse::Var
MonomialSparse::operator[](int i)
{
	std::vector<Exp>::iterator it = exp.begin();
	while (it != exp.end() && it->i < i) { ++it; }
	Var v(i, it, exp);
	return v;
}

inline MonomialSparse::ConstVar
MonomialSparse::operator[](int i) const
{
	std::vector<Exp>::const_iterator it = exp.begin();
	while (it != exp.end() && it->i < i) { ++it; }
	ConstVar v(i, it, exp);
	return v;
}

inline
MonomialSparse::Var::Var(
					 int _index,
					 std::vector<Exp>::iterator _it,
					 std::vector<Exp>& _exp)
		: index(_index), it(_it), exp(_exp)
{
}

inline void
MonomialSparse::Var::operator=(int e)
{
	if (it != exp.end() && index == it->i) {
		if (e) { it->e = e; }
		else { exp.erase(it); }
	} else {
		if (e) {
			exp.insert(it, Exp(index, e));
		}
	}
}

inline void
MonomialSparse::Var::operator+=(int e)
{
	if (it != exp.end() && index == it->i) {
		it->e += e;
		if (it->e == 0) { exp.erase(it); }
	} else {
		if (e) {
			exp.insert(it, Exp(index, e));
		}
	}
}

inline void
MonomialSparse::Var::operator-=(int e)
{
	if (it != exp.end() && index == it->i) {
		it->e -= e;
		if (it->e == 0) { exp.erase(it); }
	} else {
		if (e) {
			exp.insert(it, Exp(index, e));
		}
	}
}

inline void
MonomialSparse::Var::operator++()
{
	if (it != exp.end() && index == it->i) {
		it->e++;
		if (it->e == 0) { exp.erase(it); }
	} else {
		exp.insert(it, Exp(index, 1));
	}
}

inline void
MonomialSparse::Var::operator--()
{
	if (it != exp.end() && index == it->i) {
		it->e--;
		if (it->e == 0) { exp.erase(it); }
	} else {
		exp.insert(it, Exp(index, -1));
	}
}

inline
MonomialSparse::Var::operator int()
{
	if (it != exp.end() && index == it->i) {
		return it->e;
	} else {
		return 0;
	}
}

inline
MonomialSparse::ConstVar::ConstVar(
					 int _index,
					 std::vector<Exp>::const_iterator _it,
					 const std::vector<Exp>& _exp)
		: index(_index), it(_it), exp(_exp)
{
}

inline
MonomialSparse::ConstVar::operator int()
{
	if (it != exp.end() && index == it->i) {
		return it->e;
	} else {
		return 0;
	}
}

#endif
