#ifndef _Polynomial_H_
#define _Polynomial_H_

#include <string>
#include <vector>
#include "Monomial.h"
#include "Term.h"
#include "Q.h"

template <class K>
class Polynomial {
	private:
		/** n is the number of variables in the Polynomial ring K[x_1,...,x_n] */ 
		const int n;

		/** vectors for storing term exponent information */
		std::vector<Term<K> > terms;	

		/** Prints a plus sign if v is not rational and negative. */
		void printplus(const K& v) const;

	public:
		/* Constructors. */
		Polynomial(int num_vars);
		Polynomial(const Polynomial& poly);
		Polynomial& operator=(const Polynomial& poly);

		/* destructor */
		~Polynomial();

        typedef typename std::vector<Term<K> >::iterator iterator;
        typedef typename std::vector<Term<K> >::const_iterator const_iterator;

		void addeq(const Term<K>& t);
        void addeq(const Polynomial& p);
        void muleq(const Term<K>& t);
        void muleq(const Polynomial& p);

        void mul(const Polynomial& p1, const Polynomial& p2);
        void add(const Polynomial& p1, const Polynomial& p2);

        void swap(Polynomial& p);

        int degree() const;
        
		/* get the total number of terms in the Polynomial */
		int getNumTerms() const { return (int) terms.size();};

		/* Get the i-th term. */
		const Term<K>& operator[](int) const;

		/** Print the Polynomial in maple format. */
		void print() const;
		/** Print the Polynomial in magma format. */
        void print_magma() const;
};

/* constructing a Polynomial */ 
template <class K> inline 
Polynomial<K>::Polynomial(int num_vars) : n(num_vars) { }

template <class K> inline 
Polynomial<K>::Polynomial(const Polynomial& p) : n(p.n), terms(p.terms) { }

template <class K> inline 
Polynomial<K>&
Polynomial<K>::operator=(const Polynomial& poly)
{
	assert(n == poly.n);
	terms = poly.terms;
	return *this;
}

/* default destructor */
template <class K> inline 
Polynomial<K>::~Polynomial() { }

/* Add a term to this polynomial. */
template <class K> inline 
void Polynomial<K>::addeq(const Term<K>& t)
{
	if (t.coeff == 0) { return; }
    //terms.push_back(t);
    typename std::vector<Term<K> >::iterator it = terms.begin();
    while (it != terms.end()) {
        int res = Monomial::cmp(it->mon, t.mon);
        if (res < 0) {
            terms.insert(it, t);
            return;
        }
        else if (res == 0)  { 
            it->coeff += t.coeff;
            if (it->coeff == 0) { terms.erase(it); }
            return;
        }
        ++it;
    }
    terms.push_back(t);
}

/* Add a polynomial to this polynomial. */
template <class K> inline 
void Polynomial<K>::addeq(const Polynomial& t)
{
    typename std::vector<Term<K> >::iterator it1 = terms.begin();
    typename std::vector<Term<K> >::const_iterator it2 = t.terms.begin();

    std::vector<Term<K> > res;

	while (it1 != terms.end() && it2 != t.terms.end()) {
        int cmp = Monomial::cmp(it1->mon, it2->mon);
		if (cmp > 0) { res.push_back(*it1); ++it1; }
		else if (cmp < 0) { res.push_back(*it2); ++it2; }
        else {
			it1->coeff += it2->coeff;
            if (it1->coeff != 0) { res.push_back(*it1); }
            ++it1; ++it2;
		}
	}
    res.insert(res.end(), it1, terms.end());
    res.insert(res.end(), it2, t.terms.end());
    terms.swap(res);
}

/* Multiply this polynomial by a term. */
template <class K> inline 
void Polynomial<K>::muleq(const Term<K>& t)
{
    for (size_t i = 0; i < terms.size(); ++i) { terms[i].mul(t); }
}

template <class K> inline
int Polynomial<K>::degree() const
{
    int d = -1;
    for (size_t i = 0; i < terms.size(); ++i) { 
        int d0 = terms[i].degree();
        if (d0 > d) { d = d0; }
    }
    return d;
}

/* Multiply this polynomial by a polynomial. */
template <class K> inline 
void Polynomial<K>::muleq(const Polynomial& t)
{
    Polynomial orig_poly(n);
    swap(orig_poly);
    for (const_iterator it = t.terms.begin(); it != t.terms.end(); ++it) {
        Polynomial poly(orig_poly);
        poly.muleq(*it);
        addeq(poly);
    }
}

template <class K> inline 
void Polynomial<K>::swap(Polynomial& p)
{
    assert(n == p.n);
    terms.swap(p.terms);
}

template <class K> inline 
const Term<K>&
Polynomial<K>::operator[](int i) const
{
	return terms[i];
}

template <class K> inline 
void
Polynomial<K>::print() const
{
   if (!terms.empty()) {
      for (size_t j = 0; j < terms.size(); j++) {
			if (j != 0) { printplus(terms[j].coeff); }
			terms[j].print();
      }
   }
}

template <class K> inline 
void
Polynomial<K>::print_magma() const
{
   if (!terms.empty()) {
      for (size_t j = 0; j < terms.size(); j++) {
			if (j != 0) { printplus(terms[j].coeff); }
			terms[j].print_magma();
      }
   }
}

template <class K>
inline void
Polynomial<K>::printplus(const K& ) const
{
	std::cout << "+";
}

template <>
inline void
Polynomial<Q>::printplus(const Q& v) const
{
	if (v >= 0) { std::cout << "+"; }
}

#endif
