#ifndef _Term_H_
#define _Term_H_

#include "Monomial.h"
#include "Q.h"

template <class K>
class Term {
	public:
		Term(int num_vars);
		Term(K c, int num_vars);
		Term(K c, const std::vector<int>& v);
		Term(K c, int num_vars, const int *v);
		Term(K c, const Monomial& m);
		Term(const Term& t);
		Term(const Monomial& t);
		Term& operator=(const Term& t);
		Term& operator=(const Monomial& t);
		~Term();

		/** Multiplies two terms. */
		static void mul(const Term& t1, const Term& t2, Term& res);

        /** Returns the degree of the term. **/
        int degree() const;

        /** Multiples t with this term. */
        void mul(const Term& t);

		/** Swap the contents of two terms. */
		void swap(Term& t);

		/** Print out the term. */
		void print() const;
		void print_magma() const;

		K coeff;
		Monomial mon;
};

template <class K> inline
Term<K>::Term(int num_vars) : coeff(1), mon(num_vars) {}

template <class K> inline
Term<K>::Term(K c, int num_vars) : coeff(c), mon(num_vars) {}

template <class K> inline
Term<K>::Term(K c, const std::vector<int>& v) : coeff(c), mon(v) {}

template <class K> inline
Term<K>::Term(K c, int num_vars, const int *v) : coeff(c), mon(num_vars, v) {}

template <class K> inline
Term<K>::Term(K c, const Monomial& m) : coeff(c), mon(m) {}

template <class K> inline
Term<K>::Term(const Term& t) : coeff(t.coeff), mon(t.mon) {}

template <class K> inline
Term<K>::Term(const Monomial& m) : coeff(1), mon(m) {}

template <class K> inline
Term<K>& Term<K>::operator=(const Term<K>& t)
{
	coeff = t.coeff;
	mon = t.mon;
	return *this;
}

template <class K> inline
Term<K>& Term<K>::operator=(const Monomial& m)
{
	coeff = 1;
	mon = m;
	return *this;
}

template <class K> inline
Term<K>::~Term<K>() {}

template <class K> inline
void Term<K>::mul(const Term<K>& t1, const Term<K>& t2, Term<K>& res)
{
	res.coeff = t1.coeff*t2.coeff;
	Monomial::mul(t1.mon, t2.mon, res.mon);
}

template <class K> inline
int Term<K>::degree() const
{
    if (coeff == 0) { return -1; }
    return mon.degree();
}

template <class K> inline
void Term<K>::mul(const Term<K>& t)
{
	coeff *= t.coeff;
	mon.mul(t.mon);
}

template <class K> inline
void Term<K>::swap(Term<K>& t)
{
	// Swap the coefficients.
	K tmp = coeff; coeff = t.coeff; t.coeff = tmp;
	// Swap the monomials.
	mon.swap(t.mon);
}

template <class K> inline
void Term<K>::print() const
{
	if (mon.is_constant()) { std::cout << coeff; }
	else {
		if (coeff != 1) { std::cout << coeff << "*"; }
		mon.print();
	}
}

template <> inline
void Term<Q>::print() const
{
	if (mon.is_constant()) { std::cout << coeff; }
	else {
		if (coeff != 1) {
			if (coeff == -1) { std::cout << "-"; }
			else { std::cout << coeff << "*"; }
		}
		mon.print();
	}
}

template <class K> inline
void Term<K>::print_magma() const
{
	if (mon.is_constant()) { std::cout << coeff; }
	else {
		if (coeff != 1) { std::cout << coeff << "*"; }
		mon.print_magma();
	}
}

template <> inline
void Term<Q>::print_magma() const
{
	if (mon.is_constant()) { std::cout << coeff; }
	else {
		if (coeff != 1) {
			if (coeff == -1) { std::cout << "-"; }
			else { std::cout << coeff << "*"; }
		}
		mon.print_magma();
	}
}

#endif
