#ifndef _LinearSystemF2_H
#define _LinearSystemF2_H

/* for time measurements */
#include <time.h>
/* for vector template */
#include <vector>
#include "F2.h"

/* for 64-bit arithmetic */
#include <inttypes.h>
#include <algorithm>
#include <cmath>
#include <fstream>

#include "TwoTuple.h"

/**
 * Creates a linear system Ax = b, where A is in compressed column form.
 * Uses gaussian elimination and exact arithmetic to solve.
 * The linear system is given in the form [A|-b]. i.e. the last column of the
 * constraint matrix is assumed to be the right hand side.
 */
template <>
class LinearSystem<F2> 
{
    private:
        /** number of columns in A */ 
        int n;
        /** number of rows in A */
        int m;

        typedef std::vector<int> Row;
        typedef std::vector<Row> Matrix;
        Matrix matrix;

        /** solution vector */
        std::vector<F2> soln;

        /** pivot rows */
        std::vector<int> pivots;

        void add_rows(int i, int j); // Add row i to j.
	    void add_rows(Row& ri, Row& rj); //Add rows ri and rj.
        void swap_rows(int i, int j); // Swap rows i and j.

        void diagonalise();

        /** if consistent, find actual solution */
        void back_solve2(); 

        void check_solution();

	void make_stripe(std::vector<std::vector<int> >& col_entries, int thisCol, int s,
			 std::vector<unsigned int>& stripe, std::vector<int>& stripe_row_indices);

	int find_stripe_basis(std::vector<unsigned int>& stripe2, std::vector<int>& stripe_row_indices,
			      std::vector<int>& independentRows, std::vector<int>& independentCols);

	void diagonalize_stripe_basis(std::vector<unsigned int>& stripe, std::vector<int>& independentRows,
				      std::vector<int>& independentCols, int stripeRank);

	void sort_by_gray_code(std::vector<unsigned int>& stripe, std::vector<int>& independentCols, int stripeRank,
			       std::vector<TwoTuple>& stripe_gray_codes);

	bool eliminate_stripe(std::vector<unsigned int>& stripe, std::vector<int>& stripe_row_indices,
			      std::vector<TwoTuple>& stripe_gray_codes, std::vector<std::vector<int> >& col_entries,
			      std::vector<int>& independentCols, int stripeRank);

        /** for time measurement purposes */
        double tic (void) {return (clock () / (double) CLOCKS_PER_SEC);};
        double toc (const double &t) {double s = tic ();return (s - t);};

    public:
        //A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
        
        /** preprocessing */
        void preprocessMatrix();

        /** public constructor */
        LinearSystem<F2>(int num_col);
		LinearSystem<F2>(const LinearSystem<F2> & ls2);

        /** public destructor */
        virtual ~LinearSystem<F2>();

        /** Get the solution vector. */
        const std::vector<F2>& getSolution();

        /** add entry (i,j) to the constraint matrix. */
        void addEntry(int i, int j);
        void addEntry(int i, int j, F2 v);
        bool addEntryIfNoExist(int i, int j); //true=added, false = element already exists

        /** add entry i to the right hand side. */
        void addRHSEntry(int i);
        void addRHSEntry(int i, F2 v);
        
        /** deletes all data from all the vectors **/
        void clearMatrix(); 

        /* get number of rows in matrix */ 
        int getNumRows() const {return m;}
        /* set the number of rows in matrix */ 
        void setNumRows(Index s) { m = s; matrix.resize(s); }
        /* get number of columns in matrix (not including rhs) */ 
        int getNumCols() const {return n-1;}
        /* set number of columns in matrix (not including rhs) */ 
        void setNumCols(int _n) { n = _n+1; }

        /* Get the number of non-zero entries. */
        long int getNumNonZeros() const;

		/* input: xVar, output: b such that Ax = b */
		void matrixVectorMultiplication(std::vector<int> & xVar, std::vector<int> & b);
		
		
        /** determine if the linear system is consistent */
        bool isConsistent();
		bool normal_ge();
		bool striped_ge();
        int rank();
        int rank_reverse();
        int rank_reverse(Index cutoff);

        /** Print the linear system. */
        void print() const;
		void print_triangle() const;
        void print_dense() const;

        /** Print the solution of the linear system. */
        void printSolution() const;

        int get_codim(Index start, Index end) const;
        
        /** update n and m **/
        void updateRowCol(int newRowSize, int newColSize);

        class RowIter
        {
        public:
            RowIter(const RowIter& ri) { it = ri.it; }
            RowIter& operator=(const RowIter& ri) { it = ri.it; return *this; }
            void operator++() { ++it; }
            bool operator!=(const RowIter& i) { return it != i.it; }
            bool operator==(const RowIter& i) { return it == i.it; }
            const Index& index() { return *it; }
            const F2 coeff() { return 1; }
        protected:
            Row::const_iterator it;
            RowIter(Row::const_iterator _it) { it = _it; }
            friend class LinearSystem<F2>;
        };

        RowIter begin(Index i) const { return RowIter(matrix[i].begin()); }
        RowIter end(Index i) const { return RowIter(matrix[i].end()); }
        RowIter last(Index i) const { return RowIter(--matrix[i].end()); }
};

inline
void
LinearSystem<F2>::swap_rows(int i, int j)
{
    matrix[i].swap(matrix[j]);
}


inline
LinearSystem<F2>::LinearSystem(int _n)
    : n(_n), m(0)
{
    assert(0 <= _n);
}


inline
LinearSystem<F2>::LinearSystem(const LinearSystem<F2> & ls2)
{
	matrix.resize(ls2.matrix.size());
	
    for(int i = 0; i < (int) ls2.matrix.size(); ++i)
    	matrix[i] = ls2.matrix[i];
	
	n = ls2.n;
	m = ls2.m;
	soln = ls2.soln;
	pivots = ls2.pivots;
}


inline
LinearSystem<F2>::~LinearSystem()
{
}

/**
 * Add an entry to the constraint matrix.
 * This function assumes that the same entry is not added twice.
 */
inline
void
LinearSystem<F2>::addEntry(int i, int j)
{
    assert(0 <= j && j < n);
    if (i >= m) {
        matrix.resize(i+1);
        m = i+1;
        matrix[i].push_back(j);
    }
    else {
        //matrix[i].push_back(j);
        Row::iterator it = std::lower_bound(matrix[i].begin(), matrix[i].end(), j);
        if (it != matrix[i].end() && *it == j) { matrix[i].erase(it); }
        else { matrix[i].insert(it, j); }
    }
}

inline
void
LinearSystem<F2>::addEntry(int i, int j, F2 v)
{
    if (v != 0) { addEntry(i,j); }
}

//true=added, false = element already exists
inline
bool
LinearSystem<F2>::addEntryIfNoExist(int i, int j)
{
	//if ( i >= matrix.size())
	//	std::cout << "OH MY GOSH:LS adEntryIfNoExist" << std::endl;
		
	if ( matrix[i].end() == find (matrix[i].begin(), matrix[i].end(), j) ) {
		addEntry(i, j);

	//debug stuff	
	//	if ( matrix[i].end() == find (matrix[i].begin(), matrix[i].end(), j) )
	//		cout << "ERROR\n";
			
		return true;
	}
	else
		return false;
}

/**
 * Add an entry to the right hand side.
 * This function assumes that the same entry is not added twice.
 */
inline
void
LinearSystem<F2>::addRHSEntry(int i)
{
    addEntry(i, n-1);
}

inline
void
LinearSystem<F2>::addRHSEntry(int i, F2 v)
{
    if (v != 0) { addEntry(i, n-1); }
}


inline
void
LinearSystem<F2>::clearMatrix()
{
	for(size_t i = 0; i < matrix.size(); ++i)
		matrix[i].clear();
		
	matrix.clear();
	soln.clear();
	pivots.clear();
	
	m = n = 0;
}//clearMatrix/



inline
long int
LinearSystem<F2>::getNumNonZeros() const
{
    long int num = 0;
    for (Matrix::const_iterator i = matrix.begin(); i!= matrix.end(); ++i) {
        num += i->size();
    }
    return num;
}



//input: xVar, output: b such that A*(xVar) = b 
//Assumes xVar is sorted.
inline
void
LinearSystem<F2>::matrixVectorMultiplication(std::vector<int> & xVar, std::vector<int> & b)
{
    int count, current = 0;
	
	b.clear();
	for(int i = 0; i < m; ++i)
	{
	    count = 0;
	    current = 0;
		for (int j = 0; j < (int) matrix[i].size() && current < (int) xVar.size() ; ++j)
		{
		    while ( current < (int) xVar.size()  && matrix[i][j] > xVar[current])
				++current;
			if (current < (int) xVar.size() && matrix[i][j] == xVar[current])
				++count;
		}
		if ( count % 2)
			b.push_back(i);
	}//for every row, find the dot product of matrix[i] and xVar.
		




}
		
		

inline
void
LinearSystem<F2>::preprocessMatrix()
{
#if 1
    std::vector<bool> zeros(n, false);
    for (int r = 0; r < m; ++r) {
        if (matrix[r].size() == 1) { zeros[matrix[r][0]] = true; }
    }
    for (int r = 0; r < m; ++r) {
        if (matrix[r].size()>1) {
            for (int i = matrix[r].size()-1; i >= 0 ; --i) {
                if (zeros[matrix[r][i]]) { matrix[r].erase(matrix[r].begin()+i); }
            }
        }
    }
#endif
}

/**
 * Adds two rows together.
 * Assumes that the entries in the rows are in sorted order.
 */
inline
void
LinearSystem<F2>::add_rows(int i, int j)
{
    assert(0 <= i && i < m);
    assert(0 <= j && j < m);
    Row& ri = matrix[i];
    Row& rj = matrix[j];
    add_rows(ri, rj);
}

/**
 * Adds two rows together.
 * Assumes that the entries in the rows are in sorted order.
 */
inline
void
LinearSystem<F2>::add_rows(Row& ri, Row& rj)
{
    static Row r;
    r.reserve(ri.size() + rj.size());
    r.resize(0);

    Row::iterator iti = ri.begin();
    Row::iterator itj = rj.begin();
    while (iti != ri.end() && itj != rj.end())
    {
        if (*iti < *itj) { r.push_back(*iti); ++iti; }
        else if (*itj < *iti) { r.push_back(*itj); ++itj; }
        else { ++iti; ++itj; }
    }

    r.insert(r.end(), itj, rj.end());

    rj = r;
    rj.insert(rj.end(), iti, ri.end());
}


inline
bool
LinearSystem<F2>::isConsistent()
{
    return striped_ge();
}

/**
 * Check whether the linear system is consistent.
 * Uses ordinary Gaussian Elimination
 */
inline
bool
LinearSystem<F2>::normal_ge()
{
    preprocessMatrix();

    std::vector<std::vector<int> > col_entries(n);
    for (int r = 0; r < m; ++r) {
        if (!matrix[r].empty()) {
            if (matrix[r].front() == n-1) { return false; }
            col_entries[matrix[r].front()].push_back(r);
        }
    }

    for (int c = 0; c < n; ++c) {
        // Choose the next row.
        std::vector<int>& col = col_entries[c];
        if (col.empty()) { continue; }
        int next = col.front();
        for (int i = 1; i < (int) col.size(); ++i) {
                if (matrix[col[i]].size() < matrix[next].size()) { next = col[i]; }
        }

        pivots.push_back(next);

        for (int i = 0; i < (int) col.size(); ++i) {
            int r = col[i];
            if (r != next) {
                add_rows(next, r);
                if (!matrix[r].empty()) {
                    // Check if system is inconsistent.
                    if (matrix[r].front() == n-1) { return false; }
                    // Update first non-zero entries data structure.
                    col_entries[matrix[r].front()].push_back(r);
                }
            }
        }
    }
    return true;
}

/**
 * Computes the rank of the matrix. Processes columns increasing order.
 */
inline
int
LinearSystem<F2>::rank()
{
    preprocessMatrix();
    //print();
    std::vector<std::vector<int> > col_entries(n);
    for (int r = 0; r < m; ++r) {
        if (!matrix[r].empty()) {
                col_entries[matrix[r].front()].push_back(r);
        }
    }

    for (int c = 0; c < n; ++c) {
        // Choose the next row.
        std::vector<int>& col = col_entries[c];
        if (col.empty()) { continue; }
        int next = col.front();
        for (int i = 1; i < (int) col.size(); ++i) {
                if (matrix[col[i]].size() < matrix[next].size()) { next = col[i]; }
        }

        pivots.push_back(next);

        for (int i = 0; i < (int) col.size(); ++i) {
            int r = col[i];
            if (r != next) {
                add_rows(next, r);
                if (!matrix[r].empty()) {
                    // Update first non-zero entries data structure.
                    col_entries[matrix[r].front()].push_back(r);
                }
            }
        }
    }
    return pivots.size();
}

/**
 * Computes the rank of the matrix.  Processes columns in decreasing order.
 */
inline
int
LinearSystem<F2>::rank_reverse()
{
    return rank_reverse(0);
}

/**
 * Computes the rank of the matrix.  Processes columns in decreasing order.
 * Assumes that the rows [0,...,cutoff] have already been processed.
 */
inline
int
LinearSystem<F2>::rank_reverse(Index cutoff)
{
    preprocessMatrix();
    //print();
    std::vector<std::vector<int> > col_entries(n);
    for (int r = 0; r < m; ++r) {
        if (!matrix[r].empty()) {
            col_entries[matrix[r].back()].push_back(r);
        }
    }

    for (int c = n-1; c >=0; --c) {
        // Choose the next row.
        std::vector<int>& col = col_entries[c];
        if (col.empty()) { continue; }
        int next = col.front();
        if (next >= cutoff) {
            for (int i = 1; i < (int) col.size(); ++i) {
                if (matrix[col[i]].size() < matrix[next].size()) { next = col[i]; }
            }
        }

        for (int i = 0; i < (int) col.size(); ++i) {
            int r = col[i];
            if (r != next) {
                add_rows(next, r);
                if (!matrix[r].empty()) {
                    // Update last non-zero entries data structure.
                    col_entries[matrix[r].back()].push_back(r);
                }
            }
        }
    }
    // We now remove empty rows from the matrix.
    Index j = 0;
    for (size_t i = 0; i < matrix.size(); ++i) {
        if (!matrix[i].empty()) { matrix[i].swap(matrix[j]); ++j; }
    }
    matrix.erase(matrix.begin()+j, matrix.end());
    m = j;

    //std::cout << "Diagonalising the matrix..." << std::endl;
    //diagonalise();

    return m;
}

// Assumes that rank_reverse has been called.
inline
void
LinearSystem<F2>::diagonalise()
{
    std::vector<int>  pivots(n, -1);
    for (int r = 0; r < m; ++r) { 
        if (!matrix[r].empty()) {
            pivots[matrix[r].back()] = r;
        }
    }
    std::vector<int> row_pivots;
    for (int c = 0; c < n; ++c) {
        if (pivots[c] == -1) { continue; }
        int r = pivots[c];
        row_pivots.clear();
        Row::iterator last = --matrix[r].end();
        for (Row::iterator it = matrix[r].begin(); it != last; ++it) {
            if (pivots[*it] != -1) { row_pivots.push_back(pivots[*it]); }
        }
        for (size_t i = 0; i < row_pivots.size(); ++i) {
            add_rows(row_pivots[i], r);
        }
    }
}

// Assumes that rank_reverse has just been called.
inline
int
LinearSystem<F2>::get_codim(Index start, Index end) const
{
    int dim = 0;
    for (size_t i = 0; i < matrix.size(); ++i) {
        if (matrix[i].back() >= start && matrix[i].back() < end) { ++dim; }
    }
    return (end-start) - dim;
}

/**
 * Computes the solution of the linear system.
 */
inline
const std::vector<F2>&
LinearSystem<F2>::getSolution()
{
    back_solve2();
    //check_solution();
    return soln;
}

/**
 * Computes the solution of the linear system.
 * This function assumes that the linear system is consistent and
 * that the matrix is in upper triangle form with no zero rows.
 */
inline
void
LinearSystem<F2>::back_solve2()
{
    soln.clear();
    soln.resize(n, 0);
    soln[n-1] = 1;
    for (std::vector<int>::reverse_iterator i = pivots.rbegin(); i != pivots.rend(); ++i) {
        Row& r = matrix[*i];
        int c = r.front();
        for (int j = 1; j < (int) r.size(); ++j) {
            soln[c] = (soln[r[j]] != soln[c]);
        }
    }
    soln.resize(n-1);
}

/**
 * Forms the stripe from thisCol-s+1 to thisCol in bit-packed form.
 * Uses col_entries to only find non-zero  rows.
 * Stores bit-packed rows in stripe.
 * Stores the matrix index of corresponding rows in stripe_row_indices.
 */
inline
void
LinearSystem<F2>::make_stripe(std::vector<std::vector<int> >& col_entries, int thisCol, int s,
			      std::vector<unsigned int>& stripe, std::vector<int>& stripe_row_indices)
{
    int row, k;
    stripe.clear();
    stripe_row_indices.clear();
    for (int col = thisCol; col >= thisCol - s + 1; --col) {
        for (int i = 0; i < (int) col_entries[col].size(); ++i) {
	    unsigned int stripeRow = 0;
	    row = col_entries[col][i];
	    while(!matrix[row].empty()) {
	        k = thisCol - matrix[row].back();        //k is the bit that should be set in the stripe.
		if (k >= s) {
		    break;
		}//if k is 
		matrix[row].pop_back();                      //remove matrix(rwo, col).
		stripeRow |= (1u << k);                      //set the stripe bit
	    }
	    stripe.push_back(stripeRow);                     //save the stripe row as an int
	    stripe_row_indices.push_back(row);               //save the stripe row index.
	}//for every row with a non-zero in column col.
    }//for col.
}

/**
 * Finds a basis for the given stripe.
 * Pivots basis to the top of stripe and stripe2.
 * independentRows stores the absolute rows that correspond to pivot rows.
 * independentCols stores the columns in the stripe, counting from the right.
 * Returns the rank of the stripe.
 */
inline
int
LinearSystem<F2>::find_stripe_basis(std::vector<unsigned int>& stripe2, std::vector<int>& stripe_row_indices,
				    std::vector<int>& independentRows, std::vector<int>& independentCols)
{
    int row, col;
    int row_density;
    int stripeRank = 0;
    unsigned int col_mask;
    static std::vector<unsigned int> stripe;

    independentRows.clear();//clear list of indep. rows and cols for the current strip
    independentCols.clear();

    stripe = stripe2;
    while(1) {
        row = -1;
	row_density = n + m;	//upper bound.
	
	//find non dense row to GE other rows in stripe.
	for (int i = stripeRank; i < (int) stripe.size(); ++i) {
	    if (stripe[i] != 0 && (int) matrix[stripe_row_indices[i]].size() < row_density) {
	        row = i;
		row_density = matrix[stripe_row_indices[i]].size();
	    }
	}
	
	//GE other rows in stripe.
	if (row >= 0) {
	    col_mask = (~(stripe[row] - 1u) & stripe[row]);
	    std::swap(stripe[row], stripe[stripeRank]);
	    std::swap(stripe2[row], stripe2[stripeRank]);
	    std::swap(stripe_row_indices[row], stripe_row_indices[stripeRank]);
	    for (row = stripeRank + 1; row < (int) stripe.size(); ++row) {
	        if (stripe[row] & col_mask) {
		    stripe[row] ^= stripe[stripeRank];
		}
	    }
	    independentRows.push_back(stripe_row_indices[stripeRank]);
	    col = 0;
	    while(col_mask > 1u) {
	        col_mask >>= 1;
		++col;
	    }
	    independentCols.push_back(col);
	    ++stripeRank;
	} else {
	    break;
	}
    }//for col. GE dense stripe.
    return stripeRank;
}

/**
 * Puts the first stripeRank rows of the stripe into row echelon form.
 */
inline
void
LinearSystem<F2>::diagonalize_stripe_basis(std::vector<unsigned int>& stripe, std::vector<int>& independentRows,
					   std::vector<int>& independentCols, int stripeRank)
{
    unsigned int col_mask;
    for (int i = 0; i < stripeRank; ++i) {
       col_mask = 1u << independentCols[i];
       for (int j = i + 1; j < stripeRank; ++j) {
	   if (stripe[j] & col_mask) {
	      add_rows(independentRows[i], independentRows[j]);
	      stripe[j] ^= stripe[i];
	   }
       }
    }
}

/**
 * Creates a list of pairs of rows and gray codes, sorted by gray code.
 */
inline
void
LinearSystem<F2>::sort_by_gray_code(std::vector<unsigned int>& stripe, std::vector<int>& independentCols, int stripeRank,
				    std::vector<TwoTuple>& stripe_gray_codes)
{
    unsigned int stripe_row;
    unsigned int basis_coeffs;
    unsigned int gray_code;
    unsigned int col_mask;
    /* Used for alternate computation method 
    int shift_amt;

    shift_amt = 1;
    while (shift_amt < stripeRank) {
        shift_amt <<= 1;
    }
    */

    stripe_gray_codes.clear();

    for (int i = stripeRank; i < (int) stripe.size(); ++i) {
        stripe_row = stripe[i];
	basis_coeffs = 0u;
	for (int j = 0; j < stripeRank; ++j) {
	    col_mask = 1u << independentCols[j];
	    if (stripe_row & col_mask) {
	        basis_coeffs |= (1u << j);
		stripe_row ^= stripe[j];
	    }
	}
	gray_code = 0u;
	while (basis_coeffs != 0) {
	    gray_code ^= basis_coeffs;
	    basis_coeffs >>= 1;
	}
	/* Alternate computation method
	gray_code = basis_coeffs;
	while (shift_amt > 0) {
	    gray_code ^= (gray_code >> shift_amt);
	    shift_amt >>= 1;
	}
	*/
	
	
	stripe_gray_codes.push_back(TwoTuple(i, gray_code));
    }//for i. Make the list of 2-tuples.
    
    sort(stripe_gray_codes.begin(), stripe_gray_codes.end(), compTwoTuple);
}

/**
 * Eliminates the given stripe, using the first stripeRank rows as a basis.
 * The cursor row is taken to be an all zero row.
 */
inline
bool
LinearSystem<F2>::eliminate_stripe(std::vector<unsigned int>& stripe, std::vector<int>& stripe_row_indices,
				   std::vector<TwoTuple>& stripe_gray_codes, std::vector<std::vector<int> >& col_entries,
				   std::vector<int>& independentCols, int stripeRank)
{
    int stripe_index;             // Index of the current row in stripe
    int abs_index;                // Index of the current row in matrix
    unsigned int col_mask;
    unsigned int row_bits;        // Bits of the current row in stripe
    unsigned int cursor_row_bits; // Bits of the cursor row (in stripe)
    static Row cursor_row;        // Sparse representation of cursor row outside of stripe

    cursor_row.clear();    
    cursor_row_bits = 0u; // Representation of the cursor row in the stripe

    for (int i = 0; i < (int) stripe_gray_codes.size(); ++i) {
        stripe_index = stripe_gray_codes[i].index; //stripe_index = index to row from the 2-tuple
	row_bits = stripe[stripe_index];
	abs_index = stripe_row_indices[stripe_index];
	Row& row = matrix[abs_index];
	if (cursor_row_bits != row_bits) {
	    for (int j = 0; j < stripeRank; ++j) {
	        col_mask = 1u << independentCols[j];
		if (col_mask & (cursor_row_bits ^ row_bits)) {
		    cursor_row_bits ^= stripe[j];
		    add_rows(matrix[stripe_row_indices[j]], cursor_row);
		}
	    }
	}//if. Get the cursor row equal to the current row.

	//stripe[i] ^= cursor_row_bits; // <-- we are never going to look at this again; is zeroed out
	add_rows(cursor_row, row);
	
	if (!row.empty()) {
	    if (row.back() < 0) { return false; }
	    col_entries[row.back()].push_back(abs_index);
	} // Update col_entries, and check for solutions.
    }
    return true;
}

/**
 * Uses Striped Gaussian Elimination to put the matrix in row echelon form.
 * This version of SGE does not do any column pivoting.
 * Aborts and returns false if system is infeasible. Otherwise returns true.
 */
inline
bool
LinearSystem<F2>::striped_ge()
{
/**
 * We will walk the columns from RIGHT to LEFT. This allows us to use vector.push_back() vs 
 *   vecotr.push_front(). This also makes working with the bits of ints easier.
 * Matrix is of the form [b, A].
 * Will convert [b, A] to [ b * * .. *  * 1 ]
 *                        [ b * * .. 1      ]
 *                        [ ...             ]
 *                        [ b 1             ]
 *                        [ 0               ]
 */

    int s = (int) (2.0 * log2(m));              // This should be an upper bound on s (chosen below)
    if (s < 1) s = 1;
    
    int thisCol = n - 2;                        // current col
    std::vector<int> independentRows(s);        //list of rows.
    std::vector<int> independentCols(s);        //list of cols.
    std::vector<unsigned int> stripe(m);        //dense strip stored in the bits of an int.
    std::vector<TwoTuple> stripe_gray_codes(m); //list of (index, gray_code) tuples.
    std::vector<int> stripe_row_indices(m);     //stripe_row_indicies(r) = index of matrix row stored in stripe(r).
                                                //  This is a parallel vector to stripe.
    int stripeRank;                             //how many lin. independent rows do we have in the current stripe?
    int row, i, j;                              //for loop var's.
    unsigned int col_mask;			            //for bit shifting things.
    int rank = 0; //number of linearly independent rows found so far


    for (row = 0; row < m; ++row) {
        if (!matrix[row].empty() && matrix[row].back() == n - 1) {
	    matrix[row].pop_back();
	    matrix[row].insert(matrix[row].begin(), -1);
	}
    }// convert the matrix from [A, b] to [b, A].

    std::vector<std::vector<int> > col_entries(n); // col_entries(c) = vector of row indices for col c.
    for (row = 0; row < m; ++row) {
        if (!matrix[row].empty()) {
	    if (matrix[row].back() < 0) { return false; }
            col_entries[matrix[row].back()].push_back(row);
        }
    }//set col_entries to the row index of the first non-zero element in the column.

    //std::cout << "s: " << s << "\n";

    //start of SGE	
    while (thisCol >= s) {

        s = (int) (2.0 * log2(m - rank));
	if (s <= 1) break;

        // Find up to s independent rows and columns in the current stripe

        // (Construct dense stripe representation)

	make_stripe(col_entries, thisCol, s, stripe, stripe_row_indices);

	//Now, stripe is a dense representaton for the matrix.

	// (Gaussian elimination on dense stripe)

	stripeRank = find_stripe_basis(stripe, stripe_row_indices, independentRows, independentCols);
	rank += stripeRank;

	// Gaussian elimination on stripeRank-width band

	diagonalize_stripe_basis(stripe, independentRows, independentCols, stripeRank);

	// Elimination only if there are rows to eliminate
	if ((int) stripe.size() == stripeRank) {
	    thisCol -= s;
	    continue;
	}

        // Sort remaining rows by Gray code
	
	sort_by_gray_code(stripe, independentCols, stripeRank, stripe_gray_codes);

        //(Eliminate rows)

	if(!eliminate_stripe(stripe, stripe_row_indices, stripe_gray_codes, col_entries, independentCols, stripeRank)) {
	    return false;
	}

	// Reconstruct sparse entries for the s by s block

	for (i = 0; i < stripeRank; ++i) {
	    unsigned int stripe_row = stripe[i];
	    col_mask = 1u;
	    for (j = 0; j < s; ++j) {
	        if (stripe_row & col_mask) {
		    matrix[independentRows[i]].push_back(thisCol-j);
		}
		col_mask <<= 1;
	    }
	}

	// Increment looping variables
	thisCol -= s;

    }

    //print_triangle();

    // Gaussian elimination on remainder
    while (thisCol >= 0) {
        if (col_entries[thisCol].empty()) {
	    --thisCol;
	    continue;
	}

	// Eliminate other rows using col_entries[thisCol][0]
	int row_choice = -1;
	int row_density = n + m;	//upper bound.
	std::vector<int>& colRef = col_entries[thisCol];
	for(i = 0; i < (int) colRef.size(); ++i) {
	    if( (int) matrix[colRef[i]].size() < row_density)
	    {
	    	row_choice = colRef[i];
		row_density = (int) matrix[colRef[i]].size();
	    }
        }//for i. find least dense row.
	
	int pivot_row = row_choice; //set the pivot_row to the least dense row.
	for (i = 0; i < (int) col_entries[thisCol].size(); ++i) {
	    row = col_entries[thisCol][i];
	    if (row != pivot_row) {
	   	add_rows(pivot_row, row);
	    	if (!matrix[row].empty()) {
	            if (matrix[row].back() < 0) { return false; }
	            col_entries[matrix[row].back()].push_back(row);
		}
	    }
	}
	--thisCol;
    }

    //print_triangle();

    return true;
    // Check for feasibility

}

inline
void
LinearSystem<F2>::check_solution()
{
    soln.resize(n,1);
    for (int i = 0; i < m; ++i) {
        F2 v = 0;
        for (Row::iterator it = matrix[i].begin(); it != matrix[i].end(); ++it) {
            v += soln[*it];
        }
        if (v != 0) {
           std::cerr << "ERROR: Check solution failed!\n";
           exit(1);
        }
    }
    soln.resize(n-1);
}

// Print the linear system in sparse format.
inline
void
LinearSystem<F2>::print() const
{
    std::cout << m << " " << n << "\n";
    for (int i = 0; i < m; ++i) {
        const Row& r = matrix[i];
        if (!r.empty()) {
            std::cout << i << ":";
            for (size_t j = 0; j < r.size(); ++j) {
                std::cout << " " << r[j];
            }
            std::cout << "\n";
        }
    }
    std::cout << std::endl;
}

inline
void
LinearSystem<F2>::print_triangle() const
{
    std::cout << m << " " << n << "\n";
    int col = -1;
    while(col < n) {
      for (int i = 0; i < m; ++ i) {
	if (!matrix[i].empty() && matrix[i].back() == col) {
	  /*
	  const Row& r = matrix[i];
	  Row::const_iterator itr = r.begin();
	  for (int j = -1; j < n - 1; ++j) {
	    if (itr == r.end() || j < *itr) { std::cout << " 0"; }
	    else { std::cout << " 1"; ++itr; }
	  }
	  std::cout << "\n";
	  */
	  std::cout << i << ":";
	  for (size_t j = 0; j < matrix[i].size(); ++j) {
	    std::cout << " " << matrix[i][j];
	  }
	  std::cout << "\n";
	  
	}
      }
      ++col;
    }
    std::cout << std::endl;
}

// Print the linear system in dense format.
inline
void
LinearSystem<F2>::print_dense() const
{
    for (int i = 0; i < m; ++i) {
        const Row& r = matrix[i];
        Row::const_iterator itr = r.begin();
        for (int j = 0; j < n; ++j) {
             if (itr == r.end() || j < *itr) { std::cout << " 0"; }
             else { std::cout << " 1"; ++itr; }
        }
        std::cout << "\n";
    }
}


inline
void
LinearSystem<F2>::printSolution() const
{
    for (size_t i = 0; i < soln.size(); ++i) { std::cout << " " << soln[i]; }
    std::cout << "\n";
}



/** update n and m, clear the matrix, and resize the vectors. **/
inline
void
LinearSystem<F2>::updateRowCol(int newRowSize, int newColSize)
{

	if( newRowSize >= 0 && newColSize >= 0)	{
		clearMatrix();
		m = newRowSize;
		n = newColSize;
		
		matrix.resize(m);
	}//if.
	
}//updateRowCol()
#endif

