#include <string.h>
# include "matrix.h"
//# define DEBUG
# include "debug.h"
# include "utilities.combinatorics.h"
# include "utilities.linalg.h"
# include "exp.h"

/*******************************************************************
* THIS FILE CONTAINS:
* member functions for the "matrix" data type
********************************************************************/


//CONSTRUCTORS


/*******************************************************************
* purpose: constructs an empty matrix from an input number of d rows and n columns
* date: 06.29.98
* notes/explanation: 
********************************************************************/

matrix :: matrix (int num_rows, int num_cols) {

  COLS = num_cols;
  ROWS = num_rows;
  ENTRIES = new double * [ROWS];

  for (int i = 0; i < COLS; i++) {               //ALLOCATE SPACE FOR THE MATRIX ENTRIES

    ENTRIES[i] = new double [COLS];
  }

  ALL_DETS = new int [(long) choose(COLS, ROWS)];  //ALLOCATE THE APPROPRIATE SPACE FOR 
                                                   //THE DETERMINANTS
  COCIRCUITS = new int * [(long) choose(COLS, ROWS - 1)];  //ALLOCATE THE APPROPRIATE 
                                                           //SPACE FOR COCIRCUITS
}

/*******************************************************************
* purpose: constructs a matrix from an input number of d rows and n columns and a d by
*          n array of floating point elements.
* date: 05.25.98
* notes/explanation: 
********************************************************************/
 
matrix :: matrix (int num_rows, int num_cols, double ** entries) {

  COLS = num_cols;
  ROWS = num_rows;
  ENTRIES = new double * [ROWS];

  for (int i = 0; i < ROWS; i++) {            //FILL THE MEMBER ARRAY "ENTRIES" WITH THE
                                              //INPUT ELEMENTS.
    ENTRIES[i] = new double [COLS];           

    for (int j = 0; j < COLS; j++) {

      ENTRIES[i][j] = entries[i][j];
    }
  }

  ALL_DETS = new int [(long) choose(COLS, ROWS)];   //ALLOCATE THE ARRAY TO BE FILLED 
                                                    //WITH ALL POSSIBLE N CHOOSE D 
                                                    //DETERMINANTS
  COCIRCUITS = new int * [(long) choose(COLS, ROWS - 1)]; //ALLOCATE THE ARRAY TO BE 
                                                           //FILLED WITH ALL N CHOOSE
                                                           //D - 1 COCIRCUITS
}


/*******************************************************************
* purpose: constructs an exact matrix from a file
* date: 05.25.98
* notes/explanation: the input file should have the number of rows first, followed by the
*                    number of columns and then the number of non-zero entries.
*                    Values for the entries should be preceded by the row number and 
*                    column number for that entry.
********************************************************************/

matrix :: matrix(char* filename, int) {

  Matrix m(filename, NULL);                //CONSTRUCT AN EXACT MATRIX FROM WNK'S CODE
  COLS = m.cols;
  ROWS = m.rows;
  ENTRIES = new double * [ROWS];
  double **entries = m.genMatrix();

  for (int i = 0; i < ROWS; i++) {            //FILL THE MEMBER ARRAY "ENTRIES" WITH THE
                                              //INPUT ELEMENTS.
    ENTRIES[i] = new double [COLS];           

    for (int j = 0; j < COLS; j++) {

      ENTRIES[i][j] = entries[i][j];
    }
  }

  ALL_DETS = new int [(long) choose(COLS, ROWS)];   //ALLOCATE THE ARRAY TO BE FILLED 
                                                    //WITH ALL POSSIBLE N CHOOSE D 
                                                    //DETERMINANTS
  COCIRCUITS = new int * [(long) choose(COLS, ROWS - 1)]; //ALLOCATE THE ARRAY TO BE 
                                                           //FILLED WITH ALL N CHOOSE
                                                           //D - 1 COCIRCUITS
}

/*******************************************************************
* purpose: constructs a matrix from a file
* date: 05.25.98
* notes/explanation: the input file should have the number of rows first, followed by the
*                    number of columns, and then the entries listed by all the elements 
*                    of the first row, followed by all the elements of the second row,
*                    etc.
********************************************************************/

matrix :: matrix (char * filename) {

  ifstream in (filename);

  if (!in) {                                        
    cerr << "Could not open " << filename << endl;  //IF THE INPUT FILE DOES NOT EXIST
  }

  in >> ROWS >> COLS;                               //READ THE NUMBER OF ROWS AND COLS

  ALL_DETS = new int [(long) choose(COLS, ROWS)];   //ALLOCATE THE ARRAY TO BE FILLED 
                                                    //WITH ALL POSSIBLE N CHOOSE D 
                                                    //DETERMINANTS
  COCIRCUITS = new int * [(long) choose(COLS, ROWS - 1)]; //ALLOCATE THE ARRAY TO BE 
                                                           //FILLED WITH ALL N CHOOSE
                                                           //D - 1 COCIRCUITS
  ENTRIES = new double * [ROWS];                    //ALLOCATE SPACE FOR THE "ENTRIES",
                                                    //INDEXED FIRST BY THE ROWS

  for (int i = 0; i < ROWS; i++) {

    ENTRIES[i] = new double [COLS];                 //ALLOCATE EACH ROW AS A LIST OF COLS

    for (int j = 0; j < COLS; j++) {

      in >> ENTRIES[i][j];                          //FILL THE MEMBER ARRAY "ENTRIES" 
                                                    //WITH THE DATA FROM THE FILE
    }
  }
}



//DESTRUCTOR


/*******************************************************************
* purpose: destructor for a matrix
* date: 05.25.98
* notes/explanation: 
********************************************************************/

matrix :: ~matrix() {

  delete [] ENTRIES;
  delete [] COCIRCUITS;
  delete [] ALL_DETS;
  
}


/*******************************************************************
* purpose: destroy the allocated space for the entries
* date: 05.25.98
* notes/explanation: 
********************************************************************/

void matrix :: delEntries() {

  delete [] ENTRIES;
  return;

}


/*******************************************************************
* purpose: destroy the allocated space for the determinants
* date: 05.25.98
* notes/explanation: 
********************************************************************/

void matrix :: delAllDets() {

  delete [] ALL_DETS;
  return;

}



//OPERATORS


/*******************************************************************
* purpose: a layer of abstraction by which matrices may be output.
* date: 05.25.98
* notes/explanation: overrides the "<<" operator
********************************************************************/

ostream & operator << (ostream & a_stream, const matrix & mtrx) {

  for (int i = 0; i < mtrx.ROWS; i++) {

    a_stream << "| ";

    for (int j = 0; j < mtrx.COLS; j++) {

      printf("%5f", mtrx.ENTRIES[i][j]);
      a_stream << " ";
    }

    a_stream << "|" << endl;
  }

  return a_stream;
}




//DATA ACQUISITION


/*******************************************************************
* purpose: accesses the private data member for the number of rows in the matrix
* date: 05.25.98
* notes/explanation: 
********************************************************************/

int matrix :: rows () {

  return ROWS;

}


/*******************************************************************
* purpose: accesses the private data member for the number of columns in the matrix
* date: 05.25.98
* notes/explanation: 
********************************************************************/

int matrix :: cols () {

  return COLS;

}


/*******************************************************************
* purpose: accesses a desired element of the matrix indexed by the row and then the col
* date: 05.25.98
* notes/explanation: 
********************************************************************/

double matrix :: entry (int row, int col) {

  if ((row > ROWS) ||
      (row < 1)) {                          //IF THE GIVEN ROW DOES NOT EXIST
    cerr << "Invalid row number. \n";
    return 0;
  }

  if ((col > COLS) ||
      (col < 1)) {                          //IF THE GIVEN COLUMN DOES NOT EXIST
    cerr << "Invalid column number. \n";
    return 0;
  }      

  return ENTRIES[row - 1][col - 1];
}


/*******************************************************************
* purpose: accesses the private data member of the sign of the determinant for the given
*          index of the member array "ALL_DETS"
* date: 05.25.98
* notes/explanation: 
********************************************************************/

int matrix :: get_det_sign (int place) {

  return ALL_DETS[place];

}


/*******************************************************************
* purpose: accesses the private data member for the cocircuit list from the given index
*          of the double member array "COCIRCUITS"
* date: 06.10.98
* notes/explanation: 
********************************************************************/

int * matrix :: get_cocircuit (int which) {

  return COCIRCUITS[which];

}


/*******************************************************************
* purpose: creates and returns a square matrix consisting of the specified sub_rows and
*          sub_columns of the larger matrix
* date: 05.25.98
* notes/explanation: note that the input arrays sub_rows and sub_cols must both have 
*                    number of elements equal to the input size.  If the sub_rows or 
*                    sub_cols are null (0), then the function uses all rows or columns of
*                    the larger matrix respectively.
********************************************************************/

void matrix :: fillSquareMatrix (int * sub_rows, int * sub_cols, int size, 
				     matrix * square_matrix) {

  if (!sub_rows) {                                 //IF SUB_ROWS IS NULL, USE ALL ROWS

    for (int i = 0; i < size; i++) {

      for (int j = 0; j < size; j++) {             //FILL THE ARRAY WITH ELEMENTS FROM
	                                           //THE CORRESPONDING ROWS AND COLS OF 
	                                           //THE LARGER MATRIX.
	square_matrix->ENTRIES[i][j] = ENTRIES[i][sub_cols[j] - 1];
      }
    }
  }

  if (!sub_cols) {                                 //IF SUB_COLS IS NULL, USE ALL COLS

    for (int i = 0; i < size; i++) {

      for (int j = 0; j < size; j++) {             //FILL THE ARRAY WITH ELEMENTS FROM
	                                           //THE CORRESPONDING ROWS AND COLS OF 
	                                           //THE LARGER MATRIX.
		square_matrix->ENTRIES[i][j] = ENTRIES[sub_rows[i] - 1][j];
      }
    }
  }

  return;
}



//MODIFIERS


/*******************************************************************
* purpose: places the given cocircuit equation into the input slot of the private data 
*          member array "COCIRCUITS"
* date: 06.10.98
* notes/explanation: 
********************************************************************/

void matrix :: put_cocircuit (int where, int * cocirc) {

  COCIRCUITS[where] = cocirc;

}
  

/*******************************************************************
* purpose: changes the entry index by the row and column number to the input new entry
* date: 06.10.98
* notes/explanation: 
********************************************************************/


void matrix :: changeEntry (int row, int col, double new_entry) {

  ENTRIES[row - 1][col - 1] = new_entry;

}


/*******************************************************************
* purpose: places the given determinant sign (+1, -1, or 0) into the input slot of the 
*          private data member "ALL_DETS"
* date: 06.10.98
* notes/explanation: 
********************************************************************/

void matrix :: enter_det_sign (int place, int det_sign) {

  ALL_DETS[place] = det_sign;

}

