# include "utilities.combinatorics.h"
//# define DEBUG
# include "debug.h"

/*********************************************************
* THIS FILE CONTAINS:
* subroutines for the manipulation of binomial coefficients
* and d-tuples as elements of the reverse lexicographic order.
**********************************************************/


/*********************************************************
* purpose: computes the Logarithm of the Gamma Function.
* date: 06.08.98
* notes/explanation: Taken from the book Numerical recipes in C
* it is based on an approximation formula, as a power series, 
* that uses fixed coefficients. They are provided below.
*********************************************************/

double gammln (double xx) {

  static double cof[6] = {76.18009173, -86.50532033, 24.01409822, -1.231739516,
			  0.120858003e-2, -0.536382e-5};

  double x = xx - 1.0;
  double tmp = x + 5.5;
  tmp -= (x + 0.5) * log(tmp);
  double ser = 1.0;

  for (int j = 0; j <= 5; j++) {

    x += 1.0;
    ser += cof[j] / x;
  }

  return -tmp + log(2.50662827465 * ser);
}



/*********************************************************
* purpose: Computes the Logarithm of the factorial n!
* date: 06.08.98
* notes/explanation: Based on the fact that  n!= Gamma(n+1).
* When dealing with binomial coefficients, often one has overflow
* when using the straightforward formula for n! (n*n-1*n-2...*1).
* so we must use the logarithm.
*********************************************************/

double factln (int n) {

  static double a[101];
  
  if (n < 0) {
    cerr << "factln called with negative n. \n";
    return 0;
  }

  if (n <= 1) {
    return 0.0;
  }

  if (n <= 100) {
    return a[n] ? a[n] : (a[n] = gammln(n + 1.0));
  }

  else return gammln(n + 1.0);

}

/*********************************************************
* purpose: Computes the binomial coefficient (n choose k) as a floating-point
* number.
* date: 06.08.98
* notes/explanation: 
*********************************************************/

double choose (int n, int k) {

  return floor(0.5 + exp(factln(n) - factln(k) - factln(n - k)));

}


/*********************************************************
* purpose: Given an integer p and an empty integer array, the function fills the integer
*          array with the the pth ntuple element with respect to reverse lex order 
* date: 06.08.98
* notes/explanation: The desired n-tuple can be read off 
* from a greedy decomposition of p as the sum of binomial
* coefficients. See Kruskal-Katona theorem ( Ziegler's book
* Lectures on Polytopes chapter 8) for details.
*********************************************************/

bool encode_p (int m_max, int n_max, double p, int * p_code) {

  /*NOTE THAT IN THE FOLLOWING LOOP, THE P_CODE ARRAY IS FILLED FROM END TO START*/

  for (int i = 1; i <= n_max; i++) {   //FOR EACH OF THE N_MAX ELEMENT IN P_CODE, FIND
                                       //THE CORRESPONDING INTEGER (BETWEEN 1 AND M_MAX

    if ((p_code[n_max - i] = find_np_code(m_max, n_max - i + 1, p)) < 0) {
                                       //IF FIND_NP_CODE RETURNS A NUMBER LESS THAN ZERO
                                       //FOR THE CURRENT PLACE IN P_CODE, AN ERROR HAS
                                       //BEEN MADE, SO WE RETURN NULL.
      return false;
    }

    if (p > 0.0) {                     //IF P STILL HAS VALUE, REDUCE IT FURTHER BY THE
                                       //CURRENT P_CODE ELEMENT CHOOSE THE CURRENT P_CODE
                                       //SLOT PLUS 1.
      p -= choose(p_code[n_max - i], n_max - i + 1);

      if (p > 0.0) {          
	p_code[n_max - i]++;           //IF P STILL HAS VALUE, INCREMENT THE CURRENT
                                       //ELEMENT BY ONE
      }

      else {
	p = 0.0;                       //OTHERWISE, SET P TO EXACTLY ZERO.
      }
    }

    m_max = p_code[n_max - i] - 1;     //REDUCE M_MAX BY ONE FOR THE NEXT ELEMENT
  }

  return true;
}


/*********************************************************
* purpose: Finds the maximal m that makes the
* binomial coefficient (m choose n) fit inside a given number p.
* date: 06.08.98
* notes/explanation: 
*********************************************************/

int find_np_code (int m_max, int n, double p) {

  double temp_result;

  if (p == 0) {             
    return m_max;                          //IF P IS ALREADY COMPLETELY REDUCED, THEN
                                           //M_MAX IS THE DEFAULT CODE
  }

  /*WE LOOP FROM THE CURRENT N TO M_MAX UNTIL WE FIND THE BEST BINOMIAL COEFFICIENT, THAT
    IS, THE ONE WHICH REDUCES P THE MOST WITHOUT GOING UNDER ZERO. */
  for (int coeff_cand = n; coeff_cand <= m_max; coeff_cand++) {

    temp_result = choose(coeff_cand, n);   //THE RESULT IS COMPUTED FOR THE CURRENT COEFF
                                           //CANDIDATE

    if (temp_result > p) {
      return coeff_cand - 1;               //IF THIS CANDIDATE WOULD REDUCE P BELOW ZERO
    }                                       //THEN THE PREVIOUS CANDIDATE WAS THE BEST.

    if (temp_result == p) {
      return coeff_cand;                   //IF THIS CANDIDATE REDUCES P TO EXACTLY ZERO
    }                                       //THEN THIS CANDIDATE IS BEST.
    
  }

  PRINTPLACE("Could not find np_code. \n");  //IF THE LOOP WAS TRAVERSED WITHOUT A VALID
  return -1;                                 //RESULT, THEN AN ERROR WAS MADE AND WE    
                                             //RETURN THE INVALID SOLUTION -1.
}


/*********************************************************
* purpose: finds the reverse lex order of the input ntuple
* containing d elements.
* date: 06.12.98
* notes/explanation: this is the inverse of encode_p 
*********************************************************/

double from_ntuple_to_p (int * ntuple, int d) {

  int temp = 1;         //KEEPS TRACK OF THE LAST CONSECUTIVE MEMBER OF NTUPLE
  double result = 0;    

  if (!(ntuple)) {      //IF NTUPLE IS NULL
    return 0;
  }

  /* LOOK FOR ANY CONSECUTIVITY (ONLY HAPPENS WHEN THE FIRST ELEMENT OF THE NTUPLE IS 
     A ONE */
  for (int i = 0; i < d; i++) {

    if (temp != ntuple[i]) {
      break;
    }

    temp++;
  }

  temp--;
  result += choose(ntuple[temp - 1], temp);  //THE LAST CONSECUTIVE MEMBER IS 
                                             //THE FIRST CONTRIBUTOR TO RESULT

  /* CONTINUALLY INCREMENT RESULT BY THE NTUPLE ELEMENT - 1 CHOOSE D - THE ELEMENT NUMBER
     */
  for (int i = 0; i <= d - temp - 1; i++) {

    result += choose(ntuple[d - i - 1] - 1, d - i);
  }

  return result;
}



/*************************************************************
* purpose: Given an array of d+1 numbers with the first d already
* ordered, returns the position of the (d+1)th point with respect to
* the others. Obviously, this gives the parity of a reorder permutation!!!
* date: 06.08.98
* notes/explanation: 
*********************************************************/

int parity_of_permutation_reorder (int newvalue, int * a_list, 
				   int beginning_list, int end_list) {
  int half_length;

  if (end_list == beginning_list) {  // array has length one!! easy to decide                                      //what is the position of newvalue.
    if (a_list[beginning_list] < newvalue)
      return (beginning_list+1);
    else {
      if (a_list[beginning_list] == newvalue)
	return -2;                              // IN THIS CASE newvalue is  repeated!! 
      else
	return (beginning_list);
    }
  }
  else{                             // array has length bigger than one,     
                                    //we break it into two arrays.
    if (newvalue > a_list[end_list])
      return (end_list+1);
    else{ 
      if (newvalue == a_list[end_list]) 
	return -2;                            // IN THIS CASE newvalue is  repeated!! 
      else{ 
	half_length =static_cast<int>(floor((beginning_list+end_list-1)/2));
	if (a_list[half_length]< newvalue )
	  return parity_of_permutation_reorder(newvalue,a_list,half_length+1,end_list);
	else{
	  if (a_list[half_length]==newvalue){
	    return -2;      }               // IN THIS CASE newvalue is  repeated!! 
	  
	  else
	    return parity_of_permutation_reorder(newvalue,a_list,beginning_list,half_length);
	}
      }
    }
  }
}


/*********************************************************
* purpose: Returns a new list containing the old list with the newvalue added 
*          into the correct place.
* date: 06.13.98
* notes/explanation: Straightforward
*********************************************************/

int * place_new_value_in_list(int * tuple, int place, int newvalue,
			      int length_tuple, int * newtuple) {

  for (int j=0; j< place; j++)
    newtuple[j]=tuple[j];
  newtuple[place]=newvalue;
  for (int j=place; j<length_tuple; j++)
    newtuple[j+1]=tuple[j];
  return newtuple;
}


/*********************************************************
* purpose: Generates the cocircuit equations for the given matrix of n columns
*          and d rows and prints the equations into the two input files
* date: 06.13.98
* notes/explanation: This function writes the cocircuit equations in cplex 
* format to the file whose name is filename, based on whether opt is "Minimize"
* or "Maximize."  It also writes the signs of the determinants for the
* cocircuits into the COCIRCUITS array of the matrix.  
*********************************************************/

void cocircuit_equations_for_universal_polytope (matrix * mtrx, 
						  ofstream& file1, 
						  ofstream& file2) {

 int count = 0, encripted_newtuple, ki, temp, negs, pos, me_printed_first,
     n = mtrx->cols(), d = mtrx->rows();
 int tuple[20], newtuple[20];
 int * circ_temp; 
 char eq_temp[100][50];
 
 cout << "Generating cocircuit equations...\n";

 double how_many = choose(n, d-1);  //THESE VARIABLES ARE USED TO GIVE PROGRESS REPORTS
 int update = 5; 
 
 for (int j=1; j< how_many + 1; j++) { //FOR EACH (D-1)-TUPLE...
   
   /* GIVE A PROGRESS REPORT EVERY 5% OF THE WAY */
   if (((double) j / how_many) > ((double) update / (double) 100)) {
      cout << update << " percent finished.\n";
      update += 5;
    }

   negs = pos = count = 0;                  //NO COCIRCUITS YET 
   me_printed_first = 1;                    //RESET THE FIRST PRINTED FLAG
   circ_temp = new int [n];                 //ALLOCATE SPACE FOR THE COCIRCUIT
   encode_p(n,d-1,j,tuple);                 //THE TUPLE IS OBTAINED FROM j
   
   for (int i=1; i<n+1 ; i++) {             //FOR EACH COLUMN OF THE MATRIX... 
    
     ki=parity_of_permutation_reorder(i,tuple,0,d-2);  //GET THE PARITY OF THE
                                                       //TUPLE WITH THE COLUMN

     if (ki > -2) {                         //IF THE COLUMN IS NOT A REPEAT OF 
                                            //AN ELEMENT OF THE TUPLE...

       place_new_value_in_list(tuple,ki,i,d-1,newtuple); //GET THE D-TUPLE
       encripted_newtuple=(int) from_ntuple_to_p(newtuple,d); //DECODE THE
                                                              //D-TUPLE
       temp = mtrx->get_det_sign(encripted_newtuple - 1) * //THE SIGN FOR THE 
	      (int) pow(-1, d - ki);        //COCIRCUIT ELEMENT IS EQUAL TO THE
                                            //SIGN OF THE DETERMINANT TIMES -1
                                            //RAISED TO THE PARITY OF THE COL

       if (temp == 1) {
	 if (me_printed_first) {               //IF THE ELEMENT IS POSITIVE 
	   sprintf(eq_temp[negs + pos], "z%d", encripted_newtuple);  
	                                       //AND IS THE FIRST TO BE PRINTED
	   me_printed_first = 0;               //WE DO NOT PRINT ITS SIGN
      	 }
	 else {
	   sprintf(eq_temp[negs + pos], "+z%d", encripted_newtuple);
	 }
	 circ_temp[count] = 1;      //BUILD THE LIST OF COCIRCUIT EL'T SIGNS
	 count++;
	 pos++;                       //INCREMENT THE NUMBER OF POSITIVE EL'TS
                                      //FOR THIS COCIRCUIT 
       }

       else if (temp == -1) {
	 me_printed_first = 0;
	 sprintf(eq_temp[negs + pos], "-z%d", encripted_newtuple);
	 circ_temp[count] = -1;      //BUILD THE LIST OF COCIRCUIT EL'T SIGNS
	 count++;                   
	 negs++;                      //INCREMENT THE NUMBER OF NEGATIVE EL'TS
                                      //FOR THIS COCIRCUIT    
       }

       else if (temp == 0) {	
	 circ_temp[count] = 0;      //BUILD THE LIST OF COCIRCUIT EL'T SIGNS
	 count++;
       }

       else {                         //IF TEMP IS NEITHER 1, -1, OR 0, THERE IS AN ERROR
	 cerr << "Bad value for temp.\n";
       }
     }

     else {
       circ_temp[count] = 0;
       count++;
     }

   }

   if (negs * pos != 0) {             //IF THERE WERE BOTH NEGATIVE AND POSITIVE ELEMENTS
                                      //THEN THE HYPERPLANE CUTS THROUGH THE POLYTOPE AND
                                      //WE MAY SET THE COCIRCUIT EQUATION TO 0.
     /* PRINT THE COCIRCUIT EQUATION TO THE FILE */
     for (int k = 0; k < negs + pos; k++) {
       file1 << eq_temp[k];
       file2 << eq_temp[k];
     }
     file1 << "=0\n";
     file2 << "=0\n";

   }

   mtrx->put_cocircuit(j - 1, circ_temp);  //PUT THE COCIRCUIT EQUATION IN THE 
                                           //CORRESPONDING SLOT OF THE "COCIRCUITS" ARRAY
					   //MEMBER OF THE MATRIX
 }

 return;

}


/*********************************************************
* purpose: Generates the cocircuit equations for the given matrix of n columns
*          and d rows and returns an array containing all of the null simplices
* date: 06.13.98
* notes/explanation: This function writes the signs of the determinants for the
* cocircuits into the COCIRCUITS array of the matrix.  It does not write the cocircuits
* to a file (this function should only be used when only the chambers are desired in the
* file).  
*********************************************************/

void cocircuit_equations_for_universal_polytope (matrix * mtrx) {

 int count = 0, encripted_newtuple, ki, temp, n = mtrx->cols(), d = mtrx->rows();
 int tuple[20], newtuple[20];
 int * circ_temp; 

 cout << "Generating cocircuit equations...\n";

 double how_many = choose(n, d-1);     //THESE VARIABLES ARE FOR PROGRESS REPORTS
 int update = 5;
 
 ofstream file("cocircuits");
 for (int j=1; j< how_many + 1; j++) { //FOR EACH (D-1)-TUPLE...
   file<<"],"<<endl<<"[";
   /* GIVE A PROGRESS REPORT EVERY 5% OF THE WAY */
   if (((double) j / how_many) > ((double) update / (double) 100)) {
      cout << update << " percent finished.\n";
      update += 5;
    }

   count = 0;                               //NO COCIRCUITS YET 
   circ_temp = new int [n];                 //ALLOCATE SPACE FOR THE COCIRCUIT
   encode_p(n,d-1,j,tuple);                 //THE TUPLE IS OBTAINED FROM j
   
   for (int i=1; i<n+1 ; i++) {             //FOR EACH COLUMN OF THE MATRIX... 
    
     ki=parity_of_permutation_reorder(i,tuple,0,d-2);  //GET THE PARITY OF THE
                                                       //TUPLE WITH THE COLUMN

     if (ki > -2) {                         //IF THE COLUMN IS NOT A REPEAT OF 
                                            //AN ELEMENT OF THE TUPLE...

       place_new_value_in_list(tuple,ki,i,d-1,newtuple); //GET THE D-TUPLE
       encripted_newtuple=(int) from_ntuple_to_p(newtuple,d); //DECODE THE
                                                              //D-TUPLE
       temp = mtrx->get_det_sign(encripted_newtuple - 1) * //THE SIGN FOR THE 
	      (int) pow(-1, d - ki);        //COCIRCUIT ELEMENT IS EQUAL TO THE
                                            //SIGN OF THE DETERMINANT TIMES -1
                                            //RAISED TO THE PARITY OF THE COL

       if (temp == 1) {                     //IF THE ELEMENT IS POSITIVE
	 circ_temp[count] = 1;              //BUILD THE LIST OF COCIRCUIT EL'T SIGNS
         file <<"+1,";
	 count++; 
       }

       else if (temp == -1) {
	 circ_temp[count] = -1;      //BUILD THE LIST OF COCIRCUIT EL'T SIGNS
         file <<"-1,";
	 count++;                    //FOR THIS COCIRCUIT    
       }

       else if (temp == 0) {
	 circ_temp[count] = 0;               //BUILD THE LIST OF COCIRCUIT EL'T SIGNS
       file <<"0,";
	 count++;
       }

       else {                         //IF TEMP IS NEITHER 1, -1, OR 0, THERE IS AN ERROR
	 cerr << "Bad value for temp.\n";
       }
     }

     else {
       circ_temp[count] = 0;
       file <<"0,";
       count++;
     }
   }

   mtrx->put_cocircuit(j - 1, circ_temp);  //PUT THE COCIRCUIT EQUATION IN THE 
                                           //CORRESPONDING SLOT OF THE "COCIRCUITS" ARRAY
					   //MEMBER OF THE MATRIX
 }
 file.close();
 return;
}


/*********************************************************
* purpose: Generates the chamber equation for the matrix and writes it to the two input
*          files.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

void chamber_equation_for_universal_polytope (matrix * mtrx, ofstream& file1,
					      ofstream & file2) {

  int d = mtrx->rows(), n = mtrx->cols(), current_size = 0, 
      max_size = (long) choose(n, d-1), temp_sign;

  set<int_set> non_faces_min;
  int_set non_face_temp;

  cout << "Generating minimal set of non-faces...\n";

  int update = 5;                //THIS VARIABLE IS USED FOR PROGRESS REPORTS

  /* FOR EACH COCIRCUIT... */
  for (int i = 0; i < max_size; i++) {

    /* GIVE A PROGRESS REPORT EVERY 5% OF THE WAY */
    if (((double) i / (double) max_size) > ((double) update / (double) 100)) {
      cout << update << " percent finished.\n";
      update += 5;
    }

    temp_sign = sign_of_lowest_valid_index(mtrx->get_cocircuit(i));  //GET THE SIGN OF
                                                                     //LOWEST INDEXED
                                                                     //NON-ZERO VARIABLE

    non_face_temp = get_signed_elements(mtrx->get_cocircuit(i),  //GET INDEXES OF ALL 
					temp_sign, n);           //VARIABLES WITH THIS
                                                                 //SIGN IN THE COCIRCUIT

    add_and_refine(&non_faces_min, non_face_temp);               //BUILD A MINIMAL SET
                                                                 //OF THE NON-FACES
  }

  cout << "Finding chambers...\n";
  set<int_set> chambers = mmt(non_faces_min);     //FROM THE NON-FACES, GET THE ELEMENTS
                                                  //FOR THE CHAMBER EQUATION

  elt<int_set> * chamber_ptr = chambers.getHEAD();
  int temp_ordered_array[20];

  int count = 0;
  int p_temp;

  /* THE FIRST ELEMENT OF THE CHAMBER EQUATION SHOULD NOT BE PRECEDED BY A "+" SIGN */
  if (chamber_ptr) {
    order_int_set(chamber_ptr->value, temp_ordered_array);   //ORDER THE TUPLE.
    p_temp = (long) from_ntuple_to_p(temp_ordered_array, d); //DECODE THE TUPLE.
    file1 << "z" << p_temp;                                  //PRINT THE VARIABLE TO THE
    file2 << "z" << p_temp;                                  //FILES.
    count++;
    chamber_ptr = chamber_ptr->next;                         //MOVE TO THE NEXT ELEMENT
  }

  while (chamber_ptr) {
    
    /* EVERY TEN VARIABLES, START A NEW LINE */
    if (count == 10) {
      file1 << endl;
      file2 << endl;
      count = 0;
    }

    order_int_set(chamber_ptr->value, temp_ordered_array);   //ORDER THE TUPLE.
    p_temp = (long) from_ntuple_to_p(temp_ordered_array, d); //DECODE THE TUPLE.
    file1 << "+z" << p_temp;                                 //PRINT THE VARIABLE TO THE
    file2 << "+z" << p_temp;                                 //FILES.
    count++;
    chamber_ptr = chamber_ptr->next;                         //MOVE TO THE NEXT ELEMENT
  }

  file1 << "=1" << endl << endl;                    
  file2 << "=1" << endl << endl;

  return;

}


/*********************************************************
* purpose: This function takes an unordered set of integers as input and an empty array
*          of integers into which it writes the ordered set of integers.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

void order_int_set (int_set & unordered_set, int * ordered_array) {

  int min_temp, counter = 0;
  elt<int> * set_ptr = unordered_set.getHEAD(); //INITIALIZE SET POINTER

  while (set_ptr) {
    
    set_ptr = unordered_set.getHEAD();   //START THE SET POINTER AT THE HEAD OF THE SET
    min_temp = set_ptr->value;           //INITIALIZE THE MINIMUM ELEMENT FOR COMPARISON 

    while (set_ptr) {
      if (set_ptr->value < min_temp) {   //IF THE VALUE POINTED AT IS LESS THAN THE 
	min_temp = set_ptr->value;       //CURRENT MIN, SET THE MIN TO IT.
      }
      set_ptr = set_ptr->next;           //MOVE TO THE NEXT ELEMENT.
    }

    ordered_array[counter] = min_temp;   //ADD THE MINIMUM ELEMENT TO THE ORDERED ARRAY
    counter++;                              
    unordered_set.remove_element(min_temp); //REMOVE THE MINIMUM ELEMENT FROM THE SET 
                                            //TO FIND THE NEXT SMALLEST
    set_ptr = unordered_set.getHEAD();   //RESET THE SET POINTER
  }

  return;
}


/*********************************************************
* purpose: Returns the sign (+ or -) of the lowest numbered variable in the cocircuit
*          equation.
* date: 06.13.98
* notes/explanation: 
*********************************************************/
  
int sign_of_lowest_valid_index (int * cocircuit) {

  int slot = 0;

  while (cocircuit[slot] == 0) {
    slot++;                         //FIND THE FIRST VALUED ENTRY IN THE COCIRCUIT ARRAY
  }

  return cocircuit[slot];           //RETURN THE VALUE (+ or - 1)

}


/*********************************************************
* purpose: Returns a set of variable numbers with the input sign (+ or -) from the 
*          cocircuit equation.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

set<int> get_signed_elements (int * cocircuit, int sign, int size) {

  set<int> signed_elements;

  /* FOR EACH ELEMENT IN THE COCIRCUIT ARRAY... */
  for (int i = 0; i < size; i++) {

    if (cocircuit[i] == sign) {
      signed_elements.add_element(i + 1);  //IF THE ELEMENT HAS THE DESIRED SIGN, PUT IT
    }                                      //IN THE SET

  }

  return signed_elements;

}
  

/*********************************************************
* purpose: Adds the new element to the set of non_faces only if no member of the 
*          non_faces set is contained within the new element.  Also, if the new element
*          is added, all elements in the set of non_faces which contain the new element
*          are eliminated.  If the element is added successfully, 1 is returned (ow 0)
* date: 06.13.98
* notes/explanation: 
*********************************************************/

bool add_and_refine (set<int_set> * non_faces, const int_set & new_elt)
{
 
  elt<int_set> * non_faces_ptr = non_faces->getHEAD();
  int elt_cont_flag = 0;

  if (!new_elt.getHEAD()) {
    return false;          //IF THE NEW ELEMENT IS NULL, DO NOT ADD IT, RETURN 0
  }

  while (non_faces_ptr) {
    
    if (!elt_cont_flag) {
      if (new_elt.is_contained(non_faces_ptr->value)) {  //IF THE NEW ELEMENT CONTAINS AN
	return false;                                    //ELEMENT OF NON_FACES, DO NOT 
      }                                                  //ADD IT, RETURN 0
    }
    
    if (non_faces_ptr->value.is_contained(new_elt)) {  //IF THE NEW ELEMENT IS CONTAINED
      non_faces->remove_element(non_faces_ptr->value); //IN ANY OF THE ELEMENTS OF 
      elt_cont_flag = 1;                               //NON_FACES, REMOVE THAT ELEMENT
    }                                                  //FROM NON_FACES.

    non_faces_ptr = non_faces_ptr->next;               //MOVE TO THE NEXT ELEMENT

  }

  non_faces->add_element(new_elt);                     //ADD THE NEW ELEMENT TO THE
  return true;                                         //NON FACES

}


/*********************************************************
* purpose: Returns a set of elements consisting of the intersection of sets A and B
* date: 06.13.98
* notes/explanation: 
*********************************************************/

template< class T >
set<T> set_intersection (set<T> &A, set<T> &B) {

  set<T> intersected_set;
  elt<T> * ptr = A.getHEAD();

  while (ptr) {

    if (B.in_set(ptr->value)) {                 //IF THE PTR ELEMENT (AN ELEMENT OF A) IS
      intersected_set.add_element(ptr->value);  //IN THE SET B, ADD IT TO THE INTERSECTED
    }                                           //SET.

    ptr = ptr->next;                            //MOVE TO THE NEXT ELEMENT

  }

  return intersected_set;                       //RETURN THE INTERSECTED SET

} 


/*********************************************************
* purpose: Returns a set of elements consisting of the union of sets A and B 
* date: 06.13.98
* notes/explanation: 
*********************************************************/

template< class T >  
set<T> set_union (set<T> &A, set<T> &B) {

  set<T> union_set;
  union_set = A;        //FILL THE UNION SET WITH MEMBERS OF A

  elt<T> * ptr = B.getHEAD();

  while (ptr) {

    union_set.add_element(ptr->value);   //ADD THE ELEMENT FROM B TO THE UNION SET.
    ptr = ptr->next;                     //MOVE TO THE NEXT ELEMENT.

  }

  return union_set;                      //RETURN THE UNION SET.

}
  

/*********************************************************
* purpose: A recursive function which finds all chamber equation variables from the
*          set of non_faces
* date: 06.13.98
* notes/explanation: 
*********************************************************/

const set<int_set> mmt (const set<int_set> & sets) {

  set<int_set> erg;

  if (!(sets.getHEAD())) {            //IF THE SET IS EMPTY, RETURN A SET CONTAINIING AN
    int_set empty_set;                //EMPTY SET.
    erg.add_element(empty_set);
  }

  else {

    elt<int_set> * head = sets.getHEAD();
    set<int_set> tail = mmt(sets.without_element(head->value));
    elt<int> * head_ptr = head->value.getHEAD();
    elt<int_set> * tail_ptr = tail.getHEAD();

    while (head_ptr) {

      int_set head_ptr_set;
      head_ptr_set.add_element(head_ptr->value);

      while (tail_ptr) {
	
	add_and_refine(&erg, set_union(head_ptr_set, tail_ptr->value));
	tail_ptr = tail_ptr->next;

      }

      tail_ptr = tail.getHEAD();
      head_ptr = head_ptr->next;
    }
  }

  return erg;

}


/*********************************************************
* purpose: This function preps the minimum and maximum file for use with CPLEX.  It 
*          writes the "Minimum" and "Maximum" headers along with the objective equation
* date: 06.13.98
* notes/explanation: The integer array "zeros" tells the function which variables to 
*                    leave out of the objective function (those which are null simplices)
*********************************************************/

void CPLEX_file_prep (matrix * mtrx, const int * zeros, ofstream& file1, 
                      ofstream& file2) {

  int n = mtrx->cols(), d = mtrx->rows();

  file1 << "Minimize" << endl << "Obj:" << endl;   //FIRST PUT IN THE "MAXIMIZE" AND
  file2 << "Maximize" << endl << "Obj:" << endl;    //"MINIMIZE" HEADERS

  int count = 0, printed = 0;

  while (zeros[count]) {          //GET TO THE FIRST NON-ZERO ELEMENT
    count++;
  }

  file1 << "z" << count + 1;      //THE FIRST ELEMENT OF THE OBJECTIVE EQ SHOULD NOT BE
  file2 << "z" << count + 1;      //PRECEDED BY A "+" SIGN.
            
  count++;
  printed++;

  for (int i = count; i < choose(n, d); i++) {

    if (!(zeros[i])) {            //IF THE CURRENT ELEMENT IS NOT A ZERO, PRINT IT IN
      file1 << "+z" << i + 1;     //THE OBJECTIVE EQUATION
      file2 << "+z" << i + 1;
      printed++;
    }

    if ((printed % 20) == 0) {    //INSERT A NEW LINE AFTER EVERY 20 PRINTED VARIABLES
      file1 << endl;
      file2 << endl;
    }
  }

  file1 << endl << "Subject To" << endl << endl;   //INTRODUCE THE CONSTRAINTS TO CPLEX
  file2 << endl << "Subject To" << endl << endl;

  return;

}


/*********************************************************
* purpose: This function finishes the minimum and maximum files for use with CPLEX.  It
*          writes the list of variables and the "End" to the files.
* date: 06.13.98
* notes/explanation: The integer array "zeros" tells the function which variables to 
*                    leave out of the list of variables (those which are null simplices)
*********************************************************/

void CPLEX_file_finish (matrix * mtrx, const int * zeros, ofstream& file1, 
                        ofstream& file2) {

  int n = mtrx->cols(), d = mtrx->rows();

  file1 << endl << "Integer" << endl;        //TELL CPLEX TO FIND ONLY INTEGER SOLUTIONS
  file2 << endl << "Integer" << endl;

  /* FINALLY, LIST ALL VARIABLES (SIMPLICES) */
  for (int i = 0; i < (long) choose(n, d); i++) {

    if (!(zeros[i])) {                   //IF THE ELEMENT IS NOT A ZERO, LIST IT AS A
      file1 << "z" << i + 1 << endl;     //VARIABLE
      file2 << "z" << i + 1 << endl;
    }
  }

  file1 << "End";
  file2 << "End";

  return;

}











