/*-------------------------------UTILITIES.COMBINATORICS.H------------------------------
|                                                                                      |
| double gammln (double xx);                                                           |
|                                                                                      |
| double factln (int n);                                                               |
|                                                                                      |
| double choose (int n, int k);                                                        |
|                                                                                      |
| bool encode_p (int M_max, int n_max, double p, int * p_code);                        |
|                                                                                      |
| double from_ntuple_to_p (int * ntuple, int d);                                       |
|                                                                                      |
| int find_np_code (int M_max, int n, double p);                                       |
|                                                                                      |
| int parity_of_permutation_reorder(int newvalue,int * a_list,int beginning_list,      |
|                                   int end_list);                                     |
|                                                                                      |
| int * place_new_value_in_list (int * tuple, int place, int newvalue,                 |
|			       int length_tuple);                                      |
|                                                                                      |
| void order_int_set (int_set & unordered_set, int * ordered_array);                   |
|                                                                                      |
| void CPLEX_file_prep (matrix * mtrx, const int * zeros, ofstream& file1,             |
|                       ofstream& file2);                                              |
|                                                                                      |
| void chamber_equation_for_universal_polytope (matrix * mtrx, ofstream& file1,        |
|                                               ofstream& file2);                      |
|                                                                                      |
| int * cocircuit_equations_for_universal_polytope (matrix * mtrx, ofstream& file1,    |
|                                                  ofstream& file2);                   |
|						                                       |
| int * cocircuit_equations_for_universal_polytope (matrix * mtrx);                    |
|                                                                                      |
| void CPLEX_file_finish (matrix * mtrx, const int * zeros, ofstream& file1,           |
|                         ofstream& file2);                                            |
|                                                                                      |
| int sign_of_lowest_valid_index (int * cocircuit);                                    |
|                                                                                      |
| set<int> get_signed_elements (int * cocircuit, int sign, int size);                  |
|                                                                                      |
| int add_and_refine (set<int_set> * non_faces_array, const int_set & new_elt);        |
|                                                                                      |
| template< class T >                                                                  |
| set<T> set_union (set<T> &A, set<T> &B);                                             |
|                                                                                      |
| template< class T >                                                                  |
| set<T> set_intersection (set<T> &A, set<T> &B);                                      |
|                                                                                      |
| const set<int_set> mmt (const set<int_set> & sets);                                  |
|                                                                                      |
---------------------------------------------------------------------------------------*/


# include "matrix.h"
# include "samset.h"

/*********************************************************
* THIS FILE CONTAINS:
* declarations for combinatorial utility functions
**********************************************************/


#ifndef UTILITIES_COMBINATORICS_H
#define UTILITIES_COMBINATORICS_H


typedef set<int> int_set;


/*********************************************************
* purpose: computes the Logarithm of the Gamma Function.
* date: 06.08.98
* notes/explanation: Taken from the book Numerical recipes in C
* it is based on an approximation formula, as a power series, 
* that uses fixed coefficients. They are provided below.
*********************************************************/

double gammln (double xx);


/*********************************************************
* purpose: Computes the Logarithm of the factorial n!
* date: 06.08.98
* notes/explanation: Based on the fact that  n!= Gamma(n+1).
* When dealing with binomial coefficients, often one has overflow
* when using the straightforward formula for n! (n*n-1*n-2...*1).
* so we must use the logarithm.
*********************************************************/

double factln (int n);


/*********************************************************
* purpose: Computes the binomial coefficient (n choose k) as a floating-point
* number.
* date: 06.08.98
* notes/explanation: 
*********************************************************/

double choose (int n, int k);


/*********************************************************
* purpose: Given an integer p and an empty integer array, the function fills the integer
*          array with the the pth ntuple element with respect to reverse lex order 
* date: 06.08.98
* notes/explanation: The desired n-tuple can be read off 
* from a greedy decomposition of p as the sum of binomial
* coefficients. See Kruskal-Katona theorem ( Ziegler's book
* Lectures on Polytopes chapter 8) for details.
*********************************************************/
  
bool encode_p (int M_max, int n_max, double p, int * p_code);


/*********************************************************
* purpose: Finds the maximal m that makes the
* binomial coefficient (m choose n) fit inside a given number p.
* date: 06.08.98
* notes/explanation: 
*********************************************************/

int find_np_code (int M_max, int n, double p);


/*********************************************************
* purpose: finds the reverse lex order of the input ntuple
* containing d elements.
* date: 06.12.98
* notes/explanation: this is the inverse of encode_p 
*********************************************************/

double from_ntuple_to_p (int * ntuple, int d);


/*************************************************************
* purpose: Given an array of d+1 numbers with the first d already
* ordered, returns the position of the (d+1)th point with respect to
* the others. Obviously, this gives the parity of a reorder permutation!!!
* date: 06.08.98
* notes/explanation: 
*********************************************************/

int parity_of_permutation_reorder(int newvalue,int * a_list,int beginning_list,int end_list);


/*********************************************************
* purpose: Returns a new list containing the old list with the newvalue added 
*          into the correct place.
* date: 06.13.98
* notes/explanation: Straightforward
*********************************************************/

int * place_new_value_in_list (int * tuple, int place, int newvalue, 
			       int length_tuple);

/*********************************************************
* purpose: Generates the cocircuit equations for the given matrix of n columns
*          and d rows and prints the equations into the two input files
* date: 06.13.98
* notes/explanation: This function writes the cocircuit equations in cplex 
* format to the file whose name is filename, based on whether opt is "Minimize"
* or "Maximize."  It also writes the signs of the determinants for the
* cocircuits into the COCIRCUITS array of the matrix.  
*********************************************************/

void cocircuit_equations_for_universal_polytope (matrix * mtrx, 
	       					 ofstream& file1, 
						 ofstream & file2);


/*********************************************************
* purpose: Generates the cocircuit equations for the given matrix of n columns
*          and d rows and returns an array containing all of the null simplices
* date: 06.13.98
* notes/explanation: This function writes the signs of the determinants for the
* cocircuits into the COCIRCUITS array of the matrix.  It does not write the cocircuits
* to a file (this function should only be used when only the chambers are desired in the
* file).  
*********************************************************/

void cocircuit_equations_for_universal_polytope (matrix * mtrx);


/*********************************************************
* purpose: This function takes an unordered set of integers as input and an empty array
*          of integers into which it writes the ordered set of integers.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

void order_int_set (int_set & unordered_set, int * ordered_array);


/*********************************************************
* purpose: This function preps the minimum and maximum file for use with CPLEX.  It 
*          writes the "Minimum" and "Maximum" headers along with the objective equation
* date: 06.13.98
* notes/explanation: The integer array "zeros" tells the function which variables to 
*                    leave out of the objective function (those which are null simplices)
*********************************************************/

void CPLEX_file_prep (matrix * mtrx, const int * zeros, ofstream& file1, 
                      ofstream& file2);


/*********************************************************
* purpose: Generates the chamber equation for the matrix and writes it to the two input
*          files.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

void chamber_equation_for_universal_polytope (matrix * mtrx, ofstream& file1,
					      ofstream& file2);


/*********************************************************
* purpose: This function finishes the minimum and maximum files for use with CPLEX.  It
*          writes the list of variables and the "End" to the files.
* date: 06.13.98
* notes/explanation: The integer array "zeros" tells the function which variables to 
*                    leave out of the list of variables (those which are null simplices)
*********************************************************/

void CPLEX_file_finish (matrix * mtrx, const int * zeros, ofstream& file1, 
                        ofstream& file2);


/*********************************************************
* purpose: Returns the sign (+ or -) of the lowest numbered variable in the cocircuit
*          equation.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

int sign_of_lowest_valid_index (int * cocircuit);


/*********************************************************
* purpose: Returns a set of variable numbers with the input sign (+ or -) from the 
*          cocircuit equation.
* date: 06.13.98
* notes/explanation: 
*********************************************************/

set<int> get_signed_elements (int * cocircuit, int sign, int size);


/*********************************************************
* purpose: Adds the new element to the set of non_faces only if no member of the 
*          non_faces set is contained within the new element.  Also, if the new element
*          is added, all elements in the set of non_faces which contain the new element
*          are eliminated.  If the element is added successfully, 1 is returned (ow 0)
* date: 06.13.98
* notes/explanation: 
*********************************************************/

bool add_and_refine (set<int_set> * non_faces, const int_set & new_elt);


/*********************************************************
* purpose: Returns a set of elements consisting of the union of sets A and B 
* date: 06.13.98
* notes/explanation: 
*********************************************************/

template< class T >
set<T> set_union (set<T> &A, set<T> &B);


/*********************************************************
* purpose: Returns a set of elements consisting of the intersection of sets A and B
* date: 06.13.98
* notes/explanation: 
*********************************************************/

template< class T >
set<T> set_intersection (set<T> &A, set<T> &B);


/*********************************************************
* purpose: A recursive function which finds all chamber equation variables from the
*          set of non_faces
* date: 06.13.98
* notes/explanation: 
*********************************************************/

const set<int_set> mmt (const set<int_set> & min_non_faces);

#endif
