# include "utilities.linalg.h"
# include "utilities.combinatorics.h"
//# define DEBUG
# include "debug.h"

/*********************************************************
* THIS FILE CONTAINS:
* subroutines for the computation of all dxd determinants
* of a dxn matrix. This is the so called CHIROTOPE of 
* a point configuration.
**********************************************************/

/*********************************************************
* purpose: computes the absolute value.
* date: 06.08.98
* notes/explanation: 
*********************************************************/

double abs (double x) {

  if (x < 0) {
    return -1.0 * x;
  }

  return x;
}

/*********************************************************
* purpose: Computes the LU-decomposition of a square matrix A
* date: 06.08.98
* notes/explanation: Code modified from the book numerical recipes
* in C, by W. Press, B. Flannery, S. Teukolsky and W. Vetterling.
* Cambridge Press 1989.
*
* Idea: If one can decompose the square matrix A
* product of matrices LU, with L lower triangular and U upper
* triangular then the evaluation of the determinant is just
* the product of the products of the diagonal elements.
* One computes first a decomposition of the rowwise permutation
* of A. One keeps track of k= number of interchanges and adjust
* the product by the corresponding sign (-1)^k.
*
* REMARK: For large matrices overflow may occur with respect to
* the floating point range in the computer. The solution is that
* we actually do NOT need the determinants but only its sign!!!
* so we would just recover this from the diagonal signs.
*********************************************************/

double LU_decomp (matrix * mtrx) {

  if (mtrx->rows() != mtrx->cols()) {
    cerr << "Cannot decompose non-square matrix. \n";
    return 0;
  }
  
  int N = mtrx->rows(), i, j;
  double scaling[100];  //STORES IMPLICIT SCALING OF EACH ROW
  double row_max, temp;  
  
  //LOOP OVER ROWS TO GET IMPLICIT SCALING INFORMATION.
  for (i = 1; i <= N; i++) {
    
    row_max = 0.0;

    for (j = 1; j <= N; j++) {
      
      if ((temp = abs(mtrx->entry(i, j))) > row_max) {
	row_max = temp;
      }
    }

    if (row_max == 0.0) {
      //cerr << "Cannot decompose singular matrix. \n";
      return 0;
    }

    scaling[i] = 1.0 / row_max;  //SAVE THE SCALING.
  }

  double sum, dum, tiny = 1.0e-20;
  double parity = 1.0;              //NO ROW INTERCHANGES YET.
  int imax, k;

  //THIS IS THE LOOP OVER COLUMNS OF CROUT'S METHOD.
  for (j = 1; j <= N; j++) {

    for (i = 1; i < j; i++) {

      sum = mtrx->entry(i, j);

      for (k = 1; k < i; k++) {

	sum -= mtrx->entry(i, k) * mtrx->entry(k, j);
      }

      mtrx->changeEntry(i, j, sum);
    }

    row_max = 0.0;  //INITIALIZE FOR THE SEARCH FOR LARGEST PIVOT ELEMENT.

    for (i = j; i <= N; i++) {

      sum = mtrx->entry(i, j);

      for (k = 1; k < j; k++) {

	sum -= mtrx->entry(i, k) * mtrx->entry(k, j);
      }

      mtrx->changeEntry(i, j, sum);

      //IS THE FIGURE OF MERIT FOR THE PIVOT BETTER THAN THE BEST SO FAR?
      if ((dum = scaling[i] * abs(sum)) >= row_max) {
	row_max = dum;
	imax = i;
      }
    }

    //DO WE NEED TO INTERCHANGE ROWS?
    if (j != imax) {
      //YES, DO SO...
      for (k = 1; k <= N; k++) {

	dum = mtrx->entry(imax, k);
	mtrx->changeEntry(imax, k, mtrx->entry(j, k));
	mtrx->changeEntry(j, k, dum);
      }

      parity *= -1.0;  //...AND CHANGE THE PARITY.
      scaling[imax] = scaling[j];  //ALSO INTERCHANGE THE SCALE FACTOR.
    }

    if (mtrx->entry(j, j) == 0.0) {
      mtrx->changeEntry(j, j, tiny);
    }

    if (j != N) {
      dum = 1.0 / (mtrx->entry(j, j));  //NOW DIVIDE BY THE PIVOT ELEMENT.

      for (i = j + 1; i <= N; i++) {

	mtrx->changeEntry(i, j, mtrx->entry(i, j) * dum);
      }
    }
  }

  return parity;
}


/*********************************************************
* purpose: computes the determinant of a square matrix.
* date: 06.08.98
* notes/explanation: once we have LU decomposition it is
* simply a matter of multiplying the diagonal elements.
*********************************************************/

double determinant (matrix * mtrx) {

 /* THIS VALUE DETERMINES THE SENTITIVITY OF WHICH RESULTS WILL EQUAL
    ZERO. THE smaller the number fewer things will be picked up to be ZERO */

  double tolerance = 1.0e-6;

  if (mtrx->rows() != mtrx->cols()) {
    cerr << "Cannot compute determinant of non-square matrix. \n";
    return 0;
  }

  double parity = LU_decomp(mtrx);
  double determinant = 1.0;

  for (int j = 1; j <= mtrx->rows(); j++) {

    if ((mtrx->entry(j,j) < tolerance) &&
	(mtrx->entry(j,j) > -tolerance)) {
      return 0;
    } 
    
    determinant *= mtrx->entry(j, j);

  }

  return parity * determinant;

}

/*********************************************************
* purpose: computes ALL the dxd determinants of a dxn matrix.
* date: 06.08.98
* notes/explanation: 
*********************************************************/
 
int * allDets (matrix * mtrx) {

  cout << "Computing all dxd determinants...\n";

  int d = mtrx->rows(), n = mtrx->cols();
  int dtuple[20];
  double p = 1.0, det_temp;
  matrix * temp = new matrix(d, d);
  int * zeros = new int [(long) choose(n, d)];
  int count_zeros = 0;

  double how_many = choose(n, d);  //THESE VARIABLES ARE USED FOR PROGRESS REPORTS
  int update = 5;

  while (encode_p(n, d, p, dtuple)) {

    /* GIVE A PROGRESS REPORT EVERY 5% OF THE WAY */
    if ((p / how_many) > ((double) update / (double) 100)) {
      cout << update << " percent finished.\n";
      update += 5;
    }

    mtrx->fillSquareMatrix(0, dtuple, d, temp);  //CREATE THE SQUARE MATRIX 
    det_temp = determinant(temp);                //CORRESPONDING TO THE DTUPLE

    if (det_temp < 0) {                          //IF THE DETERMINANT IS NEGATIVE, NOTE
      mtrx->enter_det_sign((int) p - 1, -1);     //ITS SIGN
      zeros[(int) p - 1] = 0;                    //NOTE THE NON-ZERO DETERMINANT
    }

    else if (det_temp > 0) {                     //IF THE DETRMINANT IS POSITIVE, NOTE
      mtrx->enter_det_sign((int) p - 1, 1);      //ITS SIGN
      zeros[(int) p - 1] = 0;                    //NOTE THE NON-ZERO DETERMINANT
    }

    else {
      count_zeros++;
      mtrx->enter_det_sign((int) p - 1, 0);      //IF THE DETERMINANT IS ZERO, NOTE IT
      zeros[(int) p - 1] = 1;                    //BOTH IN THE LIST OF DETERMINANTS AND
    }                                            //THE LIST OF ZEROS.

    p += 1.0;

  }

  _ASSERT(count_zeros);
  delete temp;
  return zeros; 
}









