%  Example driver for the simplified band Lanczos methods "Herm_band_Lanczos",
%  "sym_band_Lanczos", "JHerm_band_Lanczos", and "Jsym_band_Lanczos"
%    
%  This driver runs the simplified versions (for Hermitian, symmetric,
%  J-Hermitian, and J-symmetric matrices A) of the band Lanczos method applied
%  to a random sparse N x N matrix A, which is generated by the functions
%  "make_sparse_randn_matrix_A" or "make_sparse_randn_matrix_A_and_J", and an
%  N x m starting block R, which is generated by the function 
%  "make_some_starting_blocks".
%
% -----------------------------------------------------------------------------
%
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

N = input('Size N of the N x N sparse matrix A = ')
%
density = input('Density of the sparse matrix A = ')
%
simplified_type = input('Type of A (Herm = 1, sym = 2, JHerm = 3, Jsym = 4): ')
%
if simplified_type > 2,
   class = input('Class of A and J (Class 1 = 1, Class 2 = 2): ')
end
%
rflag = input('Real or complex A and R (real = 1, complex = 2): ')
%
switch simplified_type
   case 1
      if rflag == 1,
         type = 2;  %  A is real symmetric
      else 
         type = 4;  %  A is Hermitian (with complex entries)
      end
   case 2
      if rflag == 1,
         type = 2;  %  A is real symmetric
      else 
         type = 3;  %  A is complex symmetric
      end    
   case 3
      if rflag == 1,
         type = 5;  %  A is real and J-symmetric
      else 
         type = 7;  %  A is complex and J-Hermitian
      end    
   case 4
      if rflag == 1,
         type = 5;  %  A is real and J-symmetric
      else 
         type = 6;  %  A is complex and J-symmetric
      end  
end
%
if type < 5,
   A = make_sparse_randn_matrix_A(N,type,density);
else
   [A,J] = make_sparse_randn_matrix_A_and_J(N,type,class,density);
end
%
R = make_some_starting_blocks(A,1,rflag);
%
sflag = input('Store all Lanczos vectors (no = 0, yes = 1): ')
%
nmax = input('Number nmax of Lanczos vectors to be generated = ')
%
tol.defl_tol = sqrt(eps);
tol.defl_flag = 1;
tol.normA_flag = 1;
tol.brk_tol = 1.e-14;
if simplified_type > 2,
   tol.normJ_flag = 1;
end
%
fprintf(' \n')
disp('**----------------------------**')
disp('** Run all nmax steps at once **')
disp('**----------------------------**')
%
switch type
   case 2
      result = Herm_band_Lanczos(A,R,nmax,sflag,tol);
   case 3
      result = sym_band_Lanczos(A,R,nmax,sflag,tol);
   case 4
      result = Herm_band_Lanczos(A,R,nmax,sflag,tol);
   case 5
      result = Jsym_band_Lanczos(A,J,R,nmax,sflag,tol); 
   case 6
      result = Jsym_band_Lanczos(A,J,R,nmax,sflag,tol);
   case 7
      result = Jsym_band_Lanczos(A,@(x) MultA(x,J),R,nmax,sflag,tol);
end


