function [R,L] = make_some_starting_blocks(A,bflag,rflag)
%
%  This function generates starting blocks (with the same number of rows as A)
%  for the various band algorithms.
%
%  The input flag "bflag" determines if right and left starting blocks are
%  generated:
%
%  bflag =  1:  only R is generated
%
%  bflag =  2:  R and L are generated
%
%  The input flag "rflag" determines if the starting blocks are real or
%  complex matrices:  
%
%  rflag =  1:  the starting blocks are real
%
%  bflag =  2:  the starting blocks are complex
%
%  The starting blocks are generated as random matrices that are then modified 
%  so that some deflations occur early in the band iteration.
%
% -----------------------------------------------------------------------------
%
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

N = size(A,1);
m = input('Number m of right starting vectors in R = ')
%
if (m <= 0) || (rem(m,1) ~= 0) ,
   error('** m must be a positive integer! **')
end
%
R = modified_rand_function(N,m,rflag);
%
if m > 1,
   R(:,2) = R(:,1) + 1e-12 * modified_rand_function(N,1,rflag);
end
%
if m > 2,
   R(:,m) = A*A*A*R(:,1) + 1e-12 * modified_rand_function(N,1,rflag);
end
%
if m > 3,
   R(:,m-1) = A*A*R(:,1) + 1e-12 * modified_rand_function(N,1,rflag);
end
%
if bflag == 1,
   return
end
%
p = input('Number p of left starting vectors in L = ')
%
if (p <= 0) || (rem(p,1) ~= 0) ,
   error('** p must be a positive integer! **')
end
%
L = modified_rand_function(N,p,rflag);
%
if p > 1,
   L(:,2) = A.'*A.'*L(:,1) + 1e-12 * modified_rand_function(N,1,rflag);
end
%
if p > 3,
   L(:,p-1) = A.'*L(:,1) + 1e-12 * modified_rand_function(N,1,rflag);
end
%
% -----------------------------------------------------------------------------
%
function M = modified_rand_function(N,m,rflag)
%
%  This function employs the Matlab function "rand" to produce a random matrix
%  M of size N x m with entries in the interval [-1,1].  The entries of M are
%  real if rflag = 1 and complex otherwise.
%
% -----------------------------------------------------------------------------
%
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

M = 2 * rand(N,m) - 1;
%
if rflag ~= 1,
   M = M + sqrt(-1) * (2 * rand(N,m) - 1);
end

