function [A,J] = make_sparse_randn_matrix_A_and_J(N,type,class,density)
%
%  This function employs the Matlab function "sprandn" to generate a random 
%  sparse N x N matrix A that is J-symmetric or J-Hermitian.  The input
%  "density" is passed on to "sprandn" and controls the sparsity of the nonzero
%  subblocks of A:  An n x m subblock has approximately
%
%               density * N * N
%
%  normally distributed nonzero entries.  Note that the Matlab function 
%  "sprandn" is designed to produce large matrices with small density and will
%  generate significantly fewer nonzeros than requested if N is small or
%  "density" is large.
%
%  The input flag "type" is used to control the type of matrix A that is
%  generated:
%
%  type = 5:  A is real and J-symmetric
%
%  type = 6:  A is complex (in general) and J-symmetric
%
%  type = 7:  A is complex (in general) and J-Hermitian
%
%  The input flag "class" is used to control the class of matrices that are
%  generated:  
%
%  class = 1:  A and J are N x N matrices of the form
%
%                     | A_1    A_2 |            | I_1    0  |
%                 A = |            |  and  J =  |           | .  
%                     | A_3     0  |            |  0   -I_2 |
% 
%              Here, the subblocks A_1 and I_1 are of size N1 x N1 where
%              N1 = floor(N/2), and I_1 and I_2 denote the identity matrix of
%              size N1 and N - N1, respectively.
%
%  class = 2:  A and J are N x N matrices of the form
%
%                     | A_1     0  |            | I_1     D |
%                 A = |            |  and  J =  |           | .
%                     | A_2    A_3 |            | D^*     0 |
%
%              Here, * = "T" or "H", the subblocks A_11 and I_1 are of size
%              N1 x N1 where N1 = floor(N/2), I_1 denotes the identity matrix  
%              of size N1, and D is a diagonal matrix.
%  
% -----------------------------------------------------------------------------
%
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

A = sparse(N,N);
%
N1 = floor(N/2);
N1p1 = N1 + 1;
N2 = N - N1;
%
switch class
%
%  Start of class = 1
%
   case 1
%
      J = speye(N,N);
      J(N1p1:N,N1p1:N) = - J(N1p1:N,N1p1:N);
%
      switch type
         case 5 
            A(1:N1,N1p1:N) = mysprandn(N1,N2,density);
            A = A - A';
            A(1:N1,1:N1) = sprandmat(N1,2,density);
         case 6
            A12 = mysprandn(N1,N2,density);
            A12 = A12 + sqrt(-1) * mysprandn(A12);
            A(1:N1,N1p1:N) = A12;
            A = A - A.';
            A(1:N1,1:N1) = sprandmat(N1,3,density);
         case 7
            A12 = mysprandn(N1,N2,density);
            A12 = A12 + sqrt(-1) * mysprandn(A12);
            A(1:N1,N1p1:N) = A12;
            A = A - A';
            A(1:N1,1:N1) = sprandmat(N1,4,density);
      end
%
%  End  of class = 1
%
%  Start of class = 2
%
   case 2
%
      J = speye(N1,N1);
%
      switch type
         case 5
            A1 = sprandmat(N1,2,density);
            M = sprandmat(N1,0,density/2);
            D = sprandmat(N1,9,density);
            Dinv = spdiags(1./spdiags(D),0,N1,N1);
            A3 = Dinv * A1 * D;
            A2 = M + Dinv * M.' * D;
            J(1:N1,N1p1:N) = D;
            J(N1p1:N,1:N1) = D;
         case 6
            A1 = sprandmat(N1,3,density);
            M = sprandmat(N1,1,density/2);
            D = sprandmat(N1,-9,density);
            Dinv = spdiags(1./spdiags(D),0,N1,N1);
            A3 = Dinv * A1 * D;
            A2 = M + Dinv * M.' * D;
            J(1:N1,N1p1:N) = D;
            J(N1p1:N,1:N1) = D;
         case 7
            A1 =  sprandmat(N1,4,density);
            M = sprandmat(N1,1,density/2);
            D = sprandmat(N1,-9,density);
            Dinv = spdiags(1./spdiags(D),0,N1,N1);
            A3 = Dinv * A1 * D;
            A2 = M + Dinv * M' * conj(D);
            J(1:N1,N1p1:N) = D;
            J(N1p1:N,1:N1) = conj(D);
      end
%
      A(1:N1,1:N1) = A1;
      A(N1p1:N,1:N1) = A2;
      A(N1p1:N,N1p1:N) = A3;
%
%  End  of class = 2
%
end
  
