function result = sym_band_Lanczos(A,R,nmax,sflag,tol,n,result)
%
%  Symmetric band Lanczos method
%
% -----------------------------------------------------------------------------
%
%  Notes:  1) The band Lanczos method simplifies when A is symmetric, i.e., 
%             A = A^T (= A.'), and the blocks of right and left starting
%             vectors are identical, i.e., R = L.  In this case, the right and
%             left Lanczos vectors are identical.  The symmetric band Lanczos 
%             method exploits this fact and generates only the right Lanczos 
%             vectors.  As a result, the computational costs and the storage
%             requirements of the function "sym_band_Lanczos" are only roughly
%             half of those of the function "band_Lanczos".
%
%          2) The matrices A and R are allowed to be complex in general.  The 
%             matrix A is assumed to be symmetric.  Note that A is complex
%             symmetric, but not Hermitian if not all entries of A are real.
%             If the matrices A and R are both real, the function 
%             "Herm_band_Lanczos.m" should be used instead of the function
%             "sym_band_Lanczos.m".
%
%          3) The "orthogonalizations" performed in the algorithm are with
%             respect to the bilinear form
% 
%                    (w,v) = w^T v (= w.' * v),
%
%             where w^T (= w.') denotes the transpose of w;  note that in the 
%             complex case, this bilinear form is not an inner product.
%
%          4) This algorithm explicitly enforces only the minimum possible 
%             amount of "orthogonalization".  Moreover, only the entries below
%             and on the diagonal of the symmetric Lanczos matrix T are
%             computed directly via inner products and norms of candidate
%             vectors and previously deflated vectors.
% 
%          5) This algorithm has no built-in look-ahead procedure to continue
%             in the case of a breakdown or near-breakdown, and instead, the
%             algorithm simply stops in this case.  
%
% -----------------------------------------------------------------------------
%
%  Usage:  result = sym_band_Lanczos(A,R,nmax)
%          result = sym_band_Lanczos(A,R,nmax,sflag)
%          result = sym_band_Lanczos(A,R,nmax,sflag,tol)
%          result = sym_band_Lanczos(A,R,nmax,sflag,tol,n,result)
% 
%          where is a symmetric matrix, or,
%
%          result = sym_band_Lanczos(@(x) afun(x,...),R,nmax)
%          result = sym_band_Lanczos(@(x) afun(x,...),R,nmax,sflag)
%          result = sym_band_Lanczos(@(x) afun(x,...),R,nmax,sflag,tol)
%          result = sym_band_Lanczos(@(x) afun(x,...),R,nmax,sflag, ...
%                                                          tol,n,result)
%
%          where "afun" is a function such that 
%
%          y = afun(x,...)
%
%          computes the matrix-vector product y = A x (= A * x) with A 
%
% -----------------------------------------------------------------------------
%         
%  Required inputs:  A = a symmetric matrix or an (anonymous) function that 
%                        computes matrix-vector products with A 
%                    R = matrix the m columns of which are the starting vectors
%                 nmax = maximum number of Lanczos vectors to be generated
%
%                It is assumed that A is a symmetric matrix and that A and R
%                have the same number of rows;  these assumptions are checked
%                when the input A is a matrix, but not when A is a function. 
%
% -----------------------------------------------------------------------------
%
%  Optional inputs:  sflag = an integer that controls the storage of the 
%                            Lanczos vectors:  sflag = 0 means that only the 
%                            vectors needed for the "orthogonalizations" are
%                            stored;  a nonzero value of sflag means that all
%                            vectors are stored
%                      tol = a structure that contains tolerances and 
%                            parameters for the deflation procedure and the
%             		     breakdown check
%                        n = a nonnegative integer;  n > 0 means that a
%                            previous call to the function "sym_band_Lanczos" 
%                            has generated n Lanczos vectors and that the
%                            current call to "sym_band_Lanczos" resumes the
%                            iteration at step n+1;  n = 0 means that the band
%                            symmetric Lanczos method is started from scratch
%                   result = the output structure of the previous call to
%                            "symm_band_Lanczos" if n > 0;  if n = 0, this
%                            input is ignored
%
%                 If sflag is not provided as an input, sflag = 0 is used.
%
%                 If "tol" is provided as an input, it needs to contain 
%
%                    tol.defl_tol = unscaled deflation tolerance
%
%                    tol.brk_tol = tolerance for breakdown check
%
%                 and values of the following two flags:
%
%                    tol.defl_flag = 0  (use unscaled deflation tolerance)
%                                    1  (use scaled deflation tolerance)
%                                    2  (use scaled deflation tolerance only
%                                                   for the starting block R)
%                                    3  (use scaled deflation tolerance except
%                                                     for the starting block R)
%
%                    tol.normA_flag = 1  (an estimate for the norm of A is
%                                           generated within the algorithm)
%                                     2  (an estimate for the norm of A is
%                                                    provided as tol.normA)
%
%                 If tol.normA_flag = 2, then an estimate for the norm of A 
%                 needs to be provided as 
%
%                    tol.normA
%
%                 If "tol" is not provided as an input, the following default
%                 values are used:
%
%                    tol.defl_tol = sqrt(eps)  (eps = machine precision)
%                    tol.brk_tol = 100 * eps;
%                    tol.defl_flag = 1
%                    tol.normA_flag = 1
%
%                 If n is not provided as an input, n = 0 is used.
%
%                 If n > 0, the input "result" needs to be provided.  It is
%                 assumed, but not checked, that "result" is the output 
%                 structure of an earlier call to the function 
%                 "sym_band_Lanczos" (applied to the same matrices A and R). 
%                 In this case, the inputs "sflag" and "tol" are ignored and
%                  
%                    tol = result.tol 
%
%                 and
%
%                    sflag = result.sflag
%
%                 are used instead.
%
% -----------------------------------------------------------------------------
%
%  On return, "result" is a structure the fields of which include the following
%  quantities:
%
%         result.n = number of Lanczos vectors that were generated
%
%         result.V = matrix V the columns of which are the stored Lanczos 
%                    vectors;  if sflag = 0, V has at most m+1 columns and
%                    these columns are the Lanczos vectors that are needed for
%                    "orthogonalization";  if sflag is nonzero, V contains the 
%                    first n Lanczos vectors
%
%   result.Vh_defl = matrix Vh_defl the columns of which are the candidate 
%                    vectors for the next mc Lanczos vectors and the m-mc 
%                    deflated vectors 
% 
%         result.D = vector D the entries of which are the products
%
%                             (V(:,k)).' * V(:,k),  k = 1, 2, ..., n
%
%         result.T = the n x n matrix T that represents the oblique projection
%                    of the matrix A onto the subspace spanned by the columns
%                    of V, and "orthogonally" to the subspace spanned by the
%                    columns of V;  A, V, and T are connected via the relation
%
%                             V.' * A * V = (V.' * V) * T
%
%       result.rho = the matrix rho that contains the coefficients used to turn
%                    the starting vectors (in R) into the first Lanczos
%                    vectors;  R, V, and rho are connected via the relation
%
%                             V.' * R = (V.' * V) * rho
%
% -----------------------------------------------------------------------------
%
%  This routine can be run in incremental fashion. 
%
%  Example 1:   
%
%      result = sym_band_Lanczos(A,R,nmax0,sflag,tol)
%
%      n = result.n					 
%
%      result = sym_band_Lanczos(A,R,nmax,sflag,tol,n,result)
%
%      The first call to the function "sym_band_Lanczos" runs the symmetric
%      band Lanczos method from scratch and generates n Lanczos vectors.
% 
%      The second call to the function of "sym_band_Lanczos" resumes the 
%      iteration at step n+1.
%
%  Example 2 (Symmetric band Lanczos method, run one step at a time):
%
%      result = sym_band_Lanczos(A,R,1,sflag,tol,0,[])
% 
%      for n = 1 : nmax - 1,
% 
%         result = sym_band_Lanczos(A,R,n+1,[],[],result.n,result)
%  
%      end
% 
%      This will run the symmetric band Lanczos method for nmax steps.
%
% -----------------------------------------------------------------------------
%
%  BANDITS: a Matlab Package of Band Krylov Subspace Iterations
%
%  Copyright (c) 2018-2019 Roland W. Freund
%  See LICENSE.txt for license
%
% -----------------------------------------------------------------------------

%  =======================
%  Begin of initialization
%  =======================
%
if nargin < 3,
   error('** Not enough input arguments! **')
end
%
if (nargin < 4) || isempty(sflag),
   sflag = 0;
else
   if (rem(sflag,1) ~= 0),
      error('** sflag needs to be an integer **')
   end  
end  
%
if (nargin < 5) || isempty(tol),
   tol.defl_flag = 1;
   tol.defl_tol = sqrt(eps);
   tol.normA_flag = 1;
   tol.brk_tol = 100 * eps;
end
%
if (nargin < 6) || isempty(n),
   n = 0;
else
   if (n < 0) || (rem(n,1) ~= 0),
      error('** n needs to be a nonnegative integer **')
   end  
end
%
if nmax <= n,
   error('** nmax is not large enough;  we need to have nmax > n **')
end  
%
[N,m] = size(R);
%
if isfloat(A) == 1,
   if isreal(A) & isreal(R),
      fprintf(' \n')
      disp('**--------------------------------------------------**')    
      disp('** Warning:  The matrices A and R are both real;    **')
      disp('** in this case, the function "Herm_band_Lanczos.m" **')
      disp('** should be used instead of "sym_band_Lanczos.m"   **')
      disp('**--------------------------------------------------**')
      fprintf(' \n') 
   end
   mvec_A = 1;
   [Nt1,Nt2] = size(A);
   if Nt1 ~= Nt2,
      error('** The matrix A is not square **')
   end
   if nnz(A-A.') > 0,
      error('** The matrix A is not symmetric **')
   end
   if Nt1 ~= N,
      error('** The matrices A and R need to have the same number of rows **')
   end
else
   mvec_A = 0;
end
%
if n > 0,
   if (nargin < 7) || isempty(result),
      error('** n > 0, but there is no input "result" **')
   end   
   V = result.V;
   Vh_defl = result.Vh_defl;
   rho = result.rho;
   T = result.T;
   D = result.D;
   mc = result.mc;
   Iv = result.Iv;
   n_check = result.n;
   sflag = result.sflag;
   tol = result.tol;
   brk_flag = result.brk_flag;
   exh_flag = result.exh_flag;
%
   if brk_flag == 1,
      fprintf(' \n')
      disp('**-------------------------------------------**')
      disp('** Previous run ended with breakdown;        **')
      disp('** look-ahead is needed in order to continue **')    
      disp('**-------------------------------------------**')
      fprintf(' \n')    
      return
   end
%  
   if exh_flag > 0,
      fprintf(' \n')
      disp('**---------------------------------**')    
      disp('** Previous run ended due to an    **')
      disp('** exhausted block Krylov subspace **')    
      disp('**---------------------------------**')
      fprintf(' \n') 
      return
   end
%
   if n ~= n_check,
      error('** n does not match the value of n in result **')
   end
%  
   n1 = n + 1;
%
else
   V = zeros(N,0);
   Vh_defl(:,1:m) = R;
   D = [];
   T = [];
   rho = [];
   mc = m;
   Iv.v = [];
   if sflag == 0,
      Iv.av = [1:m+1];
   end
   Iv.ph = [1:m];
   Iv.I = [];
   Iv.pd = [];  
   Iv.nd = 0;
   n1 = 1;
   brk_flag = 0;
   exh_flag = 0;
end
%
result.m = m;
%
%  Extract and check tolerances, flags, and norm estimate for deflation 
%  and breakdown checks
%
[defl_tol,defl_flag,normA_flag,normA,brk_tol] = check_tolerances(tol,n);
%
%  ============================================
%  End of initialization and begin of iteration
%  ============================================
%				    
for n = n1 : nmax,
%
%  =================================
%  Construct n-th Lanczos vector v_n
%  =================================
%
   foundvn = 0;
%   
%  If necessary, deflate v 
%
   while foundvn == 0,
%
      [mc,foundvn,Vh_defl,Iv,normv] = deflation(0,n,m,mc,foundvn, ...
                               Vh_defl,R,Iv,defl_flag,defl_tol,normA);
%       
%  Check if block Krylov subspace is exhausted
%
      if mc == 0,
%             
         disp('**------------------------------------------**')
         disp('** There are no more Krylov vectors, and so **')
         disp('** the algorithm has to terminate: STOP     **')
         disp('**------------------------------------------**')
         disp(['  Number of Lanczos steps performed: ' num2str(n-1)])
%
         tol.normA = normA;
         result = save_result(3,V,Vh_defl,mc,Iv,T,rho,tol,D);
%
         result.n = n - 1;
         result.sflag = sflag;   
         result.brk_flag = brk_flag;
         result.exh_flag = 1;      
         return
%         
      end
%
%  End of: while foundvn == 0
%
   end
%
%  Normalize v_n
%
   tmpv = Vh_defl(:,Iv.ph(1)) / normv;
%
%  Compute delta_n and check for breakdown
%
   delta = tmpv.' * tmpv;
%
   if abs(delta) <= brk_tol,
      disp('**------------------------------- **')
      disp('** Breakdown --- look-ahead would **')
      disp('** be needed to continue: STOP    **')
      disp('**--------------------------------**')
      disp(['  Number of Lanczos steps performed: ' num2str(n-1)])
      disp(['  Number of Lanczos steps performed: ' num2str(n-1)])
      disp(['  Tolerance for breakdown check:     ' num2str(brk_tol)])
      disp(['  Absolute value of delta = v^T * v: ' num2str(abs(delta))])
      disp('**---------------------------------------------**'), fprintf(' \n')
%
      tol.normA = normA;
      result = save_result(3,V,Vh_defl,mc,Iv,T,rho,tol,D);
%
      result.n = n - 1;
      result.sflag = sflag;      
      result.brk_flag = 1;
      result.exh_flag = 0;      
      return      
   end
%  
   D(n,1) = delta;
%    
   if sflag == 0,
      nvi = min(Iv.av);
      Iv.av = setdiff(Iv.av,nvi);
   else
      nvi = n;
   end
%  
   V(:,nvi) = Vh_defl(:,Iv.ph(1)) / normv;
   Iv.v(n) = nvi;
%
%  Make sure rho has n rows
%
   rho(n,1) = 0;
%    
   if n > mc,
      T(n,n-mc) = normv;
   else
      rho(n,n-mc+m) = normv;
   end
%
%  "Orthogonalize" the candidate vectors against v_n
%
   ivph1 = Iv.ph(1);
   Itmp = Iv.ph(2:mc);
   Iv.ph(1:mc-1) = Itmp;
   Iv.ph(mc) = ivph1;
%
   tmp = ((V(:,nvi)).' * Vh_defl(:,Itmp)) / delta;
   Vh_defl(:,Itmp) = Vh_defl(:,Itmp) - V(:,nvi) * tmp; 
%
   Ktmp = find([1:mc-1] > mc-n);
   T(n,Ktmp-mc+n) = tmp(Ktmp);
%  
   Ktmp = find([1:mc-1] <= mc-n);
   rho(n,Ktmp-mc+n+m) = tmp(Ktmp);  
%  
%  Advance block Krylov subspace by computing tmpv = A * V(:,nvi) (= A v_n)
%
   if mvec_A == 1,
      tmpv = A * V(:,nvi);
   else
      tmpv = feval(A,V(:,nvi));
   end
%
   if normA_flag == 1,
      normA = max([normA,norm(tmpv,2)]);
   end
%
%  "Orthogonalize" tmpv against v vectors
%    
   nd = Iv.nd;
   tmp = (V(:,nvi)).' * Vh_defl(:,Iv.pd(1:nd));
%    
   Itmp = Iv.I(1:nd);
   Ktmp = find(Itmp > 0);
   IKtmp = Itmp(Ktmp);
   T(n,IKtmp) = tmp(Ktmp) / delta;
   T(IKtmp,n) = (tmp(Ktmp)).' ./ D(IKtmp,1);
   tmpv = tmpv - V(:,Iv.v(IKtmp)) * T(IKtmp,n);
%
   Ktmp = find(Itmp <= 0);
   rho(n,Itmp(Ktmp)+m) = tmp(Ktmp) / delta;
%
   Ktmp = max(1,n-mc):n-1; 
   T(Ktmp,n) = ((T(n,Ktmp)).' * delta) ./ D(Ktmp,1);
   tmpv = tmpv - V(:,Iv.v(Ktmp)) * T(Ktmp,n);
%    
   T(n,n) = ((V(:,nvi)).' * tmpv) / delta;
   Vh_defl(:,Iv.ph(mc)) = tmpv - V(:,nvi) * T(n,n);
%
%  Unless all Lanczos vectors are stored, adjust available slots
%  for Lanczos vectors in V
%
   if sflag == 0,
      if n > mc,
         if (ismember(n-mc,Iv.I) == 0),
            Iv.av = union(Iv.av,Iv.v(n-mc));
         end    
      end
   end
%  
end
%
%  ================
%  End of iteration
%  ================
%	 
tol.normA = normA;
result = save_result(3,V,Vh_defl,mc,Iv,T,rho,tol,D);
%
result.n = n;
result.sflag = sflag;   
result.brk_flag = 0;
result.exh_flag = 0;

