// This is a -*- C++ -*- header file.
// ESSO: Extensible Shell with Scheme Orientation
// Written by Matthias Koeppe <mkoeppe@csmd.cs.uni-magdeburg.de>

// The following implementation variants are available:
//
// TAGGEDPOINTERS -- Pointers to objects are required to be
// 16-aligned. Type tags are attached to the pointers. Values (small
// immutable things) are stored directly in LispRefs. This will slow
// down the code slightly and increase the code size, but will save a
// few malloc()s.
//
// HEAVY -- Some rather large things are inlined. This makes the code
// faster. 
//
// OWNEXCEPTIONS -- Instead of using the heavy-weight C++ exceptions,
// a simple own exception model is implemented. It has absolutely the
// same functionality (since we are using only a small subset of C++
// exception handling), but is much faster and saves a lot of space.

#if !defined(_LISP_H)
#define _LISP_H

#include "string"
#include <map>
#include "pipebase.h"

typedef string TLispName;

// Lisp exception base class

class XLisp {
public:
  virtual string Message() const = 0;
};

// Lisp objects and references

class TLispNull;
class TLispNumber;
class TLispSymbol;
class TLispString;
class TLispLambda;
class TLispPair;
class TLispFrame;
class TLispPrimitive;
class TLispBoolean;
class TLispChar;
class TLispInputPort;
class TLispOutputPort;
class TLispPromiseBase;
class TLispPromise;
class TLispPortReadPromise;

class TLispBase {
public:
#if defined(TAGGEDPOINTERS)
  static const long TypeMask = 15L;
  static const long PtrMask = ~TypeMask;
  static const ValueShift = 4;
#endif
  typedef enum {
    Null,
    Number,
    Eof,
    Boolean,
    Char, _MaxValue = Char,
    Pair,
    String,
    Symbol,
    Lambda,
    Frame,
    Primitive,
    InputPort,
    OutputPort,
    Executable,
    Promise
  } TLispType;
  static string NameOfType(TLispType type) {
    switch (type) {
    case Null: return "null";
    case Number: return "number";
    case Pair: return "pair";
    case String: return "string";
    case Symbol: return "symbol";
    case Lambda: return "lambda";
    case Frame: return "frame";
    case Primitive: return "primitive";
    case Eof: return "eof";
    case InputPort: return "input-port";
    case OutputPort: return "output-port";
    case Boolean: return "boolean";
    case Char: return "char";
    case Promise: return "promise";
    }
    return "unknown";
  }

};

class TLispObject {
  int RefCount;
  friend class TLispRef;
  void Increase() { RefCount++; }
  void Decrease() {
    if (!--RefCount) DeleteThis(); 
  }
  TLispObject &operator=(const TLispObject &); // no copy
public:
  TLispObject()  : RefCount(0) {}
  virtual ~TLispObject()  {}
  //& operators new and delete required.
  // If we have circular references, garbage will occur.
  // This must be detected (`garbage collection').

  TLispBase::TLispType Type() { return type; }

  virtual bool IsEq(TLispObject &O) { return this == &O; }
  virtual bool IsTrue() { return true; }
  virtual TLispRef Apply(TLispRef ArgList);

  friend TLispNumber *CastNumber(TLispObject &object) {return (TLispNumber *) &object;}
  friend TLispPair *CastPair(TLispObject &object) {return (TLispPair *) &object;}
  friend TLispString *CastString(TLispObject &object) {return (TLispString *) &object;}
  friend TLispSymbol *CastSymbol(TLispObject &object) {return (TLispSymbol *) &object;}
  friend TLispLambda *CastLambda(TLispObject &object) {return (TLispLambda *) &object;}
  friend TLispFrame *CastFrame(TLispObject &object) {return (TLispFrame *) &object;}
  friend TLispPrimitive *CastPrimitive(TLispObject &object) {return (TLispPrimitive *) &object;}
  friend TLispChar *CastChar(TLispObject &object) {return (TLispChar *) &object;}
  friend TLispBoolean *CastBoolean(TLispObject &object) {return (TLispBoolean *) &object;}
  friend TLispInputPort *CastInputPort(TLispObject &object) {return (TLispInputPort *) &object;}
  friend TLispOutputPort *CastOutputPort(TLispObject &object) {return (TLispOutputPort *) &object;}

protected:
  TLispBase::TLispType type;
  void SetType(TLispBase::TLispType t) { type = t;}
  void DeleteThis();
};

class XInvalidType : public XLisp {
  TLispBase::TLispType ActualType, ExpectedType;
public:
  XInvalidType(TLispBase::TLispType actual, TLispBase::TLispType expected) :
    ActualType(actual), ExpectedType(expected) {}
  virtual string Message() const {
    return string("XInvalidType: ")
      + TLispBase::NameOfType(ActualType)
      + " (expected "
      + TLispBase::NameOfType(ExpectedType)
      + ")";
  }
};

//
// Values: Small immutable lisp data types
//

#if defined(TAGGEDPOINTERS)

class TLispValue {
  int data;
public:
  TLispValue(TLispBase::TLispType t, int value = 0) :
    data((value << TLispBase::ValueShift) | t) {}
  TLispBase::TLispType Type() { return (TLispBase::TLispType) (data & TLispBase::TypeMask); }
  int Value() { return data >> TLispBase::ValueShift; }
};

// Lisp Null

class TLispNull : public TLispValue {
public:
  TLispNull() : TLispValue(TLispBase::Null) {}
};

// Lisp Numbers

class TLispNumber : public TLispValue {
public:
  TLispNumber(int v) : TLispValue(TLispBase::Number, v) {}
};

// Lisp Booleans

class TLispBoolean : public TLispValue {
public:
  TLispBoolean(bool v) : TLispValue(TLispBase::Boolean, v) {}
};

// Lisp Characters

class TLispChar : public TLispValue {
public:
  TLispChar(char c) : TLispValue(TLispBase::Char, c) {}
};
  
// TLispEof

class TLispEof : public TLispValue {
public:
  TLispEof() : TLispValue(TLispBase::Eof) {}
};

#else

// Lisp Null

class TLispNull : public TLispObject {
public:
  TLispNull() {SetType(TLispBase::Null);}
  virtual bool IsEq(TLispObject &) { return true; }
};

// Lisp Numbers

class TLispNumber : public TLispObject {
  int value;
public:
  TLispNumber(int v) : value(v) {SetType(TLispBase::Number);}
  int &Value() { return value; }
  virtual bool IsEq(TLispObject &N) { return value == CastNumber(N)->value; }
};

// Lisp Booleans

class TLispBoolean : public TLispObject {
  bool value;
public:
  TLispBoolean(bool v) : value(v) {SetType(TLispBase::Boolean);}
  virtual bool IsEq(TLispObject &N) { return value == CastBoolean(N)->value; }
  virtual bool IsTrue() { return value; }
};

// Lisp Characters

class TLispChar : public TLispObject {
  char value;
public:
  TLispChar (char c) : value(c) {SetType(TLispBase::Char);}
  virtual bool IsEq(TLispObject &N) { return value == CastChar(N)->value; }
  char &Value() { return value; }
};
  
// TLispEof

class TLispEof : public TLispObject {
public:
  TLispEof() {SetType(TLispBase::Eof);}
  virtual bool IsEq(TLispObject &S) { return true; }
};

typedef TLispRef TLispValue;

#endif

class TLispRef {
#if defined(TAGGEDPOINTERS)
  int valueorobject;
  TLispBase::TLispType TypeTag() const {
    return (TLispBase::TLispType) (valueorobject & TLispBase::TypeMask); }
  TLispObject *ObjectPtr() const {
    return (TLispObject *) (valueorobject & TLispBase::PtrMask);
  }
  TLispValue *ValuePtr() const {
    return (TLispValue *) &valueorobject;
  }
  bool IsValue() const {
    return TypeTag() <= TLispBase::_MaxValue;
  }
  bool IsObject() const {
    return (TypeTag() > TLispBase::_MaxValue); // && ObjectPtr();
  }
public:
  TLispRef() : valueorobject(0) {}
  TLispRef(TLispObject *obj) :
    valueorobject(obj ? (obj->Increase(), (((int) obj) | obj->Type()))
		  : 0) {}
  TLispRef(const TLispValue &value) :
    valueorobject(* ((int *) &value)) {} 
  TLispRef(const TLispRef &C) : valueorobject(C.valueorobject) {
    if (IsObject()) ObjectPtr()->Increase();
  }
  ~TLispRef()  {
    if (IsObject()) ObjectPtr()->Decrease();
  }
  TLispRef &operator=(const TLispRef &C)  {
    if (valueorobject != C.valueorobject) {
      // This order ensures that `object' never stores an invalid pointer
      TLispObject *oldobject = IsObject() ? ObjectPtr() : 0;
      if (C.IsObject()) C.ObjectPtr()->Increase();
      valueorobject = C.valueorobject;
      if (oldobject) oldobject->Decrease();
    }
    return *this;
  }
  TLispBase::TLispType Type() {
    if (TypeTag() == TLispBase::Promise) DoForce();
    return TypeTag();
  }
  inline TLispObject* Object() {
    if (TypeTag() == TLispBase::Promise) DoForce();
    return ObjectPtr();
  }
  TLispPromiseBase *AsPromise() {
    if (TypeTag() == TLispBase::Promise)
      return (TLispPromiseBase *) ObjectPtr;
    else return 0;
  }
#else
  TLispObject *object;
  TLispObject *ObjectPtr() {return object;}
  TLispObject *ValuePtr() {return object;}
public:
  TLispRef()  : object(0) {}
  TLispRef(TLispObject *obj)  : object(obj) {
    if (object) object->Increase(); }
  TLispRef(const TLispRef &C)  : object(C.object) {
    if (object) object->Increase(); }
  TLispRef &operator=(const TLispRef &C)  {
    if (object != C.object) {
      // This order ensures that `object' never stores an invalid pointer
      TLispObject *oldobject = object;
      if (C.object) C.object->Increase();
      object = C.object;
      if (oldobject) oldobject->Decrease();
    }
    return *this;
  }
  ~TLispRef()  {
    if (object) object->Decrease();
  }
  inline TLispObject* Object() {
    // do implicit forcing
    if (object->Type() == TLispBase::Promise) DoForce();
    return object;
  }
  TLispBase::TLispType Type() { return Object()->Type(); }
  TLispPromiseBase *AsPromise() {
    if (object->Type() == TLispBase::Promise)
      return (TLispPromiseBase *) object;
    else return 0;
  }
#endif

  //operator TLispObject&() {return *Object();}
  string TypeName() { return TLispBase::NameOfType(Type()); }

  bool IsNull() { return Type() == TLispBase::Null; }
  bool IsNumber() { return Type() == TLispBase::Number; }
  bool IsPair() { return Type() == TLispBase::Pair; }
  bool IsString() { return Type() == TLispBase::String; }
  bool IsSymbol() { return Type() == TLispBase::Symbol; }
  bool IsLambda() { return Type() == TLispBase::Lambda; }
  bool IsPrimitive() { return Type() == TLispBase::Primitive; }
  bool IsChar() { return Type() == TLispBase::Char; }
  bool IsBoolean() { return Type() == TLispBase::Boolean; }

#if defined(HEAVY)
  inline friend void AssertType(TLispRef &ref, TLispBase::TLispType type) {
    if (ref.Type() != type) 
      ThrowInvalidType(ref.Type(), type);
  }
#else
  friend void AssertType(TLispRef &ref, TLispBase::TLispType type);
#endif
  friend TLispNumber *AssertNumber(TLispRef &ref) {
    AssertType(ref, TLispBase::Number);
    return (TLispNumber *) ref.ValuePtr();
  }
  friend void AssertNull(TLispRef &ref) {
    AssertType(ref, TLispBase::Null);
  }
  friend TLispChar *AssertChar(TLispRef &ref) {
    AssertType(ref, TLispBase::Char);
    return (TLispChar *) ref.ValuePtr();
  }
  friend TLispBoolean *AssertBoolean(TLispRef &ref) {
    AssertType(ref, TLispBase::Boolean);
    return (TLispBoolean *) ref.ValuePtr();
  }
  friend TLispPair *AssertPair(TLispRef &ref) {
    AssertType(ref, TLispBase::Pair);
    return (TLispPair *) ref.ObjectPtr();
  }
  friend TLispString *AssertString(TLispRef &ref) {
    AssertType(ref, TLispBase::String);
    return (TLispString *) ref.ObjectPtr();
  }
  friend TLispSymbol *AssertSymbol(TLispRef &ref) {
    AssertType(ref, TLispBase::Symbol);
    return (TLispSymbol *) ref.ObjectPtr();
  }
  friend TLispLambda *AssertLambda(TLispRef &ref) {
    AssertType(ref, TLispBase::Lambda);
    return (TLispLambda *) ref.ObjectPtr();
  }
  friend TLispFrame *AssertFrame(TLispRef &ref) {
    AssertType(ref, TLispBase::Frame);
    return (TLispFrame *) ref.ObjectPtr();
  }
  friend TLispPrimitive *AssertPrimitive(TLispRef &ref) {
    AssertType(ref, TLispBase::Primitive);
    return (TLispPrimitive *) ref.ObjectPtr();
  }
  friend TLispInputPort *AssertInputPort(TLispRef &ref) {
    AssertType(ref, TLispBase::InputPort);
    return (TLispInputPort *) ref.ObjectPtr();
  }
  friend TLispOutputPort *AssertOutputPort(TLispRef &ref) {
    AssertType(ref, TLispBase::OutputPort);
    return (TLispOutputPort *) ref.ObjectPtr();
  }

  friend TLispNumber *CastNumber(TLispRef &ref) {return (TLispNumber *) ref.ValuePtr();}
  friend TLispChar *CastChar(TLispRef &ref) {return (TLispChar *) ref.ValuePtr();}
  friend TLispBoolean *CastBoolean(TLispRef &ref) {return (TLispBoolean *) ref.ValuePtr();}
  friend TLispPair *CastPair(TLispRef &ref) {return (TLispPair *) ref.ObjectPtr();}
  friend TLispString *CastString(TLispRef &ref) {return (TLispString *) ref.ObjectPtr();}
  friend TLispSymbol *CastSymbol(TLispRef &ref) {return (TLispSymbol *) ref.ObjectPtr();}
  friend TLispLambda *CastLambda(TLispRef &ref) {return (TLispLambda *) ref.ObjectPtr();}
  friend TLispFrame *CastFrame(TLispRef &ref) {return (TLispFrame *) ref.ObjectPtr();}
  friend TLispPrimitive *CastPrimitive(TLispRef &ref) {return (TLispPrimitive *) ref.ObjectPtr();}
  friend TLispInputPort *CastInputPort(TLispRef &ref) {return (TLispInputPort *) ref.ObjectPtr();}
  friend TLispOutputPort *CastOutputPort(TLispRef &ref) {return (TLispOutputPort *) ref.ObjectPtr();}

  inline TLispRef &Car();
  inline TLispRef &Cdr();
  inline TLispRef &Caar();
  inline TLispRef &Cadr();
  inline TLispRef &Cdar();
  inline TLispRef &Cddr();
  inline TLispRef &Caaar();
  inline TLispRef &Caadr();
  inline TLispRef &Cadar();
  inline TLispRef &Caddr();
  inline TLispRef &Cdaar();
  inline TLispRef &Cdadr();
  inline TLispRef &Cddar();
  inline TLispRef &Cdddr();
  inline TLispRef &Caaaar();
  inline TLispRef &Caaadr();
  inline TLispRef &Caadar();
  inline TLispRef &Caaddr();
  inline TLispRef &Cadaar();
  inline TLispRef &Cadadr();
  inline TLispRef &Caddar();
  inline TLispRef &Cadddr();
  inline TLispRef &Cdaaar();
  inline TLispRef &Cdaadr();
  inline TLispRef &Cdadar();
  inline TLispRef &Cdaddr();
  inline TLispRef &Cddaar();
  inline TLispRef &Cddadr();
  inline TLispRef &Cdddar();
  inline TLispRef &Cddddr();

  inline TLispRef &CAr();
  inline TLispRef &CDr();
  inline TLispRef &CaAr();
  inline TLispRef &CaDr();
  inline TLispRef &CdAr();
  inline TLispRef &CdDr();
  inline TLispRef &CaaAr();
  inline TLispRef &CaaDr();
  inline TLispRef &CadAr();
  inline TLispRef &CadDr();
  inline TLispRef &CdaAr();
  inline TLispRef &CdaDr();
  inline TLispRef &CddAr();
  inline TLispRef &CddDr();
  inline TLispRef &CaaaAr();
  inline TLispRef &CaaaDr();
  inline TLispRef &CaadAr();
  inline TLispRef &CaadDr();
  inline TLispRef &CadaAr();
  inline TLispRef &CadaDr();
  inline TLispRef &CaddAr();
  inline TLispRef &CaddDr();
  inline TLispRef &CdaaAr();
  inline TLispRef &CdaaDr();
  inline TLispRef &CdadAr();
  inline TLispRef &CdadDr();
  inline TLispRef &CddaAr();
  inline TLispRef &CddaDr();
  inline TLispRef &CdddAr();
  inline TLispRef &CdddDr();

  inline TLispRef &CAAr();
  inline TLispRef &CADr();
  inline TLispRef &CDAr();
  inline TLispRef &CDDr();
  inline TLispRef &CaAAr();
  inline TLispRef &CaADr();
  inline TLispRef &CaDAr();
  inline TLispRef &CaDDr();
  inline TLispRef &CdAAr();
  inline TLispRef &CdADr();
  inline TLispRef &CdDAr();
  inline TLispRef &CdDDr();
  inline TLispRef &CaaAAr();
  inline TLispRef &CaaADr();
  inline TLispRef &CaaDAr();
  inline TLispRef &CaaDDr();
  inline TLispRef &CadAAr();
  inline TLispRef &CadADr();
  inline TLispRef &CadDAr();
  inline TLispRef &CadDDr();
  inline TLispRef &CdaAAr();
  inline TLispRef &CdaADr();
  inline TLispRef &CdaDAr();
  inline TLispRef &CdaDDr();
  inline TLispRef &CddAAr();
  inline TLispRef &CddADr();
  inline TLispRef &CddDAr();
  inline TLispRef &CddDDr();

  inline TLispRef &CAAAr();
  inline TLispRef &CAADr();
  inline TLispRef &CADAr();
  inline TLispRef &CADDr();
  inline TLispRef &CDAAr();
  inline TLispRef &CDADr();
  inline TLispRef &CDDAr();
  inline TLispRef &CDDDr();
  inline TLispRef &CaAAAr();
  inline TLispRef &CaAADr();
  inline TLispRef &CaADAr();
  inline TLispRef &CaADDr();
  inline TLispRef &CaDAAr();
  inline TLispRef &CaDADr();
  inline TLispRef &CaDDAr();
  inline TLispRef &CaDDDr();
  inline TLispRef &CdAAAr();
  inline TLispRef &CdAADr();
  inline TLispRef &CdADAr();
  inline TLispRef &CdADDr();
  inline TLispRef &CdDAAr();
  inline TLispRef &CdDADr();
  inline TLispRef &CdDDAr();
  inline TLispRef &CdDDDr();

  inline TLispRef &CAAAAr();
  inline TLispRef &CAAADr();
  inline TLispRef &CAADAr();
  inline TLispRef &CAADDr();
  inline TLispRef &CADAAr();
  inline TLispRef &CADADr();
  inline TLispRef &CADDAr();
  inline TLispRef &CADDDr();
  inline TLispRef &CDAAAr();
  inline TLispRef &CDAADr();
  inline TLispRef &CDADAr();
  inline TLispRef &CDADDr();
  inline TLispRef &CDDAAr();
  inline TLispRef &CDDADr();
  inline TLispRef &CDDDAr();
  inline TLispRef &CDDDDr();

  inline string &SymbolName();

#if defined(TAGGEDPOINTERS)
  bool IsTrue() { return (!IsBoolean()) || (CastBoolean(*this)->Value()); }
  bool IsEq(TLispRef Ref) {
    return valueorobject == Ref.valueorobject
      || (IsObject() && Object()->IsEq(*Ref.Object())); }
  inline TLispRef Apply(TLispRef ArgList); 
#else
  bool IsTrue() { return Object()->IsTrue(); }
  bool IsEq(TLispRef Ref) { return object->IsEq(*Ref.object); }
  TLispRef Apply(TLispRef ArgList) {return Object()->Apply(ArgList);}
#endif

protected:
  void DoForce(); //throw();
  friend void ThrowInvalidType(TLispBase::TLispType objecttype, TLispBase::TLispType type);
};

// `TLispRef on stack'

#if defined(OWNEXCEPTIONS)
//& If we have multiple stacks (multiple processes), there should be
//multiple lists as well. 
class TLispRefOS : public TLispRef {
  TLispRefOS *Next;
protected:
  static TLispRefOS *First;
  // registration
  inline void Register() {
    Next = First;
    First = this;
  }
  void Unregister();
public:
  TLispRefOS() { Register(); }
  TLispRefOS(TLispObject *obj) : TLispRef(obj) { Register(); }
#if defined(TAGGEDPOINTERS)
  TLispRefOS(const TLispValue &value) : TLispRef(value) {
    Register(); }
#endif
  TLispRefOS(const TLispRef &C) : TLispRef(C) { Register(); }
  TLispRefOS(const TLispRefOS &C) : TLispRef(C) { Register(); }
  ~TLispRefOS() {
    if (First == this) First = Next;
    else Unregister();
  }
  TLispRefOS &operator=(const TLispRefOS &C) {
    TLispRef::operator=(C);
    return *this;
  }
public:
  static void KillAllBetween(void *Lower, void *Higher);
};

class TTry {
public:
  void *minstack, *maxstack; /* identify thread */
  void *stack, *frame, *base;
  void *continuation;
  TTry *Next;
};

extern XLisp *Exception;
extern TTry *Tries;
void ThrowException(XLisp *X);
void RegisterThreadWithTries(void *minstack, void *maxstack);
void UnregisterThreadWithTries();
void TryFollowing();
#define THROW(x) ThrowException(new x)
#define TRY					\
  TryFollowing();				\
  if (!Exception) 

#else

typedef TLispRef TLispRefOS;
#define THROW(x) throw x
#define TRY try

#endif

//
// Structures: Possibly mutable, or large, data types
//

class TLispPromiseBase : public TLispObject {
protected:
  TLispRef Value;
  bool Ready;
public:
  TLispPromiseBase() : Ready(false) {SetType(TLispBase::Promise);}
  virtual TLispInputPort *InputPort() { return 0; }
  virtual TLispRef Force() = 0;
};

// Lambda Promise

class TLispPromise : public TLispPromiseBase {
  TLispRef Expression;
public:
  TLispPromise(TLispRef expression) : Expression(expression) {}
  virtual TLispRef Force();
};

// Port read promise

class TLispPortReadPromise : public TLispPromiseBase {
  TLispRef Port;
public:
  TLispPortReadPromise(TLispRef port) : Port(port) {
    AssertInputPort(Port);
  }
  virtual TLispRef Force();
  virtual TLispInputPort *InputPort() { return CastInputPort(Port); }
};

// Lisp Pairs

class TLispPair : public TLispObject {
  TLispRef car, cdr;
public:
  TLispPair(TLispRef acar, TLispRef acdr) :
    car(acar), cdr(acdr) {SetType(TLispBase::Pair);}
  TLispRef &Car() { return car; }
  TLispRef &Cdr() { return cdr; }
};

// Lisp Strings

class TLispString : public TLispObject {
  string value;
public:
  TLispString(const string &val) : value(val) {SetType(TLispBase::String);}
  string &Value() { return value; }
};
  
// Lisp Symbols

class TLispSymbol : public TLispObject {
  string name;
public:
  TLispSymbol(const string &n) : name(n) {SetType(TLispBase::Symbol);}
  string &Name() { return name; }
  virtual bool IsEq(TLispObject &S) { return name == CastSymbol(S)->name; }
};
  
// Lisp variable frames
  
class XNameNotFound : public XLisp {
  TLispName Name;
public:
  XNameNotFound(const TLispName &name) : XLisp(), Name(name) {}
  virtual string Message() const {
    return string("XNameNotFound: ") + Name;
  }
};

class TLispFrame : public TLispObject {
public:
  typedef map<TLispName, TLispRef, less<TLispName> > TLispMap;
private:
  TLispRef Parent;
  TLispMap Map;
  void ThrowXNameNotFound(const TLispName &Key);
public:
  TLispFrame(TLispRef parent) : Parent(parent) {SetType(TLispBase::Frame);}
  TLispRef *find(const TLispName &Key);
  TLispRef &operator() (const TLispName &Key) {
    TLispRef *Result = find(Key);
    if (Result) return *Result;
    ThrowXNameNotFound(Key);
  }
  void Define(const TLispName &name, TLispRef ref);
  TLispMap &GetMap() { return Map; }
  TLispRef GetParent() { return Parent; }
};

// Lisp Lambdas

class TLispLambda : public TLispObject {
  TLispRef Frame;
  TLispRef Formals;
  TLispRef Body;
public:
  TLispLambda(TLispRef frame, TLispRef formals, TLispRef body)
    : Frame(frame), Formals(formals), Body(body) {
      SetType(TLispBase::Lambda);
      AssertType(frame, TLispBase::Frame);
  }
  void ApplyAux(TLispRef ArgList, /*returns*/ TLispRef &Expression, /*returns*/ TLispRef &Frame);
  virtual TLispRef Apply(TLispRef ArgList); 
};

// TLispPrimitive

typedef TLispRef (*PrimitiveFunc)(TLispRef);

class TLispPrimitive : public TLispObject {
  PrimitiveFunc Func;
public:
  TLispPrimitive(PrimitiveFunc func) : Func(func) {SetType(TLispBase::Primitive);}
  virtual TLispRef Apply(TLispRef ArgList) {
    return (*Func)(ArgList);
  }
};

// TLispPort and derivatives

class TLispPort : public TLispObject {
public: //protected:
  TPipeBase<char> *Pipe;
  bool ShouldDelete;
public:
  TLispPort(TPipeBase<char> *pipe, bool shoulddelete = true)
    : Pipe(pipe), ShouldDelete(shoulddelete) {}
  virtual ~TLispPort() {
//     kprintf("123");
    Close();
//     kprintf("456");
  }
  virtual void Close() {
    if (ShouldDelete) {
      if (Pipe) delete Pipe;
      Pipe = 0;
    }
    else {
      if (Pipe) Pipe->Close();
      Pipe = 0;
    }
  }
};

class TLispInputPort : public TLispPort {
  char *CurrentBlock;
  char *CurrentPos;
  bool eof;
public:
  TLispInputPort(TPipeBase<char> *pipe) :
    TLispPort(pipe), CurrentPos(0), CurrentBlock(0), eof(false) {
      SetType(TLispBase::InputPort);}
  virtual ~TLispInputPort() { if (CurrentBlock) delete CurrentBlock; }
  char ReadChar(); 
  string ReadLine(); 
  bool Eof() { return eof; }
};

class TLispOutputPort : public TLispPort {
public:
  TLispOutputPort(TPipeBase<char> *pipe, bool shoulddelete = true)
    : TLispPort(pipe, shoulddelete) {SetType(TLispBase::OutputPort);}
  bool WriteChar(char c); 
  bool WriteLine(const string &s); 
};

// inlines

class XBadExternalSymbol : public XLisp {
  string SymbolName;
public:
  XBadExternalSymbol(string symbolname) : SymbolName(symbolname) {}
  virtual string Message() const {
    return string("XBadExternalSymbol: ")
      + SymbolName;
  }
};

class XBadStringIndex : public XLisp {
public:
  XBadStringIndex() {}
  virtual string Message() const {
    return string("XBadStringIndex");
  }
};

class XApply : public XLisp {
  TLispBase::TLispType OperatorType;
public:  
  XApply(TLispBase::TLispType operatortype) :
    OperatorType(operatortype) {}
  virtual string Message() const {
    return string("XApply: ")
      + TLispBase::NameOfType(OperatorType);
  }
};

class XArgCount : public XLisp {
  int ActualArgCount, RequiredArgCount;
public:
  XArgCount(int actual, int required) :
    ActualArgCount(actual), RequiredArgCount(required) {}
  virtual string Message() const {
    return string("XArgCount: ")
      + char(ActualArgCount + '0')
      + " (expected "
      + char(RequiredArgCount + '0')
      + ")";
  }
};

// note different order of `d' and `a'
// Capital 'D' and 'A' indicate `casted, not asserted'.
inline TLispRef &TLispRef::CAr() { return CastPair(*this)->Car(); }
inline TLispRef &TLispRef::CDr() { return CastPair(*this)->Cdr(); }
inline TLispRef &TLispRef::Car() { return AssertPair(*this)->Car(); }
inline TLispRef &TLispRef::Cdr() { return AssertPair(*this)->Cdr(); }

inline TLispRef &TLispRef::CAAr() { return CAr().CAr(); }
inline TLispRef &TLispRef::CADr() { return CDr().CAr(); }
inline TLispRef &TLispRef::CDAr() { return CAr().CDr(); }
inline TLispRef &TLispRef::CDDr() { return CDr().CDr(); }
inline TLispRef &TLispRef::CaAr() { return CAr().Car(); }
inline TLispRef &TLispRef::CaDr() { return CDr().Car(); }
inline TLispRef &TLispRef::CdAr() { return CAr().Cdr(); }
inline TLispRef &TLispRef::CdDr() { return CDr().Cdr(); }
inline TLispRef &TLispRef::Caar() { return Car().Car(); }
inline TLispRef &TLispRef::Cadr() { return Cdr().Car(); }
inline TLispRef &TLispRef::Cdar() { return Car().Cdr(); }
inline TLispRef &TLispRef::Cddr() { return Cdr().Cdr(); }

inline TLispRef &TLispRef::CAAAr() { return CAr().CAr().CAr(); }
inline TLispRef &TLispRef::CAADr() { return CDr().CAr().CAr(); }
inline TLispRef &TLispRef::CADAr() { return CAr().CDr().CAr(); }
inline TLispRef &TLispRef::CADDr() { return CDr().CDr().CAr(); }
inline TLispRef &TLispRef::CDAAr() { return CAr().CAr().CDr(); }
inline TLispRef &TLispRef::CDADr() { return CDr().CAr().CDr(); }
inline TLispRef &TLispRef::CDDAr() { return CAr().CDr().CDr(); }
inline TLispRef &TLispRef::CDDDr() { return CDr().CDr().CDr(); }
inline TLispRef &TLispRef::CaAAr() { return CAr().CAr().Car(); }
inline TLispRef &TLispRef::CaADr() { return CDr().CAr().Car(); }
inline TLispRef &TLispRef::CaDAr() { return CAr().CDr().Car(); }
inline TLispRef &TLispRef::CaDDr() { return CDr().CDr().Car(); }
inline TLispRef &TLispRef::CdAAr() { return CAr().CAr().Cdr(); }
inline TLispRef &TLispRef::CdADr() { return CDr().CAr().Cdr(); }
inline TLispRef &TLispRef::CdDAr() { return CAr().CDr().Cdr(); }
inline TLispRef &TLispRef::CdDDr() { return CDr().CDr().Cdr(); }
inline TLispRef &TLispRef::CaaAr() { return CAr().Car().Car(); }
inline TLispRef &TLispRef::CaaDr() { return CDr().Car().Car(); }
inline TLispRef &TLispRef::CadAr() { return CAr().Cdr().Car(); }
inline TLispRef &TLispRef::CadDr() { return CDr().Cdr().Car(); }
inline TLispRef &TLispRef::CdaAr() { return CAr().Car().Cdr(); }
inline TLispRef &TLispRef::CdaDr() { return CDr().Car().Cdr(); }
inline TLispRef &TLispRef::CddAr() { return CAr().Cdr().Cdr(); }
inline TLispRef &TLispRef::CddDr() { return CDr().Cdr().Cdr(); }
inline TLispRef &TLispRef::Caaar() { return Car().Car().Car(); }
inline TLispRef &TLispRef::Caadr() { return Cdr().Car().Car(); }
inline TLispRef &TLispRef::Cadar() { return Car().Cdr().Car(); }
inline TLispRef &TLispRef::Caddr() { return Cdr().Cdr().Car(); }
inline TLispRef &TLispRef::Cdaar() { return Car().Car().Cdr(); }
inline TLispRef &TLispRef::Cdadr() { return Cdr().Car().Cdr(); }
inline TLispRef &TLispRef::Cddar() { return Car().Cdr().Cdr(); }
inline TLispRef &TLispRef::Cdddr() { return Cdr().Cdr().Cdr(); }
  
inline TLispRef &TLispRef::CAAAAr() { return CAr().CAr().CAr().CAr(); }
inline TLispRef &TLispRef::CAAADr() { return CDr().CAr().CAr().CAr(); }
inline TLispRef &TLispRef::CAADAr() { return CAr().CDr().CAr().CAr(); }
inline TLispRef &TLispRef::CAADDr() { return CDr().CDr().CAr().CAr(); }
inline TLispRef &TLispRef::CADAAr() { return CAr().CAr().CDr().CAr(); }
inline TLispRef &TLispRef::CADADr() { return CDr().CAr().CDr().CAr(); }
inline TLispRef &TLispRef::CADDAr() { return CAr().CDr().CDr().CAr(); }
inline TLispRef &TLispRef::CADDDr() { return CDr().CDr().CDr().CAr(); }
inline TLispRef &TLispRef::CDAAAr() { return CAr().CAr().CAr().CDr(); }
inline TLispRef &TLispRef::CDAADr() { return CDr().CAr().CAr().CDr(); }
inline TLispRef &TLispRef::CDADAr() { return CAr().CDr().CAr().CDr(); }
inline TLispRef &TLispRef::CDADDr() { return CDr().CDr().CAr().CDr(); }
inline TLispRef &TLispRef::CDDAAr() { return CAr().CAr().CDr().CDr(); }
inline TLispRef &TLispRef::CDDADr() { return CDr().CAr().CDr().CDr(); }
inline TLispRef &TLispRef::CDDDAr() { return CAr().CDr().CDr().CDr(); }
inline TLispRef &TLispRef::CDDDDr() { return CDr().CDr().CDr().CDr(); }
inline TLispRef &TLispRef::CaAAAr() { return CAr().CAr().CAr().Car(); }
inline TLispRef &TLispRef::CaAADr() { return CDr().CAr().CAr().Car(); }
inline TLispRef &TLispRef::CaADAr() { return CAr().CDr().CAr().Car(); }
inline TLispRef &TLispRef::CaADDr() { return CDr().CDr().CAr().Car(); }
inline TLispRef &TLispRef::CaDAAr() { return CAr().CAr().CDr().Car(); }
inline TLispRef &TLispRef::CaDADr() { return CDr().CAr().CDr().Car(); }
inline TLispRef &TLispRef::CaDDAr() { return CAr().CDr().CDr().Car(); }
inline TLispRef &TLispRef::CaDDDr() { return CDr().CDr().CDr().Car(); }
inline TLispRef &TLispRef::CdAAAr() { return CAr().CAr().CAr().Cdr(); }
inline TLispRef &TLispRef::CdAADr() { return CDr().CAr().CAr().Cdr(); }
inline TLispRef &TLispRef::CdADAr() { return CAr().CDr().CAr().Cdr(); }
inline TLispRef &TLispRef::CdADDr() { return CDr().CDr().CAr().Cdr(); }
inline TLispRef &TLispRef::CdDAAr() { return CAr().CAr().CDr().Cdr(); }
inline TLispRef &TLispRef::CdDADr() { return CDr().CAr().CDr().Cdr(); }
inline TLispRef &TLispRef::CdDDAr() { return CAr().CDr().CDr().Cdr(); }
inline TLispRef &TLispRef::CdDDDr() { return CDr().CDr().CDr().Cdr(); }
inline TLispRef &TLispRef::CaaAAr() { return CAr().CAr().Car().Car(); }
inline TLispRef &TLispRef::CaaADr() { return CDr().CAr().Car().Car(); }
inline TLispRef &TLispRef::CaaDAr() { return CAr().CDr().Car().Car(); }
inline TLispRef &TLispRef::CaaDDr() { return CDr().CDr().Car().Car(); }
inline TLispRef &TLispRef::CadAAr() { return CAr().CAr().Cdr().Car(); }
inline TLispRef &TLispRef::CadADr() { return CDr().CAr().Cdr().Car(); }
inline TLispRef &TLispRef::CadDAr() { return CAr().CDr().Cdr().Car(); }
inline TLispRef &TLispRef::CadDDr() { return CDr().CDr().Cdr().Car(); }
inline TLispRef &TLispRef::CdaAAr() { return CAr().CAr().Car().Cdr(); }
inline TLispRef &TLispRef::CdaADr() { return CDr().CAr().Car().Cdr(); }
inline TLispRef &TLispRef::CdaDAr() { return CAr().CDr().Car().Cdr(); }
inline TLispRef &TLispRef::CdaDDr() { return CDr().CDr().Car().Cdr(); }
inline TLispRef &TLispRef::CddAAr() { return CAr().CAr().Cdr().Cdr(); }
inline TLispRef &TLispRef::CddADr() { return CDr().CAr().Cdr().Cdr(); }
inline TLispRef &TLispRef::CddDAr() { return CAr().CDr().Cdr().Cdr(); }
inline TLispRef &TLispRef::CddDDr() { return CDr().CDr().Cdr().Cdr(); }
inline TLispRef &TLispRef::CaaaAr() { return CAr().Car().Car().Car(); }
inline TLispRef &TLispRef::CaaaDr() { return CDr().Car().Car().Car(); }
inline TLispRef &TLispRef::CaadAr() { return CAr().Cdr().Car().Car(); }
inline TLispRef &TLispRef::CaadDr() { return CDr().Cdr().Car().Car(); }
inline TLispRef &TLispRef::CadaAr() { return CAr().Car().Cdr().Car(); }
inline TLispRef &TLispRef::CadaDr() { return CDr().Car().Cdr().Car(); }
inline TLispRef &TLispRef::CaddAr() { return CAr().Cdr().Cdr().Car(); }
inline TLispRef &TLispRef::CaddDr() { return CDr().Cdr().Cdr().Car(); }
inline TLispRef &TLispRef::CdaaAr() { return CAr().Car().Car().Cdr(); }
inline TLispRef &TLispRef::CdaaDr() { return CDr().Car().Car().Cdr(); }
inline TLispRef &TLispRef::CdadAr() { return CAr().Cdr().Car().Cdr(); }
inline TLispRef &TLispRef::CdadDr() { return CDr().Cdr().Car().Cdr(); }
inline TLispRef &TLispRef::CddaAr() { return CAr().Car().Cdr().Cdr(); }
inline TLispRef &TLispRef::CddaDr() { return CDr().Car().Cdr().Cdr(); }
inline TLispRef &TLispRef::CdddAr() { return CAr().Cdr().Cdr().Cdr(); }
inline TLispRef &TLispRef::CdddDr() { return CDr().Cdr().Cdr().Cdr(); }
inline TLispRef &TLispRef::Caaaar() { return Car().Car().Car().Car(); }
inline TLispRef &TLispRef::Caaadr() { return Cdr().Car().Car().Car(); }
inline TLispRef &TLispRef::Caadar() { return Car().Cdr().Car().Car(); }
inline TLispRef &TLispRef::Caaddr() { return Cdr().Cdr().Car().Car(); }
inline TLispRef &TLispRef::Cadaar() { return Car().Car().Cdr().Car(); }
inline TLispRef &TLispRef::Cadadr() { return Cdr().Car().Cdr().Car(); }
inline TLispRef &TLispRef::Caddar() { return Car().Cdr().Cdr().Car(); }
inline TLispRef &TLispRef::Cadddr() { return Cdr().Cdr().Cdr().Car(); }
inline TLispRef &TLispRef::Cdaaar() { return Car().Car().Car().Cdr(); }
inline TLispRef &TLispRef::Cdaadr() { return Cdr().Car().Car().Cdr(); }
inline TLispRef &TLispRef::Cdadar() { return Car().Cdr().Car().Cdr(); }
inline TLispRef &TLispRef::Cdaddr() { return Cdr().Cdr().Car().Cdr(); }
inline TLispRef &TLispRef::Cddaar() { return Car().Car().Cdr().Cdr(); }
inline TLispRef &TLispRef::Cddadr() { return Cdr().Car().Cdr().Cdr(); }
inline TLispRef &TLispRef::Cdddar() { return Car().Cdr().Cdr().Cdr(); }
inline TLispRef &TLispRef::Cddddr() { return Cdr().Cdr().Cdr().Cdr(); }
  
inline string &TLispRef::SymbolName() { return AssertSymbol(*this)->Name(); }

#if defined(TAGGEDPOINTERS)
inline TLispRef TLispRef::Apply(TLispRef ArgList) {
  if (IsValue()) THROW(XApply(Type()));
  return Object()->Apply(ArgList);
}
#endif

// We are using message pipes which transport `char *' strings.

typedef TPipeBase<char> TTextPipe;

// Lisp globals

// The `define' hacks below are due to lack of global constructors.

extern TLispRef *_GlobalFrame;
#define GlobalFrame (*_GlobalFrame)

extern TLispRef *_LispNull;
#define LispNull (*_LispNull)
extern TLispRef *_LispTrue;
#define LispTrue (*_LispTrue)
extern TLispRef *_LispFalse;
#define LispFalse (*_LispFalse)
extern string *SyntaxAListName;

#if defined(TAGGEDPOINTERS)
#define NEW 
#else
#define NEW new
#endif

TLispRef MakeGlobalFrame();
TLispRef lisp_read(TTextPipe &);
TLispRef lisp_load(TTextPipe &, TLispRef Frame);
TLispRef lisp_eval(TLispRef, TLispRef Frame = GlobalFrame);
void lisp_write(TTextPipe &, TLispRef);
void read_eval_write_loop(TTextPipe &input, TTextPipe &commandpipe, TTextPipe &output, 
			  TLispRef Frame = GlobalFrame);

#endif
