### 
### Maple library 
### accompanying the paper "How to integrate a polynomial over a simplex"
### by V. Baldoni, N. Berline, J. A. De Loera, M. Koeppe, and M. Vergne
###
### $Header: /home/mkoeppe/cvsroot/mkoeppe/w/papers/rat/pisa/programs/velleda-experiments-2008-05-18/integration.mpl,v 1.1 2008/09/09 19:32:10 mkoeppe Exp $
###

printlevel:=0: 
with(LinearAlgebra):  
with(combinat): 
with(linalg): 
with(numapprox,laurent):

###
### DECOMPOSITION OF POLYNOMIALS INTO POWERS OF LINEAR FORMS
###

## from_monome_to_linear
##
## INPUT:  d -- IGNORED
##         m -- A monomial represented by its exponent vector
## 
## OUTPUT: A list of elements [COEFFICIENT, LINEAR_FORM]
## 	   such that that the monomial x^m is represented as
##	   the sum of COEFFICIENT * LINEAR_FORM ^ TOTAL_DEGREE.

## FIXME: Rename, remove unused argument

from_monome_to_linear:=proc(d,m) 
  local out,M,i,bi,linear_form;
  out:=[];
  M:=add(m[i],i=1..nops(m));
  linear_form:=list_for_simplex_integral(m);#print("linear",linear_form);
     for i from 1 to nops(linear_form) 
     do 
     bi:=coeff_linear_form_expansion(linear_form[i],m,M);
     out:=[op(out),[bi,linear_form[i]]];
     od;
  out;
end:

## list_for_simplex_integral
##
## INPUT:  m -- A monomial represented by its exponent vector
## OUTPUT: A list of all exponent vectors <= m, 
##         leaving out non-primitive vectors

list_for_simplex_integral:=proc(m) 
  local j,F,L,i,f,newL;newL:=[];
  i:=1; 
  L:=[seq([j],j=0..m[1])];#print("L",L,i);
  
       for i from 2 to nops(m) do #print(i,nops(L));
       L:=[seq(seq([op(L[s]),j],j=0..m[i]),s=1..nops(L))];
       od;
  
       for j from 1 to nops(L) 
       do F:=L[j];
  
         if
         igcd(igcd(seq(F[i],i=1..nops(F)-1)),F[nops(F)])<>1
         then newL:=newL; 
         else newL:=[op(newL),F];
         fi;
       newL:
       od;
  newL;
end:

## Compute the coefficient of a power of a linear form appearing in
## the expansion of a monomial.
##
## Since the function above only lists
## the primitive representatives of proportional linear forms (such as
## [1, 2] for all of [1, 2], [2, 4], [3, 6], ...), we collect the
## corresponding coefficients in this function.
##
## INPUT:  L -- Linear form
##   	   m -- Exponent vector of a monomial
##	   M -- Its total degree
## OUTPUT: The coefficient.
##
coeff_linear_form_expansion:=proc(L,m,M) 
      local p,j,c,s,out;p:=[];
      if igcd(igcd(seq(L[i],i=1..nops(L)-1)),L[nops(L)])<>1 then print(trouble); 
      else
          for j from 1 to nops(L) do #print("j",j);
             if L[j]<>0 then
             p:=[op(p),iquo(m[j],L[j])];
             else p:=p;
             fi;
           od;
  
      c:=min(seq(p[i],i=1..nops(p)));
      s:=add(L[i],i=1..nops(L));#print("s,m,c",s,m,c);
      out:=1/M!*add((-1)^(M-k*s)*product(binomial(m[i],k*L[i]),i=1..nops(L))*k^M,k=1..c);
      fi;
  out;
end:

## waring_bound: Return the bound for the number of powers of linear forms
## 		 needed for a general homogeneous polynomial,
## 		 by the Alexander--Hirschowitz theorem
##
## INPUT:  n -- dimension
## 	   M -- degree
## OUTPUT: Bound

waring_bound := proc(n, M)
      if (M = 2) then n
      elif (M=3) and (n=5) then 8
      elif (M=4) and (n=3) then 6
      elif (M=4) and (n=4) then 10
      elif (M=4) and (n=5) then 15
      else ceil(binomial(n+M-1, M) / n);
      end if;
    end:

###
### INTEGRAL OF A MONOMIAL OVER A SIMPLEX
### BY DECOMPOSING IT IN POWERS OF LINEAR FORMS.

## without_from_monome_to_linear_integral
##
## INPUT: Here our data is a simplex S, d an integer and m a monomial.
## OUTPUT: The integral. 

without_from_monome_to_linear_integral:=proc(S,d,m) 
  local out,M,i,bi,linear_form;
  out:=0;
  M:=add(m[i],i=1..nops(m));
  linear_form:=list_for_simplex_integral(m);
   #print(numberoflinearforms, nops(linear_form)):
     for i from 1 to nops(linear_form) 
     do 
  bi:=coeff_linear_form_expansion(linear_form[i],m,M);
  #print("linear e coeff",from_monome_to_linear(d,m));
     out:=out+without_basic_simplex_integral(S,d,M,linear_form[i])*bi
     od;
  out;
end:

###
### INTEGRAL OF A MONOMIAL OVER A SIMPLEX,  
### BY MEANS OF BRION THEOREM AND ITERATED MAPLE LAURENT EXPANSIONS. 
###

#  Tool box
# Math: Primitive vector V and scalar a such that V=a*A
  primitive_vector:=proc(A) local d,n,g;
  d:=nops(A);
  n:=ilcm(seq(denom(A[i]),i=1..d));
  g:=igcd(seq(n*A[i],i=1..d));
  [[seq(n*A[i]/g,i=1..d)],n/g];
  end:
  Todd:=proc(s,x,N);
  -add(bernoulli(n,s)*x^n,n=0..N);
  end:
#  Todd(s,x,3);
#OUTPUT:                 /    1\     /1    2    \  2   /1      3   3  2\  3
#OUTPUT:            -1 - |s - -| x - |- + s  - s| x  - |- s + s  - - s | x 
#OUTPUT:                 \    2/     \6         /      \2          2   /   
  trunc_exp:=proc(x,N);
  add(x^n/n!,n=0..N);
  end:
#  Laurent expansions
# INPUT:     P is a polynomial  in variable x.
# OUTPUT: [k,newP] . k is a non negative integer. newP a polynomial  in variable x.
# MATH:     k is the order of vanishing of P at x=0. newP =P/x^k
# -------------------------------------------------------------------------------------------------------
  factor_pole_out:=proc(P,x) local k,newP;
  k:=0;
  newP:=P;
  while evalb(subs(x=0,newP)=0)=true do
       k:=k+1;
       newP:=simplify(newP/x);
  od;
  [k,newP];
  end:

# INPUT:     P,Q polynomials  in one variable x with coefficients expressions. 
# OUTPUT:   expression
# MATH:      coefficient of degree j in the Laurent expansion of R=P/Q at x=0
#                    Attention: If 0 is a pole of order k, then laurent (R,x,j) returns  the expansion at order j-k .
#                    Thus we factor out the pole prior to computing the Laurent expansion.
# --------------------------------------------------------------------------------------------------------------------------------------------
Laurent_coeff:=proc(P,Q,x,j) 
  local newQ,R,k,L;
  #st:=time();
  newQ:=factor_pole_out(Q,x)[2];
  R:=P/newQ;
  k:=factor_pole_out(Q,x)[1];
  #st2:=time();
  L:=convert(laurent(R,x,j+k+1),polynom);
  coeff(L,x,j+k);
end:
# ------------------------------------------------------------------------------------------------------------------------
# INPUT. S :: vector in R^d
#               U:: list of n vectors in R^d 
# MATH: 
#             The procedure returns the coefficient of v[1]^m[1] ... v[d]^m[d]
#             in   the meromorphic function e^<v,S>/prod(<v,U_j>, expanded using iterated Laurent series. 
# ----------------------------------------------------------------------------------------------------------------------
  Nicole_cone_expansion:=proc(S,U,d,m) local P,Q,n,k,R;
  n:=nops(U);
  P:=exp(add(v[k]*S[k],k=1..d));               #print("P",P);
  Q:=product(add(v[k]*U[i][k],k=1..d),i=1..n); #print("Q",Q);
  
  for k from d to 1 by -1 do             #print("k",k);
      R:=Laurent_coeff(P,Q,v[k],m[k]);   # print("R",R);
      P:=numer(R);                       #print("P",P);
      Q:=denom(R);                       #print("Q",Q);
  od;
  R;
  end:
#  B:=Nicole_cone_expansion(S,[[1,1,1],[0,0,1]],3,[1,1,1]);
#OUTPUT:        1                 2                                 2
#OUTPUT:        - [39, 65, 86, 15]  [97, 42, 52, 60] [46, 27, 74, 2] 
#OUTPUT:        4                                                    
#OUTPUT: 
#OUTPUT:             1                  3 /                2
#OUTPUT:           + -- [39, 65, 86, 15]  \-[46, 27, 74, 2] 
#OUTPUT:             12                                     
#OUTPUT: 
#OUTPUT:                                               \   1                  5
#OUTPUT:           - 2 [97, 42, 52, 60] [46, 27, 74, 2]/ - -- [39, 65, 86, 15] 
#OUTPUT:                                                   40                  
#OUTPUT: 
#OUTPUT:             1                  4                     
#OUTPUT:           + -- [39, 65, 86, 15]  [378, 192, 400, 128]
#OUTPUT:             48                                       
  
Nicole_cone_expansion_with_trunc_exp:=proc(S,U,d,m) local P,Q,n,k,N,R;
  n:=nops(U);             
  R:=1/product(add(v[k]*U[i][k],k=1..d),i=1..n); 
  for k from d to 1 by -1 do  
       #st1:=time();         
       # print("k",k);  
       N:=add(m[i],i=k..d)+n;  
       P:=numer(R)*trunc_exp(v[k]*S[k],N);
       #print(time()-st1,k,num); #print("P",P);
       #st2:=time();
       Q:=denom(R);                      
       #;print(time()-st2,k,den,Q);  #print("Q",Q);
       #st3:=time(); 
       R:=Laurent_coeff(P,Q,v[k],m[k]); 
       #print(time()-st3,k,lau);  
  od;
  R;
  end:
  #T:=time():
  #A:=Nicole_cone_expansion_with_trunc_exp([1$6],[[0,-1,-2,-3,-4,-5],[1,-1,-2,-3,-4,-5],[1,1,-2,-3,-4,-5],[1,1,1,-3,-4,-5],[1,1,1,1,-4,-5],[1,1,1,1,1,-5]],6,[1$6]);T:=time()-T;
  ##  Sex:=[1,1,1,1,1,1]:Uex:=[[0,-1,-2,-3,-4,-5],[1,-1,-2,-3,-4,-5],[1,1,-2,-3,-4,-6],[1,1,1,-3,-4,-5],[1,1,1,1,-4,-5],[1,1,1,1,1,-5]]:
  #st:=time():Nicole_cone_expansion_with_trunc_exp(Sex,Uex,6,[3,3,2,2,2,3]);T:=-st+time();
  
#  
# Full dimensional simplex, canonical Lebesgue measure
# INPUT.  S a list of d+1 vectors in R^d;  m a list of d non negative integers.
# MATH. The output is  the integral of   $x_1^m_1...x_d^m_d$ over the simplex S,  
# with respect to canonical Lebesgue measure.
# --------------------------------------------------------------------------------------------------------------
Nicole_simplex_integral:=proc(S,d,m)
  local A,i,U,D;
  A:=0; 
  for i from 1 to d+1 do
      U:=[seq(S[j]-S[i],j=1..i-1),seq(S[j]-S[i],j=i+1..d+1)];     
      A:= A+Nicole_cone_expansion_with_trunc_exp(S[i],U,d,m);  
  od;
  D:=abs(Determinant(Matrix([seq(S[j]-S[1],j=2..d+1)])));    #print(D);      
  D*(-1)^d* product(m[k]!,k=1..d)*A; 
  end:
##  Sr := [[7/4, 7/2, 4], [1, 4/3, 0], [1/4, 3, 2], [4/7, 1/8, 5/2]];
#OUTPUT:                 [[7  7   ]  [   4   ]  [1      ]  [4  1  5]]
#OUTPUT:                 [[-, -, 4], [1, -, 0], [-, 3, 2], [-, -, -]]
#OUTPUT:                 [[4  2   ]  [   3   ]  [4      ]  [7  8  2]]
  #T:=time(): Nicole_simplex_integral(Sr,3,[2$3]);T:=time()-T;
# Lower dimensional simplex, Lebesgue measure defined by intersection lattice
# -----------------------------------------------------------------------------------------------------------------
# relative_volume:=proc(C,d)
#OUTPUT: 
# Input: d ::  positive integer
#            C ::   list of n  vectors in Q^d   
# Math: Let L(C) be the linear span of (C_i)'s, 
#           the procedure returns the relative volume of the parallelepiped with edges C_i, 
#           with respect to the Lebesgue measure  normalized by the intersection lattice L(C) \cap Z^d. 
# 
# ----------------------------------------------------------------------------------------------------------------------------
#OUTPUT: 
  relative_volume:=proc(C,d)   local primC,Cmatrix, H,A,scale;
  primC:=[seq(primitive_vector(C[i])[1],i=1..nops(C))]; 
  Cmatrix:=Transpose(Matrix(primC));                           
  H:=HermiteForm(Cmatrix, method='integer');                 
  A:=SubMatrix(H,[1..nops(C)],[1..nops(C)]);
  scale:=[seq(primitive_vector(C[i])[2],i=1..nops(C))]; 
  abs(Determinant(A))/product(scale[i],i=1..nops(C));
  end:
#OUTPUT: 
  #relative_volume([[2,1]],2);
  relative_simplex_integral:=proc(S,d,m) local n,A,i,U,C;
  n:=nops(S)-1;
  A:=0; 
  for i from 1 to n+1 do
      U:=[seq(S[j]-S[i],j=1..i-1),seq(S[j]-S[i],j=i+1..n+1)];     
      A:= A+Nicole_cone_expansion(S[i],U,d,m);  
  od;
  C:=[seq(S[j]-S[1],j=2..n+1)];          
  relative_volume(C,d)*(-1)^n*product(m[k]!,k=1..d)*A; 
  end:
# Face of a full dimensional simplex, Lebesgue measure defined by intersection lattice
# 
# ----------------------------------------------------------------------------------------------------------------------------------
# INPUT: S a list of d+1 vectors in Q^d.
#              K a sublist of [1..d+1];
# MATH: K defines a face S(K) of S . 
#             The output is  the integral of 
#             $x_1^m_1...x_d^m_d$ over S(K), for the Lebesgue measure defined by the intersection lattice. 
#            
# -----------------------------------------------------------------------------------------------------------------------------------
  simplex_face_integral:=proc(S,d,K,m) local face;
  face:=[seq(S[K[j]],j=1..nops(K))]; #print(face);
  relative_simplex_integral(face,d,m);
  end:
  #S:=random_simplex(4,100);
  #simplex_face_integral(S,4,[1,2,3],[0$4]);
  
###
### INTEGRAL OF A POWER OF A LINEAR FORM OVER A SIMPLEX.
###

# Our Notations;

# 
# The  integer d is the dimension;
# A vector in Q^d is a list of d rational numbers.
# The simplex is the convex hull of its vertices s_i.
# If the simplex is of full dimension we have (d+1) vertices.
# A monomial x_1^m_1 dots x_d^m_d is represented as a list of d non negative integers m_i 
# 
# 
# Linear forms are called alpha, beta, rho: given by an element of Q^d.
# A "cone": [S,C] is indexed by its vertex S and  a list of vectors  C:=[V_1,V_2,...,V_k] .
#  Here the V_i are the generators of the cone.


# 
# Simplex and multiplicities.
# 
# Input: a simplex and  a linear form. 
# output: set of the elements  {[a_S, m_S]} 
# where m_S is the number of vertices S where <\alpha,S> takes the value m_S.
# 
#    
  multiplicity_alpha_simplex:=proc(S,d,alpha) local i,n,VS,m,Mult,j,c;
  n:=nops(S);
  VS:={seq(add(alpha[s]*S[i][s],s=1..d),i=1..nops(S))};
  Mult:={};
  for i from 1 to nops(VS) do 
  m:=0; 
  for j from 1 to nops(S) do 
  c[j]:=add(alpha[s]*S[j][s],s=1..d);
  if c[j]=VS[i] then m:=m+1;
  else m:=m;
  fi;
  od;
  Mult:={op(Mult),[VS[i],m]};
  od;
  Mult:
  end:
  
##  multiplicity_alpha_simplex([[0,0],[0,0],[0,1]],2,[1,1]);
#OUTPUT:                               {[1, 1], [0, 2]}
# 
# 
# 
# coeff((epsilon+vertex[1])^(M+d)*1/(epsilon+a2)^m2*1/(epsilon+a3)^m3; epsilon,m1-1)
residueloc:=proc(M,d,vertex,Multloc) 
  local f,m,L;
  f:=(epsilon+vertex[1])^(M+d);
  for m from 1 to nops(Multloc) do
  f:=f*1/(Multloc[m][1])^(Multloc[m][2])*1/(1+epsilon/Multloc[m][1])^(Multloc[m][2]);
  od; 
  f;
  L:=convert(laurent(f,epsilon,vertex[2]),polynom);
  coeff(L,epsilon,vertex[2]-1);
  end:
#  printlevel:=0;residueloc(0,3,[0,3],[-1,1]);

## without_basic_simplex_integral
##
## 
##
## INPUT: S a simplex, alpha a linear form, d an integer, M an integer 
## OUTPUT: a number int_S alpha^M
## MATH:;
## See the manual for the formula.  
## \prod_{j\neq 1}(\ell_1-\ell_j)^{-m_j}* coeff( (\ell_1+\epsilon)^{M+d}* \frac{1}{(1+\epsilon/(\ell_1-\ell_2))}^{m_2}
## \frac{1}{(1+\epsilon/(\ell_1-\ell_3))}^{m_3}\cdots ..., m_1-1)$$

without_basic_simplex_integral:=proc(S,d,M,alpha) 
  local int,Mult,output,v,i,Multloc,B,R,m,vertex;
  v:=abs(Determinant(Matrix([seq(S[j]-S[1],j=2..d+1)])));
  if v=0 then int:=0;
  else 
   Mult:=multiplicity_alpha_simplex(S,d,alpha); 
  int:=0;
  for  i from 1 to nops(Mult) do 
  vertex:=Mult[i];
   Multloc:=
  [seq([Mult[i][1]-Mult[j][1],Mult[j][2]],j=1..(i-1)),
  seq([Mult[i][1]-Mult[j][1],Mult[j][2]],j=(i+1)..nops(Mult))];
  R:=residueloc(M,d,vertex,Multloc);
  int:=int+R;   
  od;
  fi;
  M!/(M+d)!*int*v:
  end:
  
###
### INTEGRATION OF HOMOGENEOUS POLYNOMIALS
### USING POLARIZATION.
###

## integrate_with_polarization
##
## This uses the formula in the Corollary of section "A formula of
## Lasserre--Avrachenkov".
##
## INPUT:  S a list of d+1 vectors in R^d;  m a list of d non negative integers.
## OUTPUT: the integral of   $x_1^m_1...x_d^m_d$ over the simplex S,  
## 	   with respect to canonical Lebesgue measure.


iterate_increasing_vectors_aux := proc(from_value, to_value, prefix, dimension, procedure)
  local i;
  if dimension = 0 then
    procedure(prefix)
  else
    for i from from_value to to_value do 
      iterate_increasing_vectors_aux(i, to_value, [op(prefix), i], 
      				 dimension - 1, procedure);
    od;
  end if;
end:

iterate_increasing_vectors := proc(from_value, to_value, dimension, procedure)
  iterate_increasing_vectors_aux(from_value, to_value, [], dimension, procedure);
end:

eval_monomial := proc(m, arg)
  mul(arg[i]^m[i], i=1..nops(m));
end:

integrate_with_polarization := proc(S,d,m) 
  local D, M, total;			
  M:=add(m[i],i=1..nops(m));   			    
  D:=abs(Determinant(Matrix([seq(S[j]-S[1],j=2..d+1)])));
  #print(D);
  total := 0;
  
  iterate_increasing_vectors(1, d+1, M, 
    proc(i_vector)

      local Epsilons, epsilon;
      #print(i_vector);
      Epsilons := cartprod([seq([-1, 1], i=1..M)]);
      while not Epsilons[finished] do 
    
        epsilon := Epsilons[nextvalue]();
        #print(epsilon);

        total := total 
          + mul(epsilon[l], l=1..M)
            * eval_monomial(m, add(epsilon[k] * S[i_vector[k]], k=1..M));

      end do;
    end);

  D/(d!) / (2^M * M! * binomial(M+d, M)) * total;
end:
