unit GvFonts;

{ Graphics Vision High-level font support,
  Copr. 1995,1999 Matthias Koeppe

  $Id$
}

{$ifndef FPK}
{$A+,B-,F+,G+,I-,O+,P-,Q-,R-,S-,T-,V+,X+}
{$endif}

interface

{$ifdef FPK}
uses
  Dos, Objects, Drivers, Views, Validate,
  GvViews, GvDialog, GvStdDlg, GvCombo, GvValid;
{$i fpkdef.pp}
{$else !FPK}
{$ifdef Windows}
uses
  WinTypes, WinProcs, Dos, Objects, Drivers, Views, Validate,
  GvViews, WinGr, GvDialog, GvStdDlg, GvCombo, GvValid;
{$else !Windows}
uses
  Dos, Objects, Drivers, Views, Validate,
  MyFonts, GvViews, GvDialog, GvStdDlg, GvCombo, GvValid;
{$endif}
{$endif}

{ Font dialog commands
}
const
  cmChooseFont = 70;
  cmSetScales = 71;
  cmSetOptions = 72;
  cmFontOptions = 73;

{ Font face handling 
}
type
  PFontFace = ^TFontFace;
  TFontFace = object(TObject)
    Name: PString;
    Scales: PStringCollection;
    FileName: PString;
    FontType: Word;
    LastBestScale: LongInt;
    constructor Init(aName: string; AScales: PStringCollection;
      aFileName: string; AType: Word);
    destructor Done; virtual;
    function BestScale(Scaling: LongInt): LongInt;
  end;

  PFontFaceList = ^TFontFaceList;
  TFontFaceList = object(TSortedCollection)
    StdScales: PStringCollection;
    constructor Init;
    destructor Done; virtual;
    function Compare(Key1, Key2: pointer): Sw_Integer; virtual;
{$ifndef Windows}
    procedure InsertBGIFonts; virtual;
{$ifndef FPK}
    procedure InsertBiosFonts; virtual;
    procedure InsertCpiFonts; virtual;
{$endif FPK}
{$endif Windows}
    procedure InsertStdScales; virtual;
{$ifndef FPK}
    procedure InsertWinFonts; virtual;
{$endif}
    function LookUpFace(AFaceName: string; AScaling: LongInt): PFontFace;
    function GetFace(var S: TStream): PFontFace;
    procedure PutFace(var S: TStream; Face: PFontFace; AScaling: LongInt);
  end;

const
  FontFaceList: PFontFaceList = nil;

function GetFontFaceList: PFontFaceList;

{ Logical Font Handling 
}

type
  PTTOptions = ^TTTOptions;
  TTTOptions = record
    ItalicizationAngle: LongInt;
    RotationAngle: LongInt
  end;

  PFont = ^TFont;
  TFont = object(TObject)
    Face: PFontFace;
    Scaling: LongInt;
    Attr: Word;
    TTOptions: TTTOptions;
    Handle: Integer;
    constructor Init(FaceName: string; AScaling: LongInt;
      AnAttr: Word; Options: PTTOptions);
    constructor InitByFace(AFace: PFontFace; AScaling: LongInt;
      AnAttr: Word; Options: PTTOptions);
    constructor InitByFont(AFont: PFont; AScaling: LongInt;
      AnAttr: Word; Options: PTTOptions);
    constructor Load(var S: TStream);
    destructor Done; virtual;
    function GetFontHandle: Integer;
    procedure Store(var S: TStream);
  end;

function PixelToScaling(Pixel: Integer): LongInt;

{ Graphics Vision objects 
}
type
  PCharList = ^TCharList;
  TCharList = object(TListViewer)
    Font: PFont;
    CharsPerLine: Integer;
    CharDistance: Integer;
    FirstChar, LastChar: Char;
    constructor Init(var Bounds: TRect; AScrollBar: PScrollBar; AFont: PFont);
    destructor Done; virtual;
    procedure CalcParams;
    procedure ChangeBounds(var Bounds: TRect); virtual;
    procedure DrawItemText(Item: Integer; R: TRect); virtual;
    {procedure GetItemRect(Item: Integer; var R: TRect); virtual;}
    function GetItemHeight: Integer; virtual;
    procedure SetFont(aFont: PFont);
  end;

  PCharMapWindow = ^TCharMapWindow;
  TCharMapWindow = object(TWindow)
    CharList: PCharList;
    constructor Init(var Bounds: TRect);
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

  PPaddedStringCollection = ^TPaddedStringCollection;
  TPaddedStringCollection = object(TStringCollection)
    function Compare(Key1, Key2: Pointer): Sw_Integer; virtual;
  end;

  PFontFaceListBox = ^TFontFaceListBox;
  TFontFaceListBox = object(TSortedListBox)
    procedure FocusItem(Item: Integer); virtual;
    function GetKey(var S: String): Pointer; virtual;
    function GetText(Item: Integer; MaxLen: Integer): String; virtual;
  end;

  PAttrCluster = ^TAttrCluster;
  TAttrCluster = object(TCheckBoxes)
    procedure Press(Item: Integer); virtual;
  end;

{$ifndef NOVALIDATE}
  PScrollValidator = ^TScrollValidator;
  TScrollValidator = object(TGRangeValidator)
    Scrollbar: PScrollbar;
    constructor Init(AScrollbar: PScrollbar; AMin, AMax: Longint);
    function IsValid(const S: string): Boolean; virtual;
  end;
{$endif}

  PScrollInputLine = ^TScrollInputLine;
  TScrollInputLine = object(TInputLine)
    Scrollbar: PScrollbar;
    MappedMin, MappedMax: Integer;
    constructor Init(var Bounds: TRect; AMaxLen: Integer;
      AScrollbar: PScrollbar; AMappedMin, AMappedMax: Integer);
    procedure HandleEvent(var Event: TEvent); virtual;
  end;

  PFontDialogData = ^TFontDialogData;
  TFontDialogData = record
    Face: TListBoxRec;
    Scaling: LongInt;
    Scale: TListBoxRec;
    Attr: Word
  end;

  PFontDialog = ^TFontDialog;
  TFontDialog = object(TDialog)
    FontList: PFontFaceListBox;
    ScaleList: PStringLookUpListBox;
    ScaleInput: PInputLine;
    ScaleLabel: PLabel;
    AttrCluster: PGView;
    OptionsButton: PGView;
    TTOptions: TTTOptions;
    constructor Init(ATitle: string; LastFont: PFont);
    function GetFont: PFont;
    procedure HandleEvent(var Event: TEvent); virtual;
    procedure HideScales;
  end;
    
implementation

{$ifdef FPK}
uses Bgi, Misc, ExtGraph, GvTexts, GvApp;
{$else}
{$ifdef Windows}
uses {$ifndef VER80} Win31, {$endif}
  ExtGraph, GvApp, Misc, WinRes, Strings, GvTexts;
{$else}
uses ExtGraph, GvApp, Gr, Misc, WinRes, GvTexts, TTF, TTMyf;
{$endif}
{$endif}

{ Font face handling 
}

{ TFontFace object
}

constructor TFontFace.Init(aName: string; AScales: PStringCollection;
      aFileName: string; AType: Word);
begin
  inherited Init;
  Name := NewStr(AName);
  Scales := AScales;
  FileName := NewStr(AFileName);
  FontType := AType;
end;

destructor TFontFace.Done;
begin
  if Assigned(Name) then DisposeStr(Name);
  if Assigned(Scales) then Dispose(Scales, Done);
  if Assigned(FileName) then DisposeStr(FileName);
  inherited Done
end;

function TFontFace.BestScale(Scaling: LongInt): LongInt;
var
  ScaleStr: string;
  Left, Right: LongInt;
  Index: Sw_Integer;
  E: Integer;
begin
  If Scales = nil
  then LastBestScale := Scaling
  else begin
    Str(Scaling, ScaleStr);
    Scales^.Search(@ScaleStr, Index);
    If Index >= Scales^.Count then Index := Scales^.Count - 1;
    Val(PString(Scales^.At(Index))^, Right, E);
    If Index > 0
    then Val(PString(Scales^.At(Index - 1))^, Left, E)
    else Left := Right;
    If (Scaling - Left) < (Right - Scaling)
    then LastBestScale := Left
    else LastBestScale := Right
  end;
  BestScale := LastBestScale
end;

{ TFontFaceList object
}

constructor TFontFaceList.Init;
begin
  inherited Init(16, 16);
  StdScales := New(PPaddedStringCollection, Init(16, 16));
  InsertStdScales;
{$ifndef Windows}
{$ifndef FPK}
  InsertBiosFonts;
  InsertCpiFonts;
{$endif FPK}
  InsertBGIFonts;
{$endif Windows}
{$ifndef FPK}
  InsertWinFonts;
{$endif FPK}
end;

destructor TFontFaceList.Done;
begin
  Dispose(StdScales, Done);
  inherited Done
end;

function sComp(const s1, s2: string): Integer;
begin
  If s1 < s2 then sComp := -1 else
  if s1 > s2 then sComp := +1 else
  sComp := 0
end;

function TFontFaceList.Compare(Key1, Key2: pointer): Sw_Integer;
var
  s1, s2: string;

	procedure UpStr(var s: string);
	var
	  i: integer;
	begin
	  for i := 1 to Length(s) do
	    s[i] := Upcase(s[i])
	end;

begin
  s1 := PFontFace(Key1)^.Name^; UpStr(s1);
  s2 := PFontFace(Key2)^.Name^; UpStr(s2);
  Compare := sComp(s1, s2)
end;

{$ifndef Windows}
{$ifndef FPK}
procedure TFontFaceList.InsertBiosFonts;

	function OneScale(value: Integer): PStringCollection;
	var
	  coll: PStringCollection;
	  s: string;
	begin
	  New(coll, Init(1, 1));
	  Str(value, s);
	  coll^.Insert(NewStr(s));
	  OneScale := coll
	end;

	function BIOSName(Size: Integer): string;
	var
	  s: string[3];
	begin
	  Str(Size:3, s);
	  BIOSName := GetStr(252) {'BIOS'} + s + GetStr(253) {' pixels'};
	end;

begin
  Insert(New(PFontFace, Init(BIOSName(8), OneScale(8), FixedAddr(BIOS8), ftCPI)));
  Insert(New(PFontFace, Init(BIOSName(14), OneScale(14), FixedAddr(BIOS14), ftCPI)));
  Insert(New(PFontFace, Init(BIOSName(16), OneScale(16), FixedAddr(BIOS16), ftCPI)));
end;
{$endif FPK}

{$ifdef FPK}
procedure TFontFaceList.InsertBGIFonts;

function OneScale(value: Integer): PStringCollection;
var
  coll: PStringCollection;
  s: string;
begin  
  New(coll, Init(1, 1));
  Str(value, s);
  coll^.Insert(NewStr(s));
  OneScale := coll
end;

begin
  Insert(New(PFontFace, Init('Default font', OneScale(16), '', 0)));
end; { TFontFaceList.InsertBGIFonts }

{$else !FPK}

procedure TFontFaceList.InsertBGIFonts;
var
  SRec: SearchRec;
  BgiPath: DirStr;
  Dir: DirStr;
  Name: NameStr;
  Ext: ExtStr;
begin
  BgiPath := GetBgiDir;
  FindFirst(BgiPath + '*.CHR', AnyFile, SRec);
  while DosError = 0 do
  begin
    FSplit(SRec.Name, Dir, Name, Ext);
    Insert(New(PFontFace, Init(Name + GetStr(254) {' (Scalable)'},
      nil, BgiPath + SRec.Name, ftBGI)));
    FindNext(SRec)
  end
end;
{$endif FPK}

{$ifndef FPK}
procedure TFontFaceList.InsertCpiFonts;
begin
end; { TFontFaceList.InsertCpiFonts }
{$endif FPK}

{$endif Windows}

procedure TFontFaceList.InsertStdScales;
begin
  with StdScales^ do
  begin
    Insert(NewStr('8'));
    Insert(NewStr('10'));
    Insert(NewStr('12'));
    Insert(NewStr('14'));
    Insert(NewStr('16'));
    Insert(NewStr('20'));
    Insert(NewStr('24'));
    Insert(NewStr('36'));
    Insert(NewStr('48'));
    Insert(NewStr('72'));
    Insert(NewStr('96'));
    Insert(NewStr('128'))
  end
end;

{$ifdef Windows}

function InsertScaleProc(var LogFont: TLogFont; var TextMetric: TTextMetric;
  FontType: Integer; Data: PStringCollection): Integer; export;
var
  s: string;
begin
  Str(PixelToScaling(LogFont.lfHeight), s);
  Data^.Insert(NewStr(s));
  InsertScaleProc := 1
end;

function InsertFontProc(var LogFont: TLogFont; var TextMetric: TTextMetric;
  FontType: Integer; Data: PFontFaceList): Integer; export;
var
  Face: PFontFace;
  Scales: PPaddedStringCollection;
  TypeStr: string;
  FType: Word;
begin
  If FontType and (DEVICE_FONTTYPE or RASTER_FONTTYPE) <> 0
  then begin
    New(Scales, Init(16, 16));
    EnumFonts(DC, LogFont.lfFaceName, @InsertScaleProc, pointer(Scales))
  end
  else
    Scales := nil;
  If FontType and TRUETYPE_FONTTYPE <> 0
  then begin
    TypeStr := GetStr(257);
    FType := ftWinHuge
  end
  else begin
    TypeStr := '';
    FType := ftWin
  end;
  Face := New(PFontFace, Init(StrPas(LogFont.lfFaceName) + TypeStr,
    Scales, StrPas(LogFont.lfFaceName), FType));
  Data^.Insert(Face);
  InsertFontProc := 1;
end;

procedure TFontFaceList.InsertWinFonts;
begin
  EnumFonts(BeginDraw, nil, @InsertFontProc, @Self);
  EndDraw;
end;

{$else !Windows}

{$ifndef FPK}
procedure TFontFaceList.InsertWinFonts;
var
  WinSysDir: DirStr;
  SRec: SearchRec;
  FName: PathStr;
  Dir: DirStr;
  Name: NameStr;
  Ext: ExtStr;
  Face: string;
  FType: Word;
  Scales: PStringCollection;
  TTFont: TTTFont;
  LanguageID: Word;
  E: Integer;

	function HaveThisFont(Item: PFontFace): Boolean; {$ifndef FPK}far;{$endif}
	begin
	  HaveThisFont := (Item^.FileName <> nil) and (Item^.FileName^ = FName);
	end;

	function GetScales: PStringCollection;
	var
	  F: TBufStream;
	  Scales: PPaddedStringCollection;
	  FontDir: TFontDir;
	  Info: TFontInfo;
	  Scale: string[3];
	  i: Integer;
	  ResTbl: LongInt;

		procedure HandleTT;
		var
		  Size: Word;
		begin
		  F.Seek(ResTbl);
		  Size := SkipToResourceS(F, nil, rt_TTF_Path);
		  If Size > 0
		  then begin
		    If Size >= SizeOf(FName) then Size := SizeOf(FName) - 1;
		    F.Read(FName[1], Size);
		    FName[0] := Chr(Size);
		    FName[0] := Chr(Pos(#0, FName));
		    FName := FExpand(FSearch(FName, WinSysDir + ';' + GetEnv('PATH')));
		  end
		end;

	begin
	  GetScales := nil;
	  F.Init(FName, stOpenRead, 1024);
	  If SkipStub(F) and SkipToResTbl(F)
	  then begin
	    ResTbl := F.GetPos;
	    If SkipToResource(F, nil, rt_FontDir)
	    then begin
	      New(Scales, Init(16, 16));
	      F.Read(FontDir, SizeOf(TFontDir));
	      For i := 1 to FontDir.fdCount do
	      begin
		ReadFontInfoExt(F, Info, true, nil, @Face);
		if Info.dfType and 1 = 0
		then begin
		  Str(Info.dfPoints, Scale);
		  Scales^.Insert(NewStr(Scale));
		  Face := Face + GetStr(255) {' (Windows)'}
		end else
		if Info.dfType and 2 = 0
		then Face := Face + GetStr(256) {' (Windows Scalable)'}
		else begin
		  ReadStrZ(F, @Face);
		  Face := Face + GetStr(257) {' (TrueType)'};
		  FType := ftWinHuge;
		  HandleTT
		end;
	      end;
	      If Scales^.Count = 0
	      then Dispose(Scales, Done)
	      else GetScales := Scales
	    end
	  end;
	  F.Done;
	end;

begin
  WinSysDir := GetWinFontDir;
  Val(GetStr(258), LanguageID, E);
  If E <> 0 then LanguageID := $409; {US English}
  FindFirst(WinSysDir + '*.FO?', AnyFile, SRec);
  while DosError = 0 do
  begin
    FSplit(SRec.Name, Dir, Name, Ext);
    Face := Name;
    FName := WinSysDir + SRec.Name;
    FType := ftWin;
    Scales := GetScales;
    Insert(New(PFontFace, Init(Face, Scales, FName, FType)));
    FindNext(SRec)
  end;
  { Windoze95 requires us to search for TTF fonts in a second pass,
    since no FOT files are available any more.
  }
  FindFirst(WinSysDir + '*.TTF', AnyFile, SRec);
  while DosError = 0 do
  begin
    FName := WinSysDir + SRec.Name;
    If FirstThat(@HaveThisFont) = nil
    then begin
      TTFont.GetName(FName, LanguageID, Face);
      Face := Face + GetStr(257);
      Insert(New(PFontFace, Init(Face, nil, FName, ftWinHuge)));
    end;
    FindNext(SRec)
  end;
end;
{$endif !FPK}
{$endif !Windows}

function TFontFaceList.LookUpFace(AFaceName: string; AScaling: LongInt): PFontFace;
var
  VeryBestScale: LongInt;
  Key: TFontFace;
  Index: Sw_Integer;
  Face: PFontface;

	procedure CalcBestScale(Face: PFontFace); {$ifndef FPK}far;{$endif}
	begin
	  If abs(Face^.BestScale(AScaling) - AScaling) < abs (VeryBestScale - AScaling)
	  then VeryBestScale := Face^.LastBestScale
	end;

	function Fits(Face: PFontFace): Boolean;
	begin
	  Fits := Face^.LastBestScale = VeryBestScale
	end;

begin
  VeryBestScale := -1;
  ForEach(@CalcBestScale);
  Key.Init(AFaceName, nil, '', 0);
  Search(@Key, Index);
  Key.Done;
  Face := nil;
  repeat
    If Index >= Count then Face := At(Count - 1) else
    if fits(At(Index)) then Face := At(Index)
    else Inc(Index);
  until Face <> nil;
  LookUpFace := Face;
end;

function TFontFaceList.GetFace(var S: TStream): PFontFace;
var
  AFaceName: PString;
  AScaling: LongInt;
begin
  AFaceName := S.ReadStr;
  S.Read(AScaling, SizeOf(AScaling));
  GetFace := LookUpFace(AFaceName^, AScaling);
  DisposeStr(AFaceName)
end;

procedure TFontFaceList.PutFace(var S: TStream;
  Face: PFontFace; AScaling: LongInt);
begin
  S.WriteStr(Face^.Name);
  S.Write(AScaling, SizeOf(AScaling))
end;

{ Support routine
}

function GetFontFaceList: PFontFaceList;
begin
  If FontFaceList = nil
  then FontFaceList := New(PFontFaceList, Init);
  GetFontFaceList := FontFaceList
end;

{ Logical Font Handling 
}

constructor TFont.InitByFace(AFace: PFontFace; AScaling: LongInt;
  AnAttr: Word; Options: PTTOptions);
begin
  inherited Init;
  Face := AFace;
  Scaling := Face^.BestScale(AScaling);
  Attr := AnAttr;
  If Options <> nil
  then TTOptions := Options^;
  Handle := -1
end;

constructor TFont.InitByFont(AFont: PFont; AScaling: LongInt;
  AnAttr: Word; Options: PTTOptions);
begin
  TFont.InitByFace(AFont^.Face, AScaling, AnAttr, Options)
end;

constructor TFont.Init(FaceName: string; AScaling: LongInt;
  AnAttr: Word; Options: PTTOptions);
begin
  TFont.InitByFace(GetFontFaceList^.LookUpFace(FaceName, AScaling),
    AScaling, AnAttr, Options)
end;

constructor TFont.Load(var S: TStream);
begin
  Face := GetFontFaceList^.GetFace(S);
  Scaling := Face^.LastBestScale;
  S.Read(Attr, SizeOf(Attr));
  S.Read(TTOptions, SizeOf(TTOptions));
  Handle := -1;
end;

destructor TFont.Done;
begin
{$ifndef FPK}
  If Handle >= 0
  then FreeFont(Handle);
{$endif}
  inherited Done
end;

{$ifdef FPK}
function TFont.GetFontHandle: Integer;
begin
  GetFontHandle := 0
end; { TFont.GetFontHandle }
{$else !FPK}
{$ifdef Windows}
function TFont.GetFontHandle;
var
  LogFont: PLogFont;
begin
  If Handle < 0
  then begin
    New(LogFont);
    FillChar(LogFont^, SizeOf(TLogFont), 0);
    with LogFont^ do
    begin
      StrPCopy(lfFaceName, Face^.FileName^);
      If Attr and ftBold <> 0 then lfWeight := fw_Bold else
      if Attr and ftThin <> 0 then lfWeight := fw_Thin else
      lfWeight := fw_Normal;
      If Attr and ftItalic <> 0 then lfItalic := 1;
      lfHeight := Round(Scaling * GetResY / 72);
      lfCharSet := Default_CharSet;
    end;
    Handle := DefFont(PFontRec(LogFont));
  end;
  GetFontHandle := Handle
end;
{$else !Windows}
function TFont.GetFontHandle: Integer;
const
  RotationAttr: array[Boolean] of Byte = (ftNoVerticalMoves, ftVerticalMoves);
var
  Matrix: TTransMatrix;
  Font: PFontRec;

	function pt2pix(pt: Integer): Integer;
	begin
	  pt2pix := Round(pt * GetResY / 72)
	end;

	function Attrib(Font: PFontRec): PFontRec;
	begin
	  Font^.DiffAttr := Attr;
	  Attrib := Font
	end;

begin
  If Handle < 0
  then begin
    case Face^.FontType of
      ftCPI:
	Font := Attrib(LoadCPIFont(Face^.FileName^,
	  437, {pix} Scaling, 0, 0, 0));
      ftBGI:
	Font := LoadBgiFileFont(Face^.FileName^,
	  {pix} pt2pix(Scaling), GetResX, 1, GetResY, 1, 0, 0, 0, Attr);
      ftWin:
	Font := LoadWinFont(Face^.FileName^,
	  0, {pt} Scaling, {pix} 0, 0, 0, 0, Attr);
      ftWinHuge:
	Font := LoadTTFontExt(Face^.FileName^,
	  {pt} Scaling, {pt} Scaling,
	  Rotate(- TTOptions.RotationAngle,
	    Italicize(TTOptions.ItalicizationAngle,
	      Identity(@Matrix))),
	  {RotationAttr[TTOptions.RotationAngle mod 360 <> 0]} ftNoVerticalMoves);
    end;
    Handle := DefFont(Font);
  end;
  GetFontHandle := Handle
end;
{$endif !Windows}
{$endif !FPK}
	
procedure TFont.Store(var S: TStream);
begin
  GetFontFaceList^.PutFace(S, Face, Scaling);
  S.Write(Attr, SizeOf(Attr));
  S.Write(TTOptions, SizeOf(TTOptions))
end;

{ TCharList object 
}

constructor TCharList.Init(var Bounds: TRect; AScrollBar: PScrollBar; AFont: PFont);
begin
  inherited Init(Bounds, AScrollBar);
  Flags := Flags or lfPartialLines;
  Font := nil;
  If AFont <> nil
  then SetFont(AFont)
  else SetFont(New(PFont, Init('BIOS', 16, ftNormal, nil)))
end;

destructor TCharList.Done;
begin
  If Font <> nil
  then Dispose(Font, Done);
  inherited Done
end;

procedure TCharList.CalcParams;
var
  Metrics: TTextMetric;
begin
  If Font <> nil then
  SetTextParams(Font^.GetFontHandle, 0, 0, false);
  GetTextMetrics(Metrics);
  FirstChar := Chr(Metrics.tmFirstChar);
  LastChar := Chr(Metrics.tmLastChar);
  CharDistance := Metrics.tmMaxCharWidth + 10;
  CharsPerLine := (Size.x - 6 + 10) div CharDistance;
  If CharsPerLine < 1 then CharsPerLine := 1;
  SetRange((Ord(LastChar) - Ord(FirstChar) + CharsPerLine) div CharsPerLine);
  DrawView
end;

procedure TCharList.ChangeBounds(var Bounds: TRect);
begin
  inherited ChangeBounds(Bounds);
  CalcParams
end;

procedure TCharList.DrawItemText(Item: Integer; R: TRect);
var
  i: Integer;
  Sub: TRect;
  c: Integer;
Begin
  GetItemSubRect(Sub);
  Sub.Intersect(R);
  SetSubRect(Sub);
  If Font <> nil then
  SetTextParams(Font^.GetFontHandle, 0, sColor, false);
  SetTextJustify(LeftText, CenterText);
  For i := 0 to CharsPerLine - 1 do
  begin
    c := Item * CharsPerLine + i + Ord(FirstChar);
    If c <= Ord(LastChar)
    then OutTextXY(R.A.x + i * CharDistance, (R.A.y + R.B.y) div 2, Chr(c))
  end
End;

{procedure TCharList.GetItemRect;
var
  Height: Integer;
Begin
  If Font <> nil then
  SetTextParams(Font^.GetFontHandle, 0, 0, false);
  Height := TextHeight('');
  R.Assign(3, 3 + (Item - TopItem) * (Height + 3),
    Size.X - 3, 3 + (Item - TopItem + 1) * (Height + 3))
End;
}

function TCharList.GetItemHeight: Integer;
var
  Height: Integer;
Begin
  If Font <> nil then
  SetTextParams(Font^.GetFontHandle, 0, 0, false);
  Height := TextHeight('');
  GetItemheight := Height + 3;
end;

procedure TCharList.SetFont(aFont: PFont);
begin
  If Font <> nil
  then Dispose(Font, Done);
  Font := AFont;
  CalcParams
end;

{ TCharMapWindow object 
}

constructor TCharMapWindow.Init(var Bounds: TRect);
var
  R, Client: TRect;
  View: PGView;
  SBar: PScrollBar;
begin
  inherited Init(Bounds, GetStr(240) {'Char Map'}, wnNoNumber);
  GetClientRect(Client);

  R := Client;
  R.A.x := R.B.x - 130;
  R.B.x := R.A.x + 17;
  SBar := New(PScrollBar, Init(R));
  Insert(SBar);

  R := Client;
  Dec(R.B.x, 129);
  CharList := New(PCharList, Init(R, SBar, nil));
  CharList^.GrowMode := gfGrowHiX + gfGrowHiY;
  Insert(CharList);

  R := Client;
  R.A.x := R.B.x - 100;
  Dec(R.B.x, 10);
  Inc(R.A.y, 10);
  R.B.y := R.A.y + 20;
  View := New(PButton, Init(R, GetStr(241) {'~F~ont...'}, cmChooseFont, bfNormal));
  View^.GrowMode := gfGrowLoX + gfGrowHiX;
  Insert(View);

  SelectNext(false)
end;

procedure TCharMapWindow.HandleEvent(var Event: TEvent);

      procedure ChooseFont;
      var
	Dialog: PFontDialog;
      begin
	Dialog := New(PFontDialog, Init(GetStr(242) {'Choose Font'}, CharList^.Font));
	Dialog^.Options := Dialog^.Options or ofCentered;
	if Desktop^.ExecView(Dialog) = cmOK
	then CharList^.SetFont(Dialog^.GetFont);
	Dispose(Dialog, Done)
      end;
	
begin
  inherited HandleEvent(Event);
  if Event.What = evCommand then
  case Event.Command of
    cmChooseFont:
      ChooseFont;
  end
end;

{ TPaddedStringCollection object 
}

function Pad(s: string; Len: Integer): string;
var
  Padded: string;
begin
  Padded[0] := Chr(Len - Length(s));
  If Length(Padded) <> 0
  then begin
    FillChar(Padded[1], Length(Padded), ' ');
    Pad := Padded + s
  end
  else Pad := s
end;

function Max(a, b: Integer): Integer;
begin
  If a > b then Max := a else Max := b
end;

function TPaddedStringCollection.Compare(Key1, Key2: Pointer): Sw_Integer;
var
  s1, s2: string;
  Len: Integer;
begin
  s1 := PString(Key1)^;
  s2 := PString(Key2)^;
  Len := Max(Length(s1), Length(s2));
  s1 := Pad(s1, Len);
  s2 := Pad(s2, Len);
  Compare := sComp(s1, s2)
end;

{ TFontFaceListBox object
}

procedure TFontFaceListBox.FocusItem(Item: Integer);
begin
  inherited FocusItem(Item);
  if Item >= 0
  then begin
    Message(GOwner, evCommand, cmSetScales,
      PFontFace(List^.At(Item))^.Scales);
    Message(GOwner, evCommand, cmSetOptions,
{$ifdef FPK}
	    nil
{$else}
	    pointer(LongInt(Byte(PFontFace(List^.At(Item))^.FontType = ftWinHuge)))
{$endif}
	    )
  end
end;

{$ifdef FPK}
var
  Key: TFontFace;
{$endif}

function TFontFaceListBox.GetKey(var S: String): Pointer;
{$ifndef FPK}
const
  Key: TFontFace = ();
{$endif}	 
begin
  Key.Init(S, nil, '', 0);
  GetKey := @Key
end;

function TFontFaceListBox.GetText(Item: Integer; MaxLen: Integer): String;
begin
  GetText := PFontFace(List^.At(Item))^.Name^
end;

{ TAttrCluster object 
}

procedure TAttrCluster.Press(Item: Integer);
begin
  inherited Press(Item);
  If Item = 0
  then begin
    SetButtonState(2, not Mark(0));
    DrawView
  end else
  if Item = 1
  then begin
    SetButtonState(1, not Mark(1));
    DrawView
  end
end;

{ TScrollValidator object 
}

{$ifndef NOVALIDATE}

constructor TScrollValidator.Init(AScrollbar: PScrollbar;
  AMin, AMax: Longint);
begin
  inherited Init(AMin, AMax);
  Scrollbar := AScrollbar;
  Options := Options or voTransfer
end;

function Scale(Value: Integer;
  MinA, MaxA, MinB, MaxB: Integer): Integer;
begin
  Scale := LongDiv(
	     LongMul(
	       Value - MinA,
	       MaxB - MinB),
	     MaxA - MinA) +
	   MinB
end;

function TScrollValidator.IsValid(const S: string): Boolean;
var
  Value: LongInt;
begin
  IsValid := inherited IsValid(S);
  If Transfer(PString(@S)^, @Value, vtGetData) <> 0
  then begin
    Value := Scale(Value, Min, Max, Scrollbar^.Max, Scrollbar^.Min);
    Scrollbar^.SetValue(Value)
  end
end;
{$endif !NOVALIDATE}

{ TScrollInputLine object 
}

constructor TScrollInputLine.Init(var Bounds: TRect; AMaxLen: Integer;
      AScrollbar: PScrollbar; AMappedMin, AMappedMax: Integer);
begin
  inherited Init(Bounds, AMaxLen);
  Scrollbar := AScrollbar;
  MappedMin := AMappedMin;
  MappedMax := AMappedMax;
{$ifndef NOVALIDATE}
  SetValidator(New(PScrollValidator, Init(AScrollbar, AMappedMin, AMappedMax)));
{$endif NOVALIDATE}
  Options := Options or ofValidate
end;

procedure TScrollInputLine.HandleEvent(var Event: TEvent);
var
  value: LongInt;
begin
  inherited HandleEvent(Event);
  If (Event.What = evBroadcast) and (Event.Command = cmScrollbarChanged) and
     (Event.InfoPtr = Scrollbar)
  then begin
    value := Scale(Scrollbar^.Value, Scrollbar^.Max, Scrollbar^.Min,
      MappedMin, MappedMax);
    SetData(value)
  end
end;

{ TFontDialog object 
}

constructor TFontDialog.Init(ATitle: string; LastFont: PFont);
var
  R, Client: TRect;
  SBar: PScrollBar;
  RValid: PValidator;
  Data: TFontDialogData;
begin
  R.Assign(0, 0, 420, 303);
  inherited Init(R, ATitle);
  Options := Options or ofCentered;
  GetClientRect(Client);

  R.Assign(Client.B.x - 140, Client.A.y + 30, Client.B.x - 123, Client.B.y - 10);
  SBar := New(PScrollBar, Init(R));
  Insert(SBar);
  R.Assign(Client.A.x + 10, Client.A.y + 30, Client.B.x - 139, Client.B.y - 10);
  FontList := New(PFontFaceListBox, Init(R, SBar));
  Insert(FontList);
  R.Assign(Client.A.x + 10, Client.A.y + 10, Client.B.x - 139, Client.A.y + 30);
  Insert(New(PLabel, Init(R, GetStr(243) {'~F~ont Name'}, FontList)));

  R.Assign(Client.B.x - 110, Client.A.y + 30, Client.B.x - 26, Client.A.y + 50);
  ScaleInput := New(PInputLine, Init(R, 8));
  RValid := New(PGRangeValidator, Init(1, 999));
  RValid^.Options := RValid^.Options or voTransfer;
  ScaleInput^.SetValidator(RValid);
  Insert(ScaleInput);

  R.Assign(Client.B.x - 27, Client.A.y + 49, Client.B.x - 10, Client.B.y - 128);
  SBar := New(PScrollBar, Init(R));
  Insert(SBar);

  R.Assign(Client.B.x - 100, Client.A.y + 49, Client.B.x - 26, Client.B.y - 128);
  ScaleList := New(PStringLookUpListBox, Init(R, SBar, ScaleInput));
  Insert(ScaleList);
  R.Assign(Client.B.x - 110, Client.A.y + 10, Client.B.x - 26, Client.A.y + 30);
  ScaleLabel := New(PLabel, Init(R, GetStr(244) {'~S~ize in pt'}, nil));
  Insert(ScaleLabel);

  R.Assign(Client.B.x - 110, Client.B.y - 120, Client.B.x - 10, Client.B.y - 70);
  AttrCluster := Insert(New(PAttrCluster, Init(R,
    NewSItem(GetStr(245) {'Em~b~olden'},
    NewSItem(GetStr(246) {'~T~hin out'},
    NewSItem(GetStr(247) {'~I~talicize'},
    nil))))));

  R.Assign(Client.B.x - 110, Client.B.y - 100, Client.B.x - 10, Client.B.y - 80);
  OptionsButton := Insert(New(PButton, Init(R, GetStr(248) {'~O~ptions...'}, cmFontOptions, bfNormal)));
  OptionsButton^.Hide;

  R.Assign(Client.B.x - 110, Client.B.y - 60, Client.B.x - 10, Client.B.y - 40);
  Insert(New(PButton, Init(R, GetStr(12) {'OK'}, cmOK, bfDefault)));
  R.Move(0, 30);
  Insert(New(PButton, Init(R, GetStr(13) {'Cancel'}, cmCancel, bfNormal)));
  SelectNext(false);

  with Data do
  begin
    Face.List := GetFontFaceList;
    If LastFont <> nil
    then Face.Index := GetFontFaceList^.IndexOf(LastFont^.Face)
    else Face.Index := 0;
    FontList^.SetData(Face);
    If LastFont <> nil
    then begin
      ScaleInput^.SetData(LastFont^.Scaling);
      AttrCluster^.SetData(LastFont^.Attr);
      TTOptions := LastFont^.TTOptions
    end
  end;
end;

function TFontDialog.GetFont: PFont;
var
  Data: TFontDialogData;
begin
  GetData(Data);
  with Data do
    GetFont := New(PFont, InitByFace(Face.List^.At(Face.Index), Scaling,
      Attr, @TTOptions))
end;

procedure TFontDialog.HandleEvent(var Event: TEvent);

	procedure HideShow(View: PGView; DoShow: Boolean);
	var
	  SaveOptions: Word;
	begin
	  with View^ do
	  begin
	    SaveOptions := Options;
	    Options := Options and not ofSelectable;
	    SetState(sfVisible, DoShow);
	    Options := SaveOptions
	  end
	end;

	procedure FontOptions;
	var
	  R: TRect;
	  D: PDialog;

		procedure InsCtrl(var R: TRect; Banner: string; Left, Right: Integer);
		var
		  L, I, I1, I2, S: TRect;
		  SB: PScrollBar;
		  sLeft, sRight: string;
		begin
		  Str(Left, sLeft);
		  Str(Right, sRight);
		  L := R; L.B.Y := L.A.Y + 20;
		  S := R; S.A.Y := S.A.Y + 25; S.B.Y := S.A.Y + 18;
		  I := R; I.A.Y := I.A.Y + 48; I.B.Y := I.A.Y + 16;
		  SetTextParams(GetStandardFont, 0, 0, false);
		  I1 := I; I1.B.X := I1.A.X + TextWidth(sRight) + 5;
		  I2 := I; I2.A.X := I2.B.X - TextWidth(sLeft) - 5;
		  I.Grow(-90, 0);
		  with D^ do
		  begin
		    SB := PScrollbar(Insert(New(PScrollbar, Init(S))));
		    SB^.SetParams(0, Left, Right, 10, 1);
		    Insert(New(PLabel, Init(L, Banner,
		      Insert(New(PScrollInputLine, Init(I, 4, SB, Left, Right)))
		    )));
		    Insert(New(PStaticText, Init(I1, sRight)));
		    Insert(New(PStaticText, Init(I2, sLeft)))
		  end
		end;

	begin
	  R.Assign(0, 0, 260, 230);
	  D := New(gvdialog.PDialog, Init(R, GetStr(249))); {'TrueType Options'}
	  with D^ do begin
	    Options := Options or ofCentered;

	    R.Assign(20, 35, 240, 115);
	    InsCtrl(R, GetStr(250), -60, +60);

	    R.Move(0, 80);
	    InsCtrl(R, GetStr(251), -180, +180);

	    R.Assign(20, 190, 120, 210);
	    Insert(New(PButton, Init(R, GetStr(12) {'OK'}, cmOK, bfDefault)));
	    R.Move(120, 0);
	    Insert(New(PButton, Init(R, GetStr(13) {'Cancel'}, cmCancel, bfNormal)));
	    SelectNext(false);
	  end;
	  D := PDialog(Application^.ValidView(D));
	  If D = nil then Exit;
	  D^.SetData(TTOptions);
	  D^.Valid(cmOK);
	  If GOwner^.ExecView(D) = cmOK
	  then D^.GetData(TTOptions);
	  Dispose(D, Done)
	end;

begin
  inherited HandleEvent(Event);
  if Event.What = evCommand then
    case Event.Command of
      cmSetScales:
	begin
	  ScaleList^.List := nil;
	  If Event.InfoPtr = nil
	  then begin
	    ScaleList^.NewList(GetFontFaceList^.StdScales);
	    ScaleList^.Validator^.Strict := false;
	    ScaleLabel^.Link := ScaleInput
	  end
	  else begin
	    ScaleList^.NewList(Event.InfoPtr);
	    ScaleList^.Validator^.Strict := true;
	    ScaleLabel^.Link := ScaleList
	  end;
	  ScaleList^.FocusItem(0);
	  ScaleInput^.SetState(sfDisabled, Event.InfoPtr <> nil)
	end;
      cmSetOptions:
	begin
	  HideShow(AttrCluster, Event.InfoByte = 0);
	  HideShow(OptionsButton, Event.InfoByte <> 0);
	end;
      cmFontOptions:
	FontOptions;
    end
end;

procedure TFontDialog.HideScales;
begin
  ScaleList^.Hide;
  ScaleList^.Scrollbar^.Hide;
  ScaleInput^.Hide;
  ScaleLabel^.Hide;
  FontList^.Select
end;

function PixelToScaling(Pixel: Integer): LongInt;
begin
  PixelToScaling := Round(LongInt(Pixel) * 72.0 / GetResY)
end;

{ No main
}

end.
